## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.align = 'center'
)

## ----setup,message=FALSE,warning=FALSE----------------------------------------
# Primary Package #
library(invitroTKstats)
# Data Formatting Packages #
library(dplyr)
library(magrittr)
library(stringr)
library(readxl)
# Table Formatting Package #
library(flextable)

## ----raw_data_dir-------------------------------------------------------------
# the path to the applicable sub-directory after the `invitroTKstats` package repository is cloned and the R project is opened
raw_data_dir <- system.file("extdata/Kreutz-Clint", package = "invitroTKstats")

## ----raw_data_files-----------------------------------------------------------
# identify the hepatic clearance raw datasets
hep_clint_xlsx <- list.files(raw_data_dir,
                             pattern = paste(
                               paste(c("^Hep","^G\\d"),"[.]xlsx$",sep = ".+"),
                               collapse = "|"))
# show the data file names
hep_clint_xlsx
# add the raw data file directory path
hep_clint_xlsx <- paste(raw_data_dir,hep_clint_xlsx, sep = "/")

## ----rdfile1_sheets-----------------------------------------------------------
# for the first dataset, see the sheets contained in the raw data file
readxl::excel_sheets(hep_clint_xlsx[1])

## ----data_catalog-------------------------------------------------------------
DC_kreutz.pfas <- create_catalog(
  # filename (no file path)
  file = c(rep("Hep_745_949_959_082421_final.xlsx",3)), 
  # sheet name (or sheet number)
  sheet = c(rep("Data063021",3)), 
  # number of rows to skip in L0 Excel file - start for compound/analyte samples
  skip.rows = c(44,74,92), 
  # number of rows to read in from L0 Excel file for compound/analyte samples
  num.rows = c(30,18,18), 
  # date the data was generated
  #   (MMDDYY: 2-digit month, 2-digit day, 2-digit year)
  date = "063021", 
  # chemical id
  compound = c("745","949","959"),
  # internal standard compound (corresponding to chemical id)
  istd = c("MFBET","MFOET","MFHET"), 
  # column name for sample names
  sample = "Name", 
  # column name for sample types
  type = "Type",
  # column name(s) for analyte MS peak areas
  peak = c("Area...13","Area...27","Area...20"), 
  # column name(s) for internal standard MS peak areas
  istd.peak = c("Resp....16","Resp....30","Resp....23"), 
  # column name(s) for experimental concentration
  conc = c("Final Conc....11","Final Conc....25","Final Conc....18"),
  # column name(s) with analysis parameters
  analysis.param = c("RT...12", "RT...26", "RT...19"),
  # column name - row locations
  col.names.loc = 2
  # note = "RT...12"
)

## ----echo=FALSE---------------------------------------------------------------
# show the data catalog
DC_kreutz.pfas %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        
      )
    )
  ) %>% 
  set_caption(caption = "Table 1: Data Catalog for the Kruetz et al. (2023) PFAS Cl~int~ Experiment.", 
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----assay_cover_sheet_metadata-----------------------------------------------
# obtain the chemical identification mapping information from the MS-data
#   cover sheet (that is raw data summary information)
assay_cover_sheet <-
  readxl::read_xlsx(
    paste(raw_data_dir,"Hep_745_949_959_082421_final.xlsx",sep = "/"),
    sheet = "Cover Sheet",skip = 35,n_max = 4) %>% 
  as.data.frame()

## ----echo=FALSE---------------------------------------------------------------
# show the assay cover sheet
assay_cover_sheet %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        
      )
    )
  ) %>% 
  set_caption(caption = "Table 2: Chemical Mapping Information from the Raw Excel File.", 
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----chem_id_table------------------------------------------------------------
# create a chemical table necessary for the L0 compilation function, using the
#   assay cover sheet chemical identification mapping information
chem.ids <- create_chem_table(
  # input table (data.frame class) with information
  input.table = assay_cover_sheet,
  # column name with DSSTox chemical ID's
  dtxsid.col = "Analyte",
  # column name with formal compound names
  compound.col = "Name",
  # column name with lab chemical ID's
  lab.compound.col = "Sample ID"
)

## ----echo=FALSE---------------------------------------------------------------
# show the chemical ID mapping data
chem.ids %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        
      )
    )
  ) %>% 
  set_caption(caption = "Table 3: Chemical ID Mapping Information for the `merge_level0` Function.", 
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

## ----level0_comp,message=FALSE------------------------------------------------
# compile the l0 files
kreutz.pfas_L0 <- merge_level0(
  level0.catalog = DC_kreutz.pfas, # data catalog
  INPUT.DIR = raw_data_dir, # the path to your raw data files
  num.rows.col = "Number.Data.Rows", 
  istd.col = "ISTD.Name",
  type.colname.col = "Type.ColName",
  chem.ids = chem.ids, # chemical ID mapping data
  chem.lab.id.col = "Lab.Compound.Name",
  chem.name.col = "Compound.Name",
  catalog.out = FALSE, # do not export the data catalog during this function
  output.res = FALSE # do not export the compiled L0 during this function
)
# show the dimension of the the Kruetz PFAS data
dim(kreutz.pfas_L0)

## ----echo=FALSE---------------------------------------------------------------
# show the resulting merge level 0 output data
head(kreutz.pfas_L0) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
        
      )
    )
  ) %>% 
  set_caption(caption = "Table 4: Level-0 data resulting from the `merge_level0` Function.", 
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "all") %>% 
  theme_vanilla()

