# Simulation, comparison, and evaluation of alpha-stable mixture models
# =============================================================================
# MIXTURE SIMULATION
# =============================================================================


#' Simulate mixture data from alpha-stable components
#'
#' Generates synthetic data from a mixture of alpha-stable distributions using specified weights and parameters.
#'
#' @param n Number of samples to generate.
#' @param weights Vector of mixture weights.
#' @param params List of parameter sets for each component (each with alpha, beta, gamma, delta).
#' @return Numeric vector of simulated samples.
#' @importFrom stabledist rstable
#' @export
simulate_mixture <- function(n, weights, params) {
  samples <- numeric(n)
  for (i in 1:n) {
    k <- sample(length(weights), 1, prob = weights)
    p <- params[[k]]
    samples[i] <- rstable(1, p$alpha, p$beta, gamma = p$gamma, delta = p$delta)
  }
  return(samples)
}

# =============================================================================
# COMPARISON WITH GIBBS SAMPLING
# =============================================================================


#' Compare estimation methods with and without Gibbs sampling
#'
#' Visualizes and compares mixture fits from EM, ECF (kernel and empirical), and Gibbs sampling.
#'
#' @param data Numeric vector of observations.
#' @param em_params List of EM-estimated parameters.
#' @param ecf_kernel_params List of kernel ECF-estimated parameters.
#' @param ecf_empirical_params List of empirical ECF-estimated parameters.
#' @param fig_dir Optional path to save plots. Defaults to tempdir().
#' @return NULL (plots are saved to fig_dir)
#' @importFrom graphics hist lines legend par
#' @export
compare_methods_with_gibbs <- function(data, em_params, ecf_kernel_params,
                                       ecf_empirical_params, fig_dir = tempdir()) {
  if (!dir.exists(fig_dir)) dir.create(fig_dir, recursive = TRUE)

  # Run Gibbs sampling (assumes mock_gibbs_sampling is defined)
  gibbs_result <- mock_gibbs_sampling(data, n_samples = 500)
  all_samples <- gibbs_result$samples

  # Compute mean Gibbs parameters safely
  gibbs_mean <- if (length(all_samples) > 0) {
    param_matrix <- do.call(rbind, lapply(all_samples, function(x) {
      if (!is.null(x$params)) unlist(x$params) else rep(NA, 9)
    }))
    colMeans(param_matrix, na.rm = TRUE)
  } else rep(NA, 9)

  gibbs_params <- list(
    list(gibbs_mean[2], gibbs_mean[3], gibbs_mean[4], gibbs_mean[5]),
    list(gibbs_mean[6], gibbs_mean[7], gibbs_mean[8], gibbs_mean[9]),
    gibbs_mean[1]
  )

  x_vals <- seq(min(data), max(data), length.out = 1000)

  methods <- list(
    "EM Estimated"   = em_params,
    "ECF Kernel"     = ecf_kernel_params,
    "ECF Empirical"  = ecf_empirical_params,
    "Gibbs Sampling" = gibbs_params
  )

  for (name in names(methods)) {
    method_params <- methods[[name]]
    param1 <- unlist(method_params[[1]])
    param2 <- unlist(method_params[[2]])
    weight <- method_params[[3]]

    # Safe defaults
    if (length(param1) < 4) param1 <- c(param1, rep(NA, 4 - length(param1)))
    if (length(param2) < 4) param2 <- c(param2, rep(NA, 4 - length(param2)))
    if (is.na(weight)) weight <- 0.5

    pdf1 <- r_stable_pdf(x_vals, param1[1], param1[2], param1[3], param1[4])
    pdf2 <- r_stable_pdf(x_vals, param2[1], param2[2], param2[3], param2[4])
    mixture_pdf <- weight * pdf1 + (1 - weight) * pdf2

    plot_file <- file.path(fig_dir, paste0("mixture_alpha_stable_", gsub(" ", "_", tolower(name)), ".png"))
    tryCatch({
      png(plot_file, width = 10, height = 6, units = "in", res = 150)
      oldpar <- par(no.readonly = TRUE)
      on.exit(par(oldpar), add = TRUE)

      hist(data, breaks = 40, freq = FALSE, col = rgb(0.5, 0.5, 0.5, 0.5),
           main = paste(name, "vs Data"), xlab = "x", ylab = "Density")
      lines(x_vals, mixture_pdf, col = "blue", lwd = 2)
      legend("topright", legend = c("Data", paste(name, "Fit")),
             col = c("gray", "blue"), lty = c(NA, 1), lwd = c(NA, 2), pch = c(15, NA), pt.cex = 2)
      dev.off()
    }, error = function(e) {
      warning("Plotting failed for ", name, ": ", e$message)
    })
  }

  invisible(NULL)
}

# =============================================================================
# FIT QUALITY EVALUATION
# =============================================================================


#' Evaluate fit quality using RMSE and log-likelihood
#'
#' Computes RMSE and log-likelihood scores for different mixture models compared to histogram data.
#'
#' @param data Numeric vector of observations.
#' @param methods Named list of parameter sets (each with two components and a weight).
#' @param x_vals Numeric vector. Evaluation grid.
#' @return Named list of scores per method (RMSE and LogLikelihood).
#' @importFrom graphics hist
#' @importFrom stats approx
#' @export
evaluate_fit <- function(data, methods, x_vals) {
  scores <- list()

  for (name in names(methods)) {
    method_params <- methods[[name]]
    p1 <- unlist(method_params[[1]])
    p2 <- unlist(method_params[[2]])
    w  <- method_params[[3]]

    pdf1 <- r_stable_pdf(x_vals, p1[1], p1[2], p1[3], p1[4])
    pdf2 <- r_stable_pdf(x_vals, p2[1], p2[2], p2[3], p2[4])
    pdf_mix <- w * pdf1 + (1 - w) * pdf2

    hist_result <- hist(data, breaks = 50, xlim = range(x_vals),
                        freq = FALSE, plot = FALSE)
    hist_vals <- hist_result$density
    bin_centers <- hist_result$mids
    pdf_model <- approx(x_vals, pdf_mix, xout = bin_centers, rule = 2)$y

    rmse <- sqrt(mean((hist_vals - pdf_model)^2, na.rm = TRUE))

    hist_norm <- pmax(hist_vals / sum(hist_vals, na.rm = TRUE), 1e-6)
    pdf_norm  <- pmax(pdf_model / sum(pdf_model, na.rm = TRUE), 1e-6)
    ll <- -sum(hist_norm * log(pdf_norm), na.rm = TRUE)

    scores[[name]] <- list(RMSE = rmse, LogLikelihood = ll)
  }

  return(scores)
}

# =============================================================================
# ESTIMATOR COMPARISON ON SIMULATIONS
# =============================================================================


#' Compare MLE, ECF, and McCulloch estimators on simulated data
#'
#' Runs multiple simulations and compares the mean squared error (MSE) of parameter estimates
#' from MLE, kernel ECF, and McCulloch methods.
#'
#' @param n_samples Integer. Number of samples per simulation.
#' @param n_runs Integer. Number of simulation runs.
#' @param interp_alpha Optional interpolation function for alpha (McCulloch).
#' @param interp_beta Optional interpolation function for beta (McCulloch).
#' @return Data frame of simulation results with MSE and estimated parameters.
#' @export
compare_estimators_on_simulations <- function(n_samples = 1000, n_runs = 30,
                                              interp_alpha = NULL, interp_beta = NULL) {
  results_list <- vector("list", n_runs * 3)  # preallocate list
  u <- seq(0.1, 1, length.out = 20)
  idx <- 1

  for (seed in 1:n_runs) {
    sim_data <- generate_mixture_data(1, n_samples)
    data <- sim_data$data
    true_params <- sim_data$params[[1]]

    # ---- MLE ----
    mle <- mle_estimate(data)
    mse_mle <- mean((unlist(mle) - unlist(true_params)[1:4])^2)
    results_list[[idx]] <- data.frame(seed = seed, method = "MLE", mse = mse_mle,
                                      alpha = mle$alpha, beta = mle$beta,
                                      gamma = mle$gamma, delta = mle$delta)
    idx <- idx + 1

    # ---- ECF ----
    ecf <- estimate_stable_kernel_ecf(data, u)
    mse_ecf <- mean((unlist(ecf) - unlist(true_params)[1:4])^2)
    results_list[[idx]] <- data.frame(seed = seed, method = "ECF", mse = mse_ecf,
                                      alpha = ecf$alpha, beta = ecf$beta,
                                      gamma = ecf$gamma, delta = ecf$delta)
    idx <- idx + 1

    # ---- McCulloch ----
    if (!is.null(interp_alpha) && !is.null(interp_beta)) {
      mc <- mcculloch_lookup_estimate(data, interp_alpha, interp_beta)
      mse_mc <- mean((unlist(mc) - unlist(true_params)[1:4])^2)
    } else {
      mc <- list(alpha = NA, beta = NA, gamma = NA, delta = NA)
      mse_mc <- NA
    }
    results_list[[idx]] <- data.frame(seed = seed, method = "McCulloch", mse = mse_mc,
                                      alpha = mc$alpha, beta = mc$beta,
                                      gamma = mc$gamma, delta = mc$delta)
    idx <- idx + 1
  }

  results <- do.call(rbind, results_list)
  rownames(results) <- NULL
  return(results)
}
