#===============================================================================
#                      Spearman's rho Test Functions
#===============================================================================

#' @title Univariate Spearman's Rho Test
#'
#' @description The functions performs the univariate Spearman's rho test.
#'
#' @param x Numeric vector representing a data series.
#'
#' @return A named list
#'
#' - `statistic`, the estimated Spearman's rho test statistics
#' - `p.value`, the estimated p-value for the test.
#'
#' @details
#' The Spearman's Rank test is a non-parametric trend test based on rank-order,
#' It tests a null hypothesis of no trend against an alternative.
#' Given a data series \eqn{X = (x_1, x_2, ..., x_n)} of length \eqn{n}, the test statistic is given
#' by
#' \deqn{
#' D = 1 - \frac{6 \sum_{i=1}^n [R(x_i) - i]^2}{n(n^2 - 1)}
#' }
#' where \eqn{R(x_i)} is the rank of the i-th observation in the data series.
#'
#' Under the null hypothesis D has asymptotically normal distribution, with \eqn{E(D)=0},
#' and variance
#' \deqn{
#' \text{Var}(D) = \frac{1}{n-1}
#' }
#'
#' @references
#'
#' - Sneyers, R., 1990. On the Statistical Analysis of Series of Observations. World Meteorol. Organ.
#'
#' @examples
#' # Sample data (Both marginal distributions have trend):
#' dataMarginalTrend <- generate_data("marginalTrend", n = 50)
#'
#' # Perform SR test on sample data:
#' srUnivariate(dataMarginalTrend[, 1])
#'
#' srUnivariate(dataMarginalTrend[, 2])
#'
#' @importFrom stats qnorm pnorm
#' @export

srUnivariate <- function(x) {}

srUnivariate <- function(x) {

  # Check for data type
  if(!is.numeric(x)){
    stop("'x' must be a numeric vector.")
  }

  n = length(x)

  # Ranking the observations in descending order
  sorting = sort(as.matrix(x), index.return = TRUE)
  I = sorting$ix
  rg_val = 1:(n)

  # Difference between the rank and the order
  d = t(rg_val) - I

  # Computation of SR TS
  D = 1 - (6 * sum(d^2)) / (n^3 - n)

  # Computation of variance
  var_ = 1 / (n - 1)

  # Standardized TS
  Z = D / sqrt(var_)

  pvalue = 2*pnorm(-abs(Z))

  results = list("statistic" = D, "p.value" = pvalue)

  return(results)
}
