#' Calculate reliability using resampling inter-unit reliability method
#' @description
#' This function estimates reliability using the resampling inter-unit reliability method described in He et al. 2018.
#' @details
#' In the current version, this function assumes that the measure is a simple mean of outcome values
#' within each entity. However, this method is more flexible as described in He et al. 2018.
#'
#' @param df observation-level data; if null, will use the dataframe from the model object
#' @param model model; if null, will use an unadjusted model
#' @param entity data column containing the accountable entity identifier
#' @param y data column containing the outcome variable
#' @param ctrPerf parameters to control performance measure calculation
#' @param ctrRel parameters to control reliability estimation
#' @returns A list containing:
#' * `entity`: list of entities
#' * `n`: entity sample sizes
#' * `var.b`: between-entity variance
#' * `var.w`: within-entity variance
#' * `var.total`: total variance
#' * `IUR`: entity-level reliability
#' @author Kenneth Nieser (nieser@stanford.edu)
#' @references He K, Kalbfleisch JD, Yang Y, Fei Z. Inter unit reliability for nonlinear models. Stat Med. 2019 Feb 28;38(5):844-854.
#' @examples
#' # Simulate data
#' df <- simulateData(n.entity = 50, n.obs = 100, mu = .2, r = .7)
#'
#' # Calculate reliability
#' out <- calcResamplingIUR(df = df, entity = 'entity', y = 'y', ctrRel = controlRel(n.resamples = 10))
#' out$IUR
#'
#' @importFrom stats aggregate
#' @export

calcResamplingIUR <- function(df = NULL, model = NULL,  entity = 'entity', y = 'y', ctrPerf = controlPerf(), ctrRel = controlRel()){
  if (is.null(df) & is.null(model)) stop ('Please provide either a dataframe or a model object')
  if (is.null(df)){df <- model@frame}
  message('\tCurrently, Beta-Binomial reliability estimates do not account for risk-adjustment (even if you specified a model). Updates to this function to account for risk-adjustment are in progress.')

  cl <- match.call()
  n.cores     <- ctrRel$n.cores
  n.resamples <- ctrRel$n.resamples

  df <- cleanData(df, entity, y, ctrPerf)
  agg  <- aggregate(y ~ entity, data = df, length)
  entities <- agg$entity
  n <- agg$y
  n0 <- 1 / (length(n) - 1) * (sum(n) - sum(n^2) / sum(n))
  mu = mean(df$y)
  n.entity = length(entities)

  entity.means0 = aggregate(y ~ entity, data = df, mean)$y
  var.total = 1/(n0*(n.entity - 1)) * sum(n * (entity.means0 - mu)^2)

  entity.means = matrix(data = NA, nrow = n.entity, ncol = n.resamples)

  for (j in 1:n.resamples){
    # take a bootstrap resample within each entity separately
    df.resample = data.frame()
    for (i in 1:n.entity){
      entity.df <- df[df$entity == entities[i], ]
      entity.df.resample <- entity.df[sample(nrow(entity.df), nrow(entity.df), replace = T), ]
      df.resample <- rbind(df.resample, entity.df.resample)
    }

    # calculate measure by entity
    entity.means[,j] = aggregate(y ~ entity, data = df.resample, mean)$y
  }

  bootstrap.means = apply(entity.means, 1, mean)
  bootstrap.sqrd.resid = apply(entity.means, 2, function(x) (x - bootstrap.means)^2)
  bootstrap.var = 1/(n.resamples - 1) * apply(bootstrap.sqrd.resid, 1, sum)
  var.w = sum((n - 1) * bootstrap.var) / (sum(n) - n.entity)

  IUR = (var.total - var.w) / var.total

  results = list(call = cl,
                 entity = as.vector(entities),
                 n = n,
                 var.b = var.total - var.w,
                 var.w = var.w,
                 var.total = var.total,
                 IUR = IUR)

  return(results)
}
