#' Silhouette
#'
#'
#' This function calculates the Silhouette as is defined in Rousseeuw (1987) without imposing that the use of the euclidean distance. This allows calculating the Silhouette using different distances. Note that the Silhouette must be calculated using a distance that is a a ratio scale (Rousseeuw, 1987).
#'
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param FHW_output List. List with:
#'
#' \itemize{
#'   \item centers: the information of the centers updated.
#'   \item grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'    }
#'
#' @return Returns a vector. The component i contains the Silhouette value of the point in the row i of the \code{data} matrix.
#'
#'
#'
#'
#' @examples
#'
#'set.seed(451)
#'data=rbind(matrix(runif(20,1,5), nrow = 2, ncol = 10),
#'           matrix(runif(20,20,30), nrow = 2, ncol = 10),
#'           matrix(runif(20,50,70), nrow = 2, ncol = 10))
#'k=3
#'seed=5
#'
#'FHW_output=Hartigan_and_Wong(data,
#'                             Euclideandistance,
#'                             k,
#'                             centers_function_mean,
#'                             init_centers_random,
#'                             seed=seed,
#'                             10)
#'
#'Silhouette(data, FHW_output, Euclideandistance)
#'
#'@references  Rousseeuw, P.J. (1987) Silhouettes: A graphical aid to the interpretation and validation of cluster analysis. J. Comput. Appl. Math., 20, 53--65.
#'
#' @export



Silhouette<-function(data, FHW_output, distance){

  a=rep(0,dim(data)[1])
  b=rep(0,dim(data)[1])
  silouette=rep(0,dim(data)[1])

for(i in 1:(dim(data)[1])){

  Gi=encontrar_componente(FHW_output$grouping, i)
  #Calculating a
  if(length(FHW_output$grouping[[Gi]])==1){#if there is only one component in the group
    a[i]=0
  }else{#If there are several components, we calculate the mean of the dissimilarity between i and all the points in the cluster to which i belongs
    Other_group=FHW_output$grouping[[Gi]][FHW_output$grouping[[Gi]]!=i]
    dG=rep(0,length(Other_group) )
    for(j in 1:length(Other_group)){
      dG[j]=distance(data[i,],data[Other_group[j],])
    }
    a[i]=mean(dG)
  }

  #Calculating b
  NoGi=c(1:length(FHW_output$grouping))[-Gi]
  mean_dGNoGi=rep(0,length(NoGi))
      for (m in 1:length(NoGi)){
        mean_dGNoGi[m]=mean(d_i_other_group(data, i, distance, FHW_output,NoGi[[m]]))#average dissimilarity between i and all the observation in group NoGi[[m]]
      }
  b[i]=min(mean_dGNoGi)

  silouette[i]=(b[i]-a[i])/max(a[i],b[i])

}

return(silouette)

}
