% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seq_bayes.R
\name{seq_bayes}
\alias{seq_bayes}
\title{Sequential Bayes (seqB)}
\usage{
seq_bayes(cases, mu, kappa = 20, post = FALSE)
}
\arguments{
\item{cases}{Vector of case counts. The vector must only contain non-negative
integers, and have at least two positive integers.}

\item{mu}{Mean of the serial distribution. This must be a positive number.
The value should match the case counts in time units. For example, if case
counts are weekly and the serial distribution has a mean of seven days,
then \code{mu} should be set to \code{1}. If case counts are daily and the serial
distribution has a mean of seven days, then \code{mu} should be set to \code{7}.}

\item{kappa}{Largest possible value of the uniform prior (defaults to \code{20}).
This must be a number greater than or equal to \code{1}. It describes the prior
belief on the ranges of R0, and should be set to a higher value if R0 is
believed to be larger.}

\item{post}{Whether to return the posterior distribution of R0 instead of the
estimate of R0 (defaults to \code{FALSE}). This must be a value identical to
\code{TRUE} or \code{FALSE}.}
}
\value{
If \code{post} is identical to \code{TRUE}, a list containing the following
components is returned:
\itemize{
\item \code{supp} - the support of the posterior distribution of R0
\item \code{pmf} - the probability mass function of the posterior distribution of R0
}

Otherwise, if \code{post} is identical to \code{FALSE}, only the estimate of R0 is
returned. Note that the estimate is equal to \code{sum(supp * pmf)} (i.e., the
posterior mean).
}
\description{
This function implements a sequential Bayesian estimation method of R0 due to
Bettencourt and Ribeiro (PloS One, 2008). See details for important
implementation notes.
}
\details{
The method sets a uniform prior distribution on R0 with possible values
between \code{0} and \code{kappa}, discretized to a fine grid. The distribution of R0
is then updated sequentially, with one update for each new case count
observation. The final estimate of R0 is the mean of the (last) posterior
distribution. The prior distribution is the initial belief of the
distribution of R0, which is the uninformative uniform distribution with
values between \code{0} and \code{kappa}. Users can change the value of \code{kappa} only
(i.e., the prior distribution cannot be changed from the uniform). As more
case counts are observed, the influence of the prior distribution should
lessen on the final estimate.

This method is based on an approximation of the SIR model, which is most
valid at the beginning of an epidemic. The method assumes that the mean of
the serial distribution (sometimes called the serial interval) is known. The
final estimate can be quite sensitive to this value, so sensitivity testing
is strongly recommended. Users should be careful about units of time (e.g.,
are counts observed daily or weekly?) when implementing.

Our code has been modified to provide an estimate even if case counts equal
to zero are present in some time intervals. This is done by grouping the
counts over such periods of time. Without grouping, and in the presence of
zero counts, no estimate can be provided.
}
\examples{
# Weekly data.
cases <- c(1, 4, 10, 5, 3, 4, 19, 3, 3, 14, 4)

# Obtain R0 when the serial distribution has a mean of five days.
seq_bayes(cases, mu = 5 / 7)

# Obtain R0 when the serial distribution has a mean of three days.
seq_bayes(cases, mu = 3 / 7)

# Obtain R0 when the serial distribution has a mean of seven days, and R0 is
# believed to be at most 4.
estimate <- seq_bayes(cases, mu = 1, kappa = 4)

# Same as above, but return the posterior distribution of R0 instead of the
# estimate.
posterior <- seq_bayes(cases, mu = 1, kappa = 4, post = TRUE)

# Display the support and probability mass function of the posterior.
posterior$supp
posterior$pmf

# Note that the following always holds:
estimate == sum(posterior$supp * posterior$pmf)
}
\references{
Bettencourt and Ribeiro (PloS One, 2008)
\doi{doi:10.1371/journal.pone.0002185}
}
\seealso{
\code{vignette("seq_bayes_post", package = "Rnaught")} for examples of
using the posterior distribution.
}
