#' percentile
#'
#' @description
#' Calculates specified percentiles from length-frequency data.
#'
#' @param LF Numeric vector of length-frequency data (e.g., `data$length`).
#' @param probs Numeric vector of probabilities in \eqn{[0,1]} indicating which
#'   percentiles to calculate. Default is `c(0.025, 0.975)` as per SlotLim.
#' @param na.rm Logical; if `TRUE` (default), `NA`s are removed before computing
#'   percentiles. If `FALSE`, `NA` values may propagate to the result.
#' @param sort_probs Logical; if `TRUE` (default), `probs` are sorted ascending
#'   (labels follow the returned order). If `FALSE`, percentiles are returned in
#'   the input order.
#' @param unique_probs Logical; if `TRUE` (default), duplicate `probs` are
#'   deduplicated (first occurrence kept for labeling).
#'
#' @return A named list (length = length of `probs`) where each element
#'   corresponds to the requested percentile. Names are formatted as `L_x`,
#'   where `x` is the percentile value in percent (e.g., `L_2.5`, `L_97.5`).
#'
#' @details
#' Uses `stats::quantile(..., type = 7)`, the R default. Labels drop trailing
#' zeros (e.g., `L_5` not `L_5.0`).
#'
#' @examples
#' length_data <- c(10, 9, 7, 10, 11, 13, NA, 11, 6, 20)
#' percentile(length_data)  # default 2.5th and 97.5th
#' percentile(length_data, probs = c(0.05, 0.95)) # 5th and 95th percentiles
#'
#' @export
percentile <- function(LF = NULL,
                       probs = c(0.025, 0.975),
                       na.rm = TRUE,
                       sort_probs = TRUE,
                       unique_probs = TRUE) {
  # ---- validate LF ----
  if (!is.numeric(LF)) {
    stop("`LF` must be a numeric vector.", call. = FALSE)
  }
  # allow NA but reject other non-finite values
  if (any(!is.finite(LF[!is.na(LF)]))) {
    stop("`LF` contains non-finite values (Inf/-Inf/NaN). Only NA is allowed.", call. = FALSE)
  }
  if (na.rm) {
    if (sum(!is.na(LF)) == 0L) {
      stop("All values in `LF` are NA; nothing to compute with `na.rm = TRUE`.", call. = FALSE)
    }
  } else if (length(LF) == 0L) {
    stop("`LF` must contain at least one value.", call. = FALSE)
  }

  # ---- validate probs ----
  if (!is.numeric(probs) || length(probs) < 1L || any(!is.finite(probs))) {
    stop("`probs` must be a non-empty numeric vector of finite values.", call. = FALSE)
  }
  if (any(probs < 0 | probs > 1)) {
    stop("All `probs` must lie within '[0, 1]'.", call. = FALSE)
  }

  # optional unique/sort handling
  if (unique_probs) {
    probs <- probs[!duplicated(probs)]
  }
  if (sort_probs) {
    probs <- probs[order(probs)]
  }

  # ---- compute percentiles ----
  q <- stats::quantile(LF, probs = probs, na.rm = na.rm, names = FALSE)

  # ---- format names: "L_2.5", dropping trailing zeros ----
  pct_values <- probs * 100
  lab_base <- sub("\\.?0+$", "", formatC(pct_values, format = "f", digits = 2))
  names(q) <- paste0("L_", lab_base)

  # return as named list
  as.list(q)
}
