#' Descriptive plots for the assessment of agreement
#'
#' @description
#' This function implements some basic descriptive plots that can be analysed in the context of agreement. The plots implemented are:
#' \itemize{
#'    \item{a scatter plot of all the possible paired-measurements of one rater against the other}
#'    \item{a boxplot of the measurements per rater and, if it applies, replicate}
#'    \item{a histogram of the measurements per rater and, if it applies, replicate}
#'    \item{a normal quantile-quantile plot of the measurements per rater and, if it applies, replicate}
#'    \item{a histogram of the paired-differences}
#'    \item{a normal quantile-quantile plot of the paired-differences}}
#'
#'
#' @usage descriptive.plots(data, y, id, met, rep = NA,
#'                   plots = c("scatterplot", "boxplot",
#'                             "histogram.y", "qqplot.y",
#'                             "histogram.d", "qqplot.d"),
#'                   rater.col = c("#cc6e5c", "#06402b"), d.col = "lavender",
#'                   scatterplot.xlab = NULL, scatterplot.ylab = NULL,
#'                   histogram.y.main = NULL, histogram.y.xlab = NULL,
#'                   qqplot.y.main = NULL,
#'                   line.col = "red",
#'                   histogram.d.density = TRUE, histogram.d.bw = "nrd0",
#'                   density.col = "red", ...)
#'
#' @param data name of the dataset, of class \code{data.frame}, containing at least 3 columns (quantitative measurement, subject effect, rater effect).
#' @param y quantitative measurement column name.
#' @param id subject effect column name. The corresponding column of \code{data} must be a factor.
#' @param met rater effect column name. The corresponding column of \code{data} must be a factor.
#' @param rep replicate effect column name. When there are no replicates the user should use \code{rep = NA}. When there are replicates, the corresponding column of data must be a factor. \cr
#'            The default value is \code{NA}.
#' @param plots name of the descriptive plots to display. The options are:
#'               \code{"scatterplot"} (a scatter plot of all the possible paired-measurements of one rater against the other), \code{"boxplot"} (a boxplot of the measurements per rater and, if it applies, replicate),
#'               \code{"histogram.y"} (a histogram of the measurements per rater and, if it applies, replicate), \code{"qqplot.y"} (a QQ plot of the measurements per rater and, if it applies, replicate),
#'               \code{"histogram.d"} (a histogram of the paired-differences) and \code{"qqplot.d"} (a QQ plot of the paired-differences).
#'               This argument is not case-sensitive and is passed to \code{\link[base]{match.arg}}. \cr
#'               The default value is \code{c("scatterplot", "boxplot", "histogram.y", "qqplot.y", "histogram.d", "qqplot.d")}, so all plots are executed by default.
#' @param rater.col colour palette to be used in the plots involving the measurements separated per rater. If neither \code{"boxplot"} nor \code{"histogram.y"} are selected in \code{plots}, this argument is ignored. \cr
#'                  The default value is \code{c("#cc6e5c", "#06402b")} for the first and second level, respectively, of the variable met in \code{data}.
#' @param d.col colour to be used in the histogram involving the paired-differences. If \code{"histogram.d"} is not selected in \code{plots}, this argument is ignored. \cr
#'              The default value is \code{"lavender"}.
#' @param scatterplot.xlab a label to use for the x-axis in the scatterplot. If \code{"scatterplot"} is not selected in \code{plots}, this argument is ignored. \cr
#'                         The default value, \code{NULL}, indicates that the label \emph{"}\code{y}\emph{" from "1st level of} \code{met} \emph{column"} should be used.
#' @param scatterplot.ylab a label to use for the y-axis in the scatterplot. If \code{"scatterplot"} is not selected in \code{plots}, this argument is ignored. \cr
#'                         The default value, \code{NULL}, indicates that the label \emph{"}\code{y}\emph{" from "2nd level of} \code{met} \emph{column"} should be used.
#' @param histogram.y.main overall title for the histograms of the measurements (to be passed to \code{main} argument in \code{\link[graphics]{hist}}).
#'                         Notice that all histograms (one per rater and, if it applies, replicate) will have the same title.
#'                         If \code{"histogram.y"} is not selected in \code{plots}, this argument is ignored. \cr
#'                         The default value, \code{NULL}, indicates that the label \emph{"\eqn{j}-th level of} \code{met} \emph{column"} should be used in the case of no replicates, where \eqn{j \in} \{1,2\}, and the label
#'                         \emph{"\eqn{i}-th level of} \code{rep} \emph{column" from "\eqn{j}-th level of} \code{met} \emph{column"} should be used in the case of replicates, where \eqn{i \in} \{1,..., number of replicates\} and \eqn{j \in} \{1,2\}.
#' @param histogram.y.xlab a label to use for the x-axis in the histograms of the measurements (to be passed to \code{xlab} argument in \code{\link[graphics]{hist}}).
#'                         Notice that all histograms (one per rater and, if it applies, replicate) will have the same label.
#'                         If \code{"histogram.y"} is not selected in \code{plots}, this argument is ignored. \cr
#'                         The default value, \code{NULL}, indicates that the label \emph{"}\code{y}\emph{"} should be used.
#' @param qqplot.y.main overall title for the normal quantile-quantile plots of the measurements (to be passed to \code{main} argument in \code{\link[stats]{qqplot}}).
#'                      Notice that all QQ plots (one per rater and, if it applies, replicate) will have the same title.
#'                      If \code{"qqplot.y"} is not selected in \code{plots}, this argument is ignored. \cr
#'                      The default value, \code{NULL}, indicates that the label \emph{"\eqn{j}-th level of} \code{met} \emph{column"} should be used in the case of no replicates, where \eqn{j \in} \{1,2\}, and the label
#'                      \emph{"\eqn{i}-th level of} \code{rep} \emph{column" from "\eqn{j}-th level of} \code{met} \emph{column"} should be used in the case of replicates, where \eqn{i \in} \{1,..., number of replicates\} and \eqn{j \in} \{1,2\}.
#' @param line.col colour to be used in the theoretical line added to the scatterplot or the normal quantile-quantile plots. If none of \code{"scatterplot"}, \code{"qqplot.y"}, or \code{"qqplot.d"} are selected in \code{plots}, this argument is ignored. \cr
#'                 The default value is \code{"red"}.
#' @param histogram.d.density logical indicating whether the density should be added as a solid curve to the histogram of the paired-differences.
#'                            If \code{"histogram.d"} is not selected in \code{plots}, this argument is ignored. \cr
#'                            The default value is \code{TRUE}.
#' @param histogram.d.bw the smoothing bandwidth to be used in the density curve added to the histogram involving the paired differences (to be passed to \code{bw} argument in \code{\link[stats]{density}}).
#'                       If \code{"histogram.d"} is not selected in \code{plots} or \code{histogram.d.density} is \code{FALSE}, this argument is ignored. \cr
#'                       The default value is \code{"nrd0"}.
#' @param density.col colour to be used in the density curve added to the histogram involving the paired differences. If \code{"histogram.d"} is not selected in \code{plots} or \code{histogram.d.density} is \code{FALSE}, this argument is ignored. \cr
#'                    The default value is \code{"red"}.
#' @param ... other graphical parameters (to be passed to each plot). To be used calling only one plot for optimal performance.
#'
#' @returns The desired plots are shown one after the other.
#'
#' @examples
#' # normal data
#'
#' set.seed(2025)
#'
#' n <- 100
#'
#' mu.ind <- rnorm(n, 0, 7)
#'
#' epsA1 <- rnorm(n, 0, 3)
#' epsA2 <- rnorm(n, 0, 3)
#' epsB1 <- rnorm(n, 0, 3)
#' epsB2 <- rnorm(n, 0, 3)
#'
#' y_A1 <- 50 + mu.ind + epsA1 # rater A, replicate 1
#' y_A2 <- 50 + mu.ind + epsA2 # rater A, replicate 2
#' y_B1 <- 40 + mu.ind + epsB1 # rater B, replicate 1
#' y_B2 <- 40 + mu.ind + epsB2 # rater B, replicate 2
#'
#' ex_data <- data.frame(y = c(y_A1, y_A2, y_B1, y_B2),
#'                       rater = factor(rep(c("A", "B"), each = 2*n)),
#'                       replicate = factor(rep(rep(1:2, each = n), 2)),
#'                       subj = factor(rep(1:n, 4)))
#'
#' descriptive.plots(ex_data, y, subj, rater, replicate, plots = "scatterplot")
#' descriptive.plots(ex_data, y, subj, rater, replicate, plots = "boxplot",
#'                   rater.col = c("blue", "red"), names = rep(c("1st rep.", "2nd rep."), 2),
#'                   xlab = "", main = "Boxplots per rater and replicate")
#' legend("topright", legend = c("A", "B"), fill = c("blue", "red"))
#'
#' # non-normal data
#'
#' # involving the measurements
#' def.par <- par(no.readonly = TRUE)
#' par(mfcol = c(4, 2), las = 1)
#' descriptive.plots(AMLad, mrd, id, met, rep, plots = "histogram.y",
#'                   ylim = c(0, 120), xlim = c(0, 100),
#'                   breaks = seq(from = 0, to = 100, by = 2.5))
#' descriptive.plots(AMLad, mrd, id, met, rep, plots = "qqplot.y",
#'                   ylim = c(0, 8), xlim = c(-3, 3))
#' par(def.par)
#'
#' # inolving the paired-differences
#' def.par <- par(no.readonly = TRUE)
#' par(mfrow = c(1, 2), las = 1)
#' descriptive.plots(AMLad, mrd, id, met, rep, plots = "histogram.d",
#'                   histogram.d.bw = 2,
#'                   ylim = c(0, 0.15), breaks = seq(-20, 100, 5),
#'                   xlab = "Paired-differences", main = "",
#'                   line.col = "darkred", density.col = "darkred")
#' descriptive.plots(AMLad, mrd, id, met, rep, plots = "qqplot.d",
#'                   ylim = c(-2, 8), xlim = c(-4, 4), main = "")
#' par(def.par)
#'
#' @importFrom furniture wide
#' @importFrom graphics points boxplot hist
#' @importFrom stats qqline qqnorm density
#'
#' @export

descriptive.plots <- function(data, y, id, met, rep = NA,
                              plots = c("scatterplot", "boxplot",
                                        "histogram.y", "qqplot.y",
                                        "histogram.d", "qqplot.d"),
                              rater.col = c("#cc6e5c", "#06402b"), d.col = "lavender",
                              scatterplot.xlab = NULL, scatterplot.ylab = NULL,
                              histogram.y.main = NULL, histogram.y.xlab = NULL,
                              qqplot.y.main = NULL,
                              line.col = "red",
                              histogram.d.density = TRUE, histogram.d.bw = "nrd0",
                              density.col = "red", ...) {

  # columns
  y <- paste(deparse(substitute(y)), collapse = "")
  y <- gsub("  ", "", y)
  id <- paste(deparse(substitute(id)), collapse = "")
  id <- gsub("  ", "", id)
  met <- paste(deparse(substitute(met)), collapse = "")
  met <- gsub("  ", "", met)
  rep <- paste(deparse(substitute(rep)), collapse = "")
  rep <- gsub("  ", "", rep)

  # argument type checks
  if (!is.data.frame(data)) {
    stop("'data' must be a data.frame")
  }
  if (rep != "NA") {
    if (!all(c(y, id, met, rep) %in% names(data))) {
      stop("'y', 'id', 'met' and 'rep' must be columns in 'data'")
    }
    if (any(duplicated(c(y, id, met, rep)))) {
      stop("two of the column identifiers are the same")
    }
    if (!is.factor(data[, id]) | !is.factor(data[, met]) | !is.factor(data[, rep])) {
      stop("'id', 'met' and 'rep' columns must be factors")
    }
  } else{
    rep <- NA
    if (!all(c(y, id, met) %in% names(data))) {
      stop("'y', 'id' and 'met' must be columns in 'data'")
    }
    if (any(duplicated(c(y, id, met)))) {
      stop("two of the column identifiers are the same")
    }
    if (!is.factor(data[, id]) | !is.factor(data[, met])) {
      stop("'id' and 'met' columns must be factors")
    }
  }
  choices.plots <- c("scatterplot", "boxplot", "histogram.y", "qqplot.y", "histogram.d", "qqplot.d")
  plots <- match.arg(plots, choices.plots, several.ok = TRUE)
  if (length(rater.col) != 2) {
    stop("'rater.col' must be of length 2")
    if (any(!isColor(rater.col))) {
      stop("'rater.col' must only contain valid colours")
    }
  }
  if (length(d.col) != 1) {
    stop("'d.col' must be of length 1")
    if (any(!isColor(d.col))) {
      stop("'d.col' must only contain a valid colour")
    }
  }
  if (!is.null(scatterplot.xlab)) {
    if (!is.character(scatterplot.xlab)) {
      stop("'scatterplot.xlab' must be a character")
    }
  }
  if (!is.null(scatterplot.ylab)) {
    if (!is.character(scatterplot.ylab)) {
      stop("'scatterplot.ylab' must be a character")
    }
  }
  if (!is.null(histogram.y.main)) {
    if (!is.character(histogram.y.main)) {
      stop("'histogram.y.main' must be a character")
    }
  }
  if (!is.null(histogram.y.xlab)) {
    if (!is.character(histogram.y.xlab)) {
      stop("'histogram.y.xlab' must be a character")
    }
  }
  if (!is.null(qqplot.y.main)) {
    if (!is.character(qqplot.y.main)) {
      stop("'qqplot.y.main' must be a character")
    }
  }
  if (length(line.col) != 1) {
    stop("'line.col' must be of length 1")
    if (any(!isColor(line.col))) {
      stop("'line.col' must only contain a valid colour")
    }
  }
  if (!is.logical(histogram.d.density)) {
    stop("'histogram.d.density' must be logical")
  }
  if (!is.numeric(histogram.d.bw)) {
    if (!histogram.d.bw %in% c("nrd0", "nrd", "ucv", "bcv", "SJ")){
      stop("'histogram.d.bw' should be numeric or one of: 'nrd0', 'nrd', 'ucv', 'bcv' or 'SJ'")
    }
  }
  if (length(density.col) != 1) {
    stop("'density.col' must be of length 1")
    if (any(!isColor(density.col))) {
      stop("'density.col' must only contain a valid colour")
    }
  }


  # warning checks
  if (any(table(data[, id]) == 0)) {
    warning("'id' column has (at least) one empty level. Calculation proceeds ignoring the empty level(s)")
    data[, id] <- factor(data[, id])
  }
  if (any(table(data[, met]) == 0)) {
    warning("'met' column has (at least) one empty level. Calculation proceeds ignoring the empty level(s)")
    data[, met] <- factor(data[, met])
  }
  if (nlevels(data[, met]) != 2) {
    stop("this package only supports two levels in 'met' column")
  }
  if (!is.na(rep)) {
    if (any(table(data[, rep]) == 0)) {
      warning("'rep' column has (at least) one empty level. Calculation proceeds ignoring the empty level(s)")
      data[, rep] <- factor(data[, rep])
    }
    if (nlevels(data[, rep]) == 1) {
      warning("'rep' column only has one level. Consider omitting 'rep' call. Calculation proceeds assuming rep = NA")
      rep <- NA
    }
  }
  if (nlevels(data[, id]) < 6){
    warning("number of subjects less than 6. Proceed with caution")
  }


  # stop checks
  if (!all(table(data[, met]) == table(data[, met])[1])) {
    stop("the design must be balanced, there must be the same number of measurements per method")
  }
  if (is.na(rep)){
    if (sum(table(data[, id], data[, met]) != 1) > 0) {
      stop("the design must be balanced, each subject should have one measurement per method")
    }
  } else{
    if (sum(table(data[, id], data[, met], data[, rep]) != 1) > 0) {
      stop("the design must be balanced, each subject should have one measurement per method and replicate")
    }
  }


  # extracting number of replicate
  if (!is.na(rep)) {
    m <- nlevels(data[, rep])
  } else{
    m <- 1
  }

  # extracting necessary columns of data
  if (!is.na(rep)) {
    data.long <- data.frame(y = data[, y], id = data[, id],
                            met = data[, met], rep = data[, rep])
    data.long$z <- interaction(data.long$met, data.long$rep)

    data.wide <- wide(data.long[, c("y", "id", "z")],
                      v.names = "y",
                      timevar = "z",
                      id = "id")
  } else{
    data.long <- data.frame(y = data[, y], id = data[, id], met = data[, met])
    data.wide <- wide(data.long[, c("y", "id", "met")],
                      v.names = "y",
                      timevar = "met",
                      id = "id")
  }

  # stop check NAs
  if (any(is.na(data.long))) {
    stop("NAs not supported for TDI estimation. Rerun without NAs")
  }


  # SCATTERPLOT
  if ("scatterplot" %in% plots) {
    if (is.null(scatterplot.xlab)) {
      scatterplot.xlab <- paste0(y, " from ", levels(data.long$met)[1])
    }
    if (is.null(scatterplot.ylab)) {
      scatterplot.ylab <- paste0(y, " from ", levels(data.long$met)[2])
    }
    if (is.na(rep)) {
      plot(data.long[data.long$met == levels(data.long$met)[1], ]$y,
           data.long[data.long$met == levels(data.long$met)[2], ]$y,
           xlab = scatterplot.xlab, ylab = scatterplot.ylab, ...)
      abline(a = 0, b = 1, col = line.col)
    } else{
      met1 <- data.long[data.long$met == levels(data.long$met)[1], c("id", "y")]
      met2 <- data.long[data.long$met == levels(data.long$met)[2], c("id", "y")]
      names(met1)[2] <- "met1"
      names(met2)[2] <- "met2"
      data.scatterplot <- merge(met1, met2, by = "id")

      plot(data.scatterplot$met1, data.scatterplot$met2,
           xlab = scatterplot.xlab, ylab = scatterplot.ylab, ...)
      abline(a = 0, b = 1, col = line.col)
    }
  }
  # BOXPLOT
  if ("boxplot" %in% plots) {
    if (is.na(rep)) {
      boxplot(y ~ met, data = data.long, col = rater.col, ...)
    } else{
      boxplot(y ~ rep + met, data = data.long, col = rep(rater.col, each = m), ...)
    }
  }
  # HISTOGRAM OF THE MEASUREMENTS
  if ("histogram.y" %in% plots) {
    if (is.null(histogram.y.xlab)) {
      histogram.y.xlab <- y
    }
    if (is.null(histogram.y.main)) {
      if (is.na(rep)) {
        hist(data.long[data.long$met == levels(data.long$met)[1], ]$y, col = rater.col[1],
             xlab = histogram.y.xlab, main = levels(data.long$met)[1], ...)
        hist(data.long[data.long$met == levels(data.long$met)[2], ]$y, col = rater.col[2],
             xlab = histogram.y.xlab, main = levels(data.long$met)[2], ...)
      } else{
        for (i in 1:m) {
          hist(data.long[data.long$met == levels(data.long$met)[1] & data.long$rep == levels(data.long$rep)[i], ]$y, col = rater.col[1],
               xlab = histogram.y.xlab, main = paste0(levels(data.long$rep)[i], " from ", levels(data.long$met)[1]), ...)
          hist(data.long[data.long$met == levels(data.long$met)[2] & data.long$rep == levels(data.long$rep)[i], ]$y, col = rater.col[2],
               xlab = histogram.y.xlab, main = paste0(levels(data.long$rep)[i], " from ", levels(data.long$met)[2]), ...)
        }
      }
    } else{
      if (is.na(rep)) {
        hist(data.long[data.long$met == levels(data.long$met)[1], ]$y, col = rater.col[1],
             xlab = y, main = histogram.y.main, ...)
        hist(data.long[data.long$met == levels(data.long$met)[2], ]$y, col = rater.col[2],
             xlab = y, main = histogram.y.main, ...)
      } else{
        for (i in 1:m) {
          hist(data.long[data.long$met == levels(data.long$met)[1] & data.long$rep == levels(data.long$rep)[i], ]$y, col = rater.col[1],
               xlab = y, main = histogram.y.main, ...)
          hist(data.long[data.long$met == levels(data.long$met)[2] & data.long$rep == levels(data.long$rep)[i], ]$y, col = rater.col[2],
               xlab = y, main = histogram.y.main, ...)
        }
      }
    }
  }
  # QQ PLOT OF THE MEASUREMENTS
  if ("qqplot.y" %in% plots) {
    if (is.null(qqplot.y.main)) {
      if (is.na(rep)) {
        qqnorm(scale(data.long[data.long$met == levels(data.long$met)[1], ]$y),
               main = levels(data.long$met)[1], ...)
        qqline(scale(data.long[data.long$met == levels(data.long$met)[1], ]$y),
               col = line.col)
        qqnorm(scale(data.long[data.long$met == levels(data.long$met)[2], ]$y),
               main = levels(data.long$met)[2], ...)
        qqline(scale(data.long[data.long$met == levels(data.long$met)[2], ]$y),
               col = line.col)
      } else{
        for (i in 1:m) {
          qqnorm(scale(data.long[data.long$met == levels(data.long$met)[1] & data.long$rep == levels(data.long$rep)[i], ]$y),
                 main = paste0(levels(data.long$rep)[i], " from ", levels(data.long$met)[1]), ...)
          qqline(scale(data.long[data.long$met == levels(data.long$met)[1] & data.long$rep == levels(data.long$rep)[i], ]$y),
                 col = line.col)
          qqnorm(scale(data.long[data.long$met == levels(data.long$met)[2] & data.long$rep == levels(data.long$rep)[i], ]$y),
                 main = paste0(levels(data.long$rep)[i], " from ", levels(data.long$met)[2]), ...)
          qqline(scale(data.long[data.long$met == levels(data.long$met)[2] & data.long$rep == levels(data.long$rep)[i], ]$y),
                 col = line.col)
        }
      }
    } else{
      if (is.na(rep)) {
        qqnorm(scale(data.long[data.long$met == levels(data.long$met)[1], ]$y),
               main = qqplot.y.main, ...)
        qqline(scale(data.long[data.long$met == levels(data.long$met)[1], ]$y),
               col = line.col)
        qqnorm(scale(data.long[data.long$met == levels(data.long$met)[2], ]$y),
               main = qqplot.y.main, ...)
        qqline(scale(data.long[data.long$met == levels(data.long$met)[2], ]$y),
               col = line.col)
      } else{
        for (i in 1:m) {
          qqnorm(scale(data.long[data.long$met == levels(data.long$met)[1] & data.long$rep == levels(data.long$rep)[i], ]$y),
                 main = qqplot.y.main, ...)
          qqline(scale(data.long[data.long$met == levels(data.long$met)[1] & data.long$rep == levels(data.long$rep)[i], ]$y),
                 col = line.col)
          qqnorm(scale(data.long[data.long$met == levels(data.long$met)[2] & data.long$rep == levels(data.long$rep)[i], ]$y),
                 main = qqplot.y.main, ...)
          qqline(scale(data.long[data.long$met == levels(data.long$met)[2] & data.long$rep == levels(data.long$rep)[i], ]$y),
                 col = line.col)
        }
      }
    }
  }
  # creating the vector of the paired-differences
  if ("histogram.d" %in% plots | "qqplot.d" %in% plots) {
    if (is.na(rep)) {
      d <- data.wide[, paste0("y.", levels(data.long$met)[1])] - data.wide[, paste0("y.", levels(data.long$met)[2])]
    } else{
      for (k in 1:((ncol(data.wide)-1)/2)) {
        for (l in 1:((ncol(data.wide)-1)/2)) {
          dif <- data.wide[, paste0("y.", levels(data.long$met)[1], ".", levels(data.long$rep)[k])] -
            data.wide[, paste0("y.", levels(data.long$met)[2], ".", levels(data.long$rep)[l])]
          if (k == 1 & l == 1) {
            d <- dif
          } else{
            d <- append(d, dif)
          }
        }
      }
    }
  }
  # HISTOGRAM OF THE DIFFERENCES
  if ("histogram.d" %in% plots) {
    hist(d, col = d.col, freq = FALSE, ...)
    if (histogram.d.density) {
      lines(density(d, bw = histogram.d.bw), col = density.col)
    }
  }
  # QQPLOT OF THE DIFFERENCES
  if ("qqplot.d" %in% plots) {
    qqnorm(scale(d), ...)
    qqline(scale(d), col = line.col)
  }


}
