% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/impu_boos.R
\name{impu_boost}
\alias{impu_boost}
\title{Component-Wise Gradient Boosting Across Multiply Imputed Datasets}
\usage{
impu_boost(
  X_list,
  y_list,
  X_list_val = NULL,
  y_list_val = NULL,
  ny = 0.1,
  mstop = 250,
  type = c("gaussian", "logistic"),
  MIBoost = TRUE,
  pool = TRUE,
  pool_threshold = 0,
  center = c("auto", "force", "off")
)
}
\arguments{
\item{X_list}{List of length M; each element is an \eqn{n \times p} numeric
predictor matrix from one imputed dataset.}

\item{y_list}{List of length M; each element is a length-\eqn{n} numeric
response vector from one imputed dataset.}

\item{X_list_val}{Optional validation list (same structure as \code{X_list}).}

\item{y_list_val}{Optional validation list (same structure as \code{y_list}).}

\item{ny}{Learning rate. Defaults to \code{0.1}.}

\item{mstop}{Number of boosting iterations (default \code{250}).}

\item{type}{Type of loss function. One of:
\code{"gaussian"} (mean squared error) for continuous responses,
or \code{"logistic"} (binomial deviance) for binary responses.}

\item{MIBoost}{Logical. If \code{TRUE}, applies the MIBoost algorithm,
which enforces uniform variable selection across all imputed datasets. If
\code{FALSE}, variables are selected independently within each imputed
dataset, and pooling is governed by \code{pool_threshold}.}

\item{pool}{Logical. If \code{TRUE}, models across the \eqn{M} imputed
datasets are aggregated into a single final model. If \code{FALSE},
\eqn{M} separate models are returned.}

\item{pool_threshold}{Only used when \code{MIBoost = FALSE} and \code{pool = TRUE}.
Controls the pooling rule when aggregating the \eqn{M} models obtained from
the imputed datasets into a single final model. A candidate variable is
included only if it is selected in at least \code{pool_threshold} (a value
in (0, 1)) proportion of the imputed datasets; coefficients of all other
variables are set to zero. A value of \code{0} corresponds to
estimate-averaging, while values \code{> 0} correspond to
selection-frequency thresholding.}

\item{center}{One of \code{c("auto", "off", "force")}. Controls
centering of \code{X} within each imputed dataset.
With \code{"auto"} (recommended), centering is applied only if the training
matrix is not already centered. With \code{"force"}, centering is always
applied. With \code{"off"}, centering is skipped. If \code{X_list_val} is
provided, validation sets are centered using the means from the
corresponding training set.}
}
\value{
A list with elements:
\itemize{
\item \code{INT}: intercept(s). A scalar if \code{pool = TRUE}, otherwise
a length-M vector.
\item \code{BETA}: coefficient estimates. A length-p vector if
\code{pool = TRUE}, otherwise an M \eqn{\times} p matrix.
\item \code{CV_error}: vector of validation errors (if validation data
were provided), otherwise \code{NULL}.
}
}
\description{
Applies component-wise gradient boosting to multiply imputed datasets.
Depending on the settings, either a separate model is reported for each
imputed dataset, or the M models are pooled to yield a single final model.
For pooling, one can choose the novel \emph{MIBoost} algorithm, which enforces
a uniform variable-selection scheme across all imputations, or the more
conventional ad-hoc approaches of estimate-averaging and
selection-frequency thresholding.
}
\details{
This function supports \emph{MIBoost}, which enforces uniform variable
selection across multiply imputed datasets.
For full methodology, see Kuchen (2025).
}
\examples{
\dontshow{if (requireNamespace("mice", quietly = TRUE) && requireNamespace("miceadds", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{

  set.seed(123)
  utils::data(booami_sim)

  M <- 2
  n <- nrow(booami_sim)
  x_cols <- grepl("^X\\\\d+$", names(booami_sim))

  tr_idx <- sample(seq_len(n), floor(0.8 * n))
  dat_tr <- booami_sim[tr_idx, , drop = FALSE]
  dat_va <- booami_sim[-tr_idx, , drop = FALSE]

  pm_tr <- mice::quickpred(dat_tr, method = "spearman",
                           mincor = 0.30, minpuc = 0.60)

  imp_tr <- mice::mice(dat_tr, m = M, predictorMatrix = pm_tr,
                       maxit = 1, printFlag = FALSE)
  imp_va <- mice::mice.mids(imp_tr, newdata = dat_va, maxit = 1, printFlag = FALSE)

  X_list      <- vector("list", M)
  y_list      <- vector("list", M)
  X_list_val  <- vector("list", M)
  y_list_val  <- vector("list", M)
  for (m in seq_len(M)) {
    tr_m <- mice::complete(imp_tr, m)
    va_m <- mice::complete(imp_va, m)
    X_list[[m]]     <- data.matrix(tr_m[, x_cols, drop = FALSE])
    y_list[[m]]     <- tr_m$y
    X_list_val[[m]] <- data.matrix(va_m[, x_cols, drop = FALSE])
    y_list_val[[m]] <- va_m$y
  }

  fit <- impu_boost(
    X_list, y_list,
    X_list_val = X_list_val, y_list_val = y_list_val,
    ny = 0.1, mstop = 50, type = "gaussian",
    MIBoost = TRUE, pool = TRUE, center = "auto"
  )

  which.min(fit$CV_error)
  head(fit$BETA)
  fit$INT
}

\dontrun{
# Heavier demo (more imputations and iterations; for local runs)

  set.seed(2025)
  utils::data(booami_sim)

  M <- 10
  n <- nrow(booami_sim)
  x_cols <- grepl("^X\\\\d+$", names(booami_sim))

  tr_idx <- sample(seq_len(n), floor(0.8 * n))
  dat_tr <- booami_sim[tr_idx, , drop = FALSE]
  dat_va <- booami_sim[-tr_idx, , drop = FALSE]

  pm_tr <- mice::quickpred(dat_tr, method = "spearman",
                           mincor = 0.20, minpuc = 0.40)

  imp_tr <- mice::mice(dat_tr, m = M, predictorMatrix = pm_tr,
                       maxit = 5, printFlag = TRUE)
  imp_va <- mice::mice.mids(imp_tr, newdata = dat_va, maxit = 1, printFlag = FALSE)

  X_list      <- vector("list", M)
  y_list      <- vector("list", M)
  X_list_val  <- vector("list", M)
  y_list_val  <- vector("list", M)
  for (m in seq_len(M)) {
    tr_m <- mice::complete(imp_tr, m)
    va_m <- mice::complete(imp_va, m)
    X_list[[m]]     <- data.matrix(tr_m[, x_cols, drop = FALSE])
    y_list[[m]]     <- tr_m$y
    X_list_val[[m]] <- data.matrix(va_m[, x_cols, drop = FALSE])
    y_list_val[[m]] <- va_m$y
  }

  fit_heavy <- impu_boost(
    X_list, y_list,
    X_list_val = X_list_val, y_list_val = y_list_val,
    ny = 0.1, mstop = 250, type = "gaussian",
    MIBoost = TRUE, pool = TRUE, center = "auto"
  )
  str(fit_heavy)
}
\dontshow{\}) # examplesIf}
}
\references{
Kuchen, R. (2025). \emph{MIBoost: A Gradient Boosting Algorithm for Variable
Selection After Multiple Imputation}. arXiv:2507.21807.
\doi{10.48550/arXiv.2507.21807} \url{https://arxiv.org/abs/2507.21807}.
}
\seealso{
\code{\link{simulate_booami_data}}, \code{\link{cv_boost_raw}}, \code{\link{cv_boost_imputed}}
}
