% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/readNWISdata.R
\name{readNWISdata}
\alias{readNWISdata}
\title{General Data Import from NWIS}
\usage{
readNWISdata(..., asDateTime = TRUE, convertType = TRUE, tz = "UTC")
}
\arguments{
\item{\dots}{see \url{https://waterservices.usgs.gov/docs/site-service/} for
a complete list of options.  A
list of arguments can also be supplied. One important argument to include is
"service". Possible values are "iv"
(for instantaneous),
"dv" (for daily values), "gwlevels" (for groundwater levels),
"site" (for site service), "measurement", and "stat" (for
statistics service). Note: "measurement" calls go to:
\url{https://nwis.waterdata.usgs.gov/usa/nwis} for data requests, and use different call requests schemes.
The statistics service has a limited selection of arguments
(see \url{https://waterservices.usgs.gov/docs/site-service/}).}

\item{asDateTime}{logical, if \code{TRUE} returns date and time as POSIXct, if \code{FALSE}, Date}

\item{convertType}{logical, defaults to \code{TRUE}. If \code{TRUE}, the
function will convert the data to dates, datetimes,
numerics based on a standard algorithm. If false, everything is returned as a character}

\item{tz}{character to set timezone attribute of dateTime. Default is "UTC", and converts the
date times to UTC, properly accounting for daylight savings times based on the data's provided tz_cd column.
Possible values to provide are "America/New_York", "America/Chicago", "America/Denver", "America/Los_Angeles",
"America/Anchorage", as well as the following which do not use daylight savings time: "America/Honolulu",
"America/Jamaica", "America/Managua", "America/Phoenix", and "America/Metlakatla". See also  \code{OlsonNames()}
for more information on time zones.}
}
\value{
A data frame with the following columns:
\tabular{lll}{
Name \tab Type \tab Description \cr
agency \tab character \tab The NWIS code for the agency reporting the data\cr
site \tab character \tab The USGS site number \cr
dateTime \tab POSIXct \tab The date and time (if applicable) of the measurement,
converted to UTC for unit value data. R only allows one time zone attribute per column. For unit data
spanning a time zone change, converting the data to UTC solves this problem. For daily data,
the time zone attribute is the time zone of the first returned measurement.
\cr
tz_cd \tab character \tab The time zone code for dateTime column\cr
code \tab character \tab Any codes that qualify the corresponding value\cr
value \tab numeric \tab The numeric value for the parameter \cr
}
Note that code and value are repeated for the parameters requested. The names are of the form
X_D_P_S, where X is literal,
D is an option description of the parameter,
P is the parameter code,
and S is the statistic code (if applicable).

There are also several useful attributes attached to the data frame:
\tabular{lll}{
Name \tab Type \tab Description \cr
url \tab character \tab The url used to generate the data \cr
siteInfo \tab data.frame \tab A data frame containing information on the requested sites \cr
variableInfo \tab data.frame \tab A data frame containing information on the requested parameters \cr
statisticInfo \tab data.frame \tab A data frame containing information on the requested statistics on the data \cr
queryTime \tab POSIXct \tab The time the data was returned \cr
}
}
\description{
Returns data from the NWIS web service.
Arguments to the function should be based on \url{https://waterservices.usgs.gov} service calls.
See examples below for ideas of constructing queries.
}
\details{
This function requires users to create their own arguments
based on the NWIS web services. It is a more complicated function to use
compared to other NWIS functions such as \code{\link[=readNWISdv]{readNWISdv()}}, \code{\link[=readNWISuv]{readNWISuv()}},
\code{\link[=readNWISgwl]{readNWISgwl()}}, etc. However, this function adds a lot of
flexibility to the possible queries. This function will also behave exactly
as NWIS when it comes to date queries. NWIS by default will only return the latest
value for the daily and instantaneous services. So if you do not provide
a starting date, you will only get back the latest value. If you want
the full period of record, you can use "startDate = '1900-01-01'". Other options
for dates are periods, such as "period = 'P7D'" which translates to a period
of 7 days. For period, use only a positive ISO-8601 duration format, which should
not be expressed in periods of less than a day, or in increments of months M or years Y.
period returns data for a site generally from now to a time in the past.
Note that when period is used all data up to the most recent value are returned.
}
\examples{
\dontshow{if (is_dataRetrieval_user()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
# Examples not run for time considerations

instFlow <- readNWISdata(
  sites = "05114000", service = "iv",
  parameterCd = "00060",
  startDate = "2014-05-01T00:00Z", endDate = "2014-05-01T12:00Z"
)

instFlowCDT <- readNWISdata(
  sites = "05114000", service = "iv",
  parameterCd = "00060",
  startDate = "2014-05-01T00:00", endDate = "2014-05-01T12:00",
  tz = "America/Chicago"
)

multiSite <- readNWISdata(
  sites = c("04025500", "040263491"),
  service = "iv", parameterCd = "00060"
)


GWL <- readNWISdata(site_no = c("392725077582401", 
                                "375907091432201"),
                    parameterCd = "62610",
                    service = "gwlevels")
                    
levels <- readNWISdata(stateCd = "WI", 
                       service = "gwlevels",
                       startDate = "2024-05-01",
                       endDate = "2024-05-30") 
                    
meas <- readNWISdata(
  state_cd = "WI", service = "measurements",
  format = "rdb_expanded"
)

waterYearStat <- readNWISdata(
  site = c("01646500"),
  service = "stat",
  statReportType = "annual",
  statYearType = "water",
  missingData = "on"
)
monthlyStat <- readNWISdata(
  site = c("01646500"),
  service = "stat",
  statReportType = "monthly"
)

dailyStat <- readNWISdata(
  site = c("01646500"),
  service = "stat",
  statReportType = "daily",
  statType = c("p25", "p50", "p75", "min", "max"),
  parameterCd = "00060"
)

arg.list <- list(
  site = "03111548",
  statReportType = "daily",
  statType = c("p25", "p50", "p75", "min", "max"),
  parameterCd = "00060"
)
allDailyStats_2 <- readNWISdata(arg.list, service = "stat")


site_id <- "01594440"
rating_curve <- readNWISdata(service = "rating", site_no = site_id, file_type = "base")
all_sites_base <- readNWISdata(service = "rating", file_type = "base")
all_sites_core <- readNWISdata(service = "rating", file_type = "corr")
all_sites_exsa <- readNWISdata(service = "rating", file_type = "exsa")
all_sites_24hrs <- readNWISdata(service = "rating", file_type = "exsa", period = 24)

peak_data <- readNWISdata(
  service = "peak",
  site_no = c("01594440", "040851325"),
  range_selection = "data_range"
)

peak_data <- readNWISdata(
  service = "peak",
  state_cd = "PA"
)

peak_data <- readNWISdata(
  service = "peak",
  huc2_cd = "20"
)
}
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=read_waterdata]{read_waterdata()}}
}
