#' Compute t-values for regression coefficients
#'
#' @param beta_hat Numeric vector. Estimated regression coefficients.
#' @param Y_fuzzy List. Observed fuzzy responses.
#' @param Y_pred List. Predicted fuzzy responses.
#' @param XtX_inv Matrix. Inverse of the XtX matrix.
#' @return Numeric vector of t-values for the regression coefficients.
#' @examples
#' # Example setup
#' beta_hat <- c(0.5, 1.2)  # Example regression coefficients
#' Y_fuzzy <- list(
#'   list(l = 2.1, x = 2.3, r = 2.5),
#'   list(l = 3.1, x = 3.3, r = 3.5),
#'   list(l = 4.1, x = 4.3, r = 4.5)
#' )  # Example fuzzy response
#' Y_pred <- list(
#'   list(l = 2.0, x = 2.2, r = 2.4),
#'   list(l = 3.0, x = 3.2, r = 3.4),
#'   list(l = 4.0, x = 4.2, r = 4.4)
#' )  # Example predicted values
#' XtX_inv <- matrix(c(0.1, 0.2, 0.2, 0.4), ncol = 2)  # Example XtX_inv matrix
#' t_values <- compute_t_values(beta_hat, Y_fuzzy, Y_pred, XtX_inv)
#' print(t_values)
#' @export
compute_t_values <- function(beta_hat, Y_fuzzy, Y_pred, XtX_inv) {
  n <- length(Y_fuzzy)
  p <- length(beta_hat) - 1
  df <- n - (p + 1)
  if (df <= 0) stop("Degrees of freedom are non-positive. Check your model or data.")

  residuals <- compute_res(Y_fuzzy, Y_pred)
  rss <- sum(sapply(residuals, function(res) sum(unlist(res)^2)))
  sigma_squared <- rss / df

  se_beta <- sqrt(diag(sigma_squared * XtX_inv))
  t_values <- beta_hat / se_beta

  t_values
}


#' Compute p-values for regression coefficients
#'
#' @param t_values Numeric vector. T-values for regression coefficients.
#' @param df Integer. Degrees of freedom.
#' @return Numeric vector of p-values for each coefficient.
#' @examples
#' t_values <- c(2.5, 3.0)  # Example t-values
#' df <- 30  # Example degrees of freedom
#' p_values <- compute_p_values(t_values, df)
#' print(p_values)
#' @importFrom stats pt
#' @export
compute_p_values <- function(t_values, df) {
  if (any(is.na(t_values))) {
    stop("t_values contain NA values. Check your data or model.")
  }
  if (df <= 0) {
    stop("Degrees of freedom must be positive.")
  }

  # Two-tailed p-values
  2 * pt(-abs(t_values), df = df)
}

#' Compute confidence intervals for regression coefficients
#'
#' @param beta_hat Numeric vector. Estimated regression coefficients.
#' @param se_beta Numeric vector. Standard errors of coefficients.
#' @param df Integer. Degrees of freedom.
#' @param alpha Numeric. Significance level (default: 0.05).
#' @return A list containing lower and upper bounds of confidence intervals.
#' @examples
#' beta_hat <- c(0.5, 1.2)  # Example regression coefficients
#' se_beta <- c(0.1, 0.2)  # Example standard errors
#' df <- 30  # Example degrees of freedom
#' ci <- compute_ci(beta_hat, se_beta, df)
#' print(ci)
#' @importFrom stats qt
#' @export
compute_ci <- function(beta_hat, se_beta, df, alpha = 0.05) {
  if (any(is.na(se_beta)) || df <= 0) {
    stop("Invalid input for standard errors or degrees of freedom.")
  }

  # Critical t-value for confidence intervals
  t_critical <- qt(1 - alpha / 2, df = df)

  # Compute confidence intervals
  lower_bounds <- beta_hat - t_critical * se_beta
  upper_bounds <- beta_hat + t_critical * se_beta

  list(
    lower = lower_bounds,
    upper = upper_bounds
  )
}

.add_significance_stars <- function(p_values) {
  if (any(is.na(p_values))) {
    stop("p_values contain NA values. Check your data or model.")
  }

  sapply(p_values, function(p) {
    if (p < 0.001) "***"
    else if (p < 0.01) "**"
    else if (p < 0.05) "*"
    else if (p < 0.1) "."
    else " "
  })
}

