% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/genDiag.R
\name{Maf}
\alias{Maf}
\alias{Maf.data.frame}
\alias{genDiag}
\alias{Maf.rmult}
\alias{Maf.aplus}
\alias{Maf.rplus}
\alias{Maf.ccomp}
\alias{Maf.rcomp}
\alias{Maf.acomp}
\alias{UWEDGE}
\alias{UWEDGE.default}
\alias{UWEDGE.acomp}
\alias{UWEDGE.rcomp}
\alias{RJD}
\alias{RJD.default}
\alias{RJD.acomp}
\alias{RJD.rcomp}
\title{Generalised diagonalisations
Calculate several generalized diagonalisations out of a data set and its empirical
variogram}
\usage{
Maf(x, ...)

\method{Maf}{data.frame}(x, vg, i = 2, ...)

\method{Maf}{rmult}(x, vg, i = 2, ...)

\method{Maf}{aplus}(x, vg, i = 2, ...)

\method{Maf}{rplus}(x, vg, i = 2, ...)

\method{Maf}{ccomp}(x, vg, i = 2, ...)

\method{Maf}{rcomp}(x, vg, i = 2, ...)

\method{Maf}{acomp}(x, vg, i = 2, ...)

UWEDGE(x, ...)

\method{UWEDGE}{default}(x, ...)

\method{UWEDGE}{acomp}(x, vg, i = NULL, ...)

\method{UWEDGE}{rcomp}(x, vg, i = NULL, ...)

RJD(x, ...)

\method{RJD}{default}(x, ...)

\method{RJD}{acomp}(x, vg, i = NULL, ...)

\method{RJD}{rcomp}(x, vg, i = NULL, ...)
}
\arguments{
\item{x}{a data set, typically of class "data.frame" or of a compositional class}

\item{...}{generic functionality arguments}

\item{vg}{empirical variogram, of a kind fitting to the data provided}

\item{i}{a slicer for the variogram, typically this will be one or more indices of
the lag distance to take.}
}
\value{
An object extending \code{c("princomp.CLASSOF(x)",}"\code{\link[stats]{princomp}}")
with classes "\code{genDiag}", and an extra class argument depending on the
diagonalisation method chosen.

Function \code{Maf} results carry the extra class "\code{maf}", and they correspond
to minimum/maximum autocorrelation factors (MAF) as proposed by Switzer and Green
(1984). In this case, the
slicer is typically just the index of one lag distance (defaults to i=2). MAF
provides the analytical solution to the joint diagonalisation of two matrices,
the covariance of increments provided by the slicing and the conventional covariance
matrix (of the idt transformed values, if appropriate). Resulting factors are ordered
in decreasing order of spatial continuity, which might produce surprising
scree-plots for those who are used to see a screeplot of a principal component analysis.

Function \code{UWEDGE} (Uniformly Weighted Exhaustive Diagonalization with Gauss
iterations; Tichavsky and Yeredor, 2009) results carry the extra class "\code{uwedge}".
The function
is a wrapper on \code{jointDiag::uwedge} from package \code{jointDiag} (Gouy-Pailler, 2017).
In this case the
slicer is typically just a subset of indices of lag distances to consider
(defaults to the nearest indexes to mininum, maximum and mean lag distances of the
variogram). UWEDGE iteratively seeks for a pair of matrices (a mixing and a
demixing matrices) diagonalises the set of matrices \eqn{M_1, M_2, \ldots, M_K}
given, by minimizing the target quantity
\deqn{Q_{uwedge}(A, W) = \sum_{k=1}^K Tr[E_k^t\cdot E_k],}
where \eqn{E_k = (W^t\cdot M_k \cdot W- A\cdot \Lambda_k\cdot A^t)} and
\eqn{\Lambda_k = diag(W^t\cdot M_k \cdot W)} is the resulting diagonalized version of
each matrix. Obtained factors are ordered
in decreasing order of spatial continuity, which might produce surprising
scree-plots for those who are used to see a screeplot of a principal component analysis.

Function \code{RJD} results carry the extra class "\code{rjd}". The function
is a wrapper on \code{JADE::rjd} (Miettinen, Nordhausen and Taskinen, 2017),
implementing the Rotational joint diagonalisation method (Cardoso and Souloumiac, 1996).
In this case the
slicer is typically just a subset of indices of lag distances to consider
(defaults to the nearest indexes to mininum, maximum and mean lag distances).
This algorithm also served for quasi-diagonalising a set of matrices as in UWEDGE,
just that in this case the quantity to minimise is the sum of sequares of all off-diagonal
elements of \eqn{A^t\cdot M_k\cdot A} for all \eqn{k=1, 2, \ldots K}.

All these functions produce output mimicking \code{\link{princomp}}, i.e. with
elements
\describe{
\item{sdev}{contrary to the output in PCA, this contains the square root of the
metric variance of the predictions obtained for each individual factor; this is the
quantity needed for \code{\link[stats]{screeplot}} to create plots of explained variance
by factor}
\item{loadings}{matrix of contributions of each (cdt-transformed) original variable to the new factors}
\item{center}{center of the data set (eventually, represented through \code{\link[compositions]{cdt}}),
in compositional methods}
\item{scale}{the scalings applied to each original variable}
\item{n.obs}{number of observations}
\item{scores}{the scores of the supplied data on the new factors}
\item{call}{the call to the function (attention: it actually may come much later)}
}
and additionally some of the following arguments, in different order
\describe{
\item{invLoadings}{matrix of contributions of each factor onto each original variable}
\item{Center}{compositional methods return here the cdt-backtransformed center}
\item{InvLoadings}{compositional methods return here the clr-backtransformed inverse loadings, so that
each column of this matrix can be understood as a composition on itself}
\item{DownInvLoadings}{compositional methods return here the clr-backtransformed "minus inverse loadings", so that
each column of this matrix can be understood as a composition on itself; details in
\code{\link[compositions]{princomp.acomp}} }
\item{C1, C2}{Maf returns the two matrices that were diagonalised}
\item{eigenvalues}{Maf returns the generalized eigenvalues of the diagonalisation of C1 and C2}
\item{gof}{UWEDGE returns the values of the goodness of fit criterion across sweeps}
\item{diagonalized}{RJD returns the diagonalized matrices, in an array of (K,D,D)-dimensions, being
D the number of variables in \code{x}}
\item{type}{a string describing which package and which function was used as a workhorse for
the calculation}
}

NOTE: if the arguments you provide to RJD and UWEDGE are not of the appropriate type
(i.e. data.frames or equivalent) the default method of these functions will just attempt
to call the basis functions JADE:rjd and JointDiag:uwedge respectively.
This will be the case if you provide \code{x} as a "\code{matrix}", or as
an "\code{array}". In those cases, the output will NOT be structured as an extension
to princomp results; instead they will be native output from those functions.
}
\description{
Generalised diagonalisations
Calculate several generalized diagonalisations out of a data set and its empirical
variogram
}
\section{Functions}{
\itemize{
\item \code{Maf}: Generalised diagonalisations

\item \code{Maf.rmult}: Generalised diagonalisations

\item \code{Maf.aplus}: Generalised diagonalisations

\item \code{Maf.rplus}: Generalised diagonalisations

\item \code{Maf.ccomp}: Generalised diagonalisations

\item \code{Maf.rcomp}: Generalised diagonalisations

\item \code{Maf.acomp}: Generalised diagonalisations

\item \code{UWEDGE}: Generalised diagonalisations

\item \code{UWEDGE.default}: Generalised diagonalisations

\item \code{UWEDGE.acomp}: Generalised diagonalisations

\item \code{UWEDGE.rcomp}: Generalised diagonalisations

\item \code{RJD}: Generalised diagonalisations

\item \code{RJD.default}: Generalised diagonalisations

\item \code{RJD.acomp}: Generalised diagonalisations

\item \code{RJD.rcomp}: Generalised diagonalisations
}}

\examples{
require("magrittr")
require("gstat")
require("compositions")
data("jura", package="gstat")
gs = gstat(id="Cd", formula=log(Cd)~1, locations=~Xloc+Yloc, data=jura.pred) \%>\% 
gstat(id="Co", formula=log(Cd)~1, locations=~Xloc+Yloc, data=jura.pred) \%>\% 
gstat(id="Cr", formula=log(Cr)~1, locations=~Xloc+Yloc, data=jura.pred) \%>\% 
gstat(id="Cu", formula=log(Cu)~1, locations=~Xloc+Yloc, data=jura.pred) \%>\% 
gstat(id="Ni", formula=log(Ni)~1, locations=~Xloc+Yloc, data=jura.pred) \%>\% 
gstat(id="Pb", formula=log(Pb)~1, locations=~Xloc+Yloc, data=jura.pred) \%>\% 
gstat(id="Zn", formula=log(Zn)~1, locations=~Xloc+Yloc, data=jura.pred)
vg = variogram(gs)
mf = Maf(aplus(jura.pred[, -(1:6)]), vg=vg)
mf
mf$loadings
biplot(mf)
}
\references{
Cardoso, J. K. and Souloumiac A. 1996. Jacobi angles for simultaneous
diagonalization. SIAM Journal of Matrix Analysis and Applications 17(1), 161-164.

Gouy-Pailler C., 2017. jointDiag: Joint approximate diagonalization of a set of
square matrices. R package version 0.3. https://CRAN.R-project.org/package=jointDiag

Miettinen J., Nordhausen K., and Taskinen, S., 2017. Blind source separation based
on Joint diagonalization in R: The packages JADE and BSSasymp. Journal of Statistical
Software 76(2), 1-31.

Switzer P. and Green A.A., 1984. Min/Max autocorrelation factors for multivariate
spatial imaging, Stanford University, Palo Alto, USA, 14pp.

Tichavsky, P. and Yeredor, A., 2009. Fast approximate joint diagonalization
incorporating weight matrices. IEEE Transactions on Signal Processing 57, 878 ??? 891.
}
\seealso{
Other generalised Diagonalisations: 
\code{\link{coloredBiplot.genDiag}()},
\code{\link{predict.genDiag}()}
}
\concept{generalised Diagonalisations}
