% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tab_create_modify.R
\name{tab_stub_indent}
\alias{tab_stub_indent}
\title{Control indentation of row labels in the stub}
\usage{
tab_stub_indent(data, rows, indent = "increase")
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{rows}{\emph{Rows to target}

\verb{<row-targeting expression>} // \strong{required}

The rows to consider for the indentation change. We can supply a vector of
row ID values within \code{c()}, a vector of row indices, or use select helpers
here (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}},
\code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}). We can also use expressions to filter
down to the rows we need (e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{indent}{\emph{Indentation directive}

\verb{scalar<character|numeric|integer>} // \emph{default:} \code{"increase"}

An indentation directive either as a keyword describing the indentation
change or as an explicit integer value for directly setting the indentation
level. The keyword \code{"increase"} (the default) will increase the indentation
level by one; \code{"decrease"} will do the same in the reverse direction. The
starting indentation level of \code{0} means no indentation and this values
serves as a lower bound. The upper bound for indentation is at level \code{5}.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Indentation of row labels is an effective way for establishing structure in a
table stub. \code{tab_stub_indent()} allows for fine control over row label
indentation in the stub. We can use an explicit definition of an indentation
level (with a number between \code{0} and \code{5}), or, employ an indentation
directive using keywords (\code{"increase"}/\code{"decrease"}).
}
\section{Compatibility of arguments with the \code{from_column()} helper function}{


\code{\link[=from_column]{from_column()}} can be used with the \code{indent} argument of \code{tab_stub_indent()}
to obtain varying parameter values from a specified column within the table.
This means that each row label could be indented a little bit differently.

Please note that for this argument (\code{indent}), a \code{\link[=from_column]{from_column()}} call needs
to reference a column that has data of the \code{numeric} or \code{integer} type.
Additional columns for parameter values can be generated with \code{\link[=cols_add]{cols_add()}}
(if not already present). Columns that contain parameter data can also be
hidden from final display with \code{\link[=cols_hide]{cols_hide()}}.
}

\section{Examples}{


Using a subset of the \code{\link{photolysis}} dataset within a \strong{gt} table, we can
provide some indentation to all of the row labels in the stub via
\code{tab_stub_indent()}. Here we provide an \code{indent} value of \code{3} for a very
prominent indentation that clearly shows that the row labels are subordinate
to the two row group labels in this table (\code{"inorganic reactions"} and
\code{"carbonyls"}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{photolysis |>
  dplyr::select(cmpd_name, products, type, l, m, n) |>
  dplyr::slice_head(n = 10) |>
  gt(groupname_col = "type", rowname_col = "cmpd_name") |>
  fmt_chem(columns = products) |>
  fmt_scientific(columns = l) |>
  tab_stub_indent(
    rows = everything(),
    indent = 3
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_tab_stub_indent_1.png" alt="This image of a table was generated from the first code example in the `tab_stub_indent()` help file." style="width:100\%;">
}}

Let's use a summarized version of the \code{\link{pizzaplace}} dataset to create a
another \strong{gt} table with row groups and row labels. With \code{\link[=summary_rows]{summary_rows()}},
we'll generate summary rows at the top of each row group. Using
\code{tab_stub_indent()} we can add indentation to the row labels in the stub.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{pizzaplace |>
  dplyr::group_by(type, size) |>
  dplyr::summarize(
    sold = dplyr::n(),
    income = sum(price),
    .groups = "drop"
  ) |>
  gt(rowname_col = "size", groupname_col = "type") |>
  tab_header(title = "Pizzas Sold in 2015") |>
  fmt_integer(columns = sold) |>
  fmt_currency(columns = income) |>
  summary_rows(
    fns = list(label = "All Sizes", fn = "sum"),
    side = "top",
    fmt = list(
      ~ fmt_integer(., columns = sold),
      ~ fmt_currency(., columns = income)
    )
  ) |>
  tab_options(
    summary_row.background.color = "gray95",
    row_group.background.color = "#FFEFDB",
    row_group.as_column = TRUE
  ) |>
  tab_stub_indent(
    rows = everything(),
    indent = 2
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_tab_stub_indent_2.png" alt="This image of a table was generated from the second code example in the `tab_stub_indent()` help file." style="width:100\%;">
}}

Indentation of entries in the stub can be controlled by values within a
column. Here's an example of that using the \code{\link{constants}} dataset, where
variations of a row label are mutated to eliminate the common leading text
(replacing it with \code{"..."}). At the same time, the indentation for those rows
is set to \code{4} in the \code{indent} column (value is \code{0} otherwise). The
\code{tab_stub_indent()} statement uses \code{\link[=from_column]{from_column()}}, which passes values from
the \code{indent} column to the namesake argument. We hide the \code{indent} column
from view by use of \code{\link[=cols_hide]{cols_hide()}}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{constants |>
  dplyr::select(name, value, uncert, units) |>
  dplyr::filter(
    grepl("^atomic mass constant", name) |
      grepl("^Rydberg constant", name) |
      grepl("^Bohr magneton", name)
  ) |>
  dplyr::mutate(
    indent = ifelse(grepl("constant |magneton ", name), 4, 0),
    name = gsub(".*constant |.*magneton ", "...", name)
  ) |>
  gt(rowname_col = "name") |>
  tab_stubhead(label = "Physical Constant") |>
  tab_stub_indent(
    rows = everything(),
    indent = from_column(column = "indent")
  ) |>
  fmt_scientific(columns = c(value, uncert)) |>
  fmt_units(columns = units) |>
  cols_hide(columns = indent) |>
  cols_label(
    value = "Value",
    uncert = "Uncertainty",
    units = "Units"
  ) |>
  cols_width(
    stub() ~ px(250),
    c(value, uncert) ~ px(150),
    units ~ px(80)
  ) |>
  tab_style(
    style = cell_text(indent = px(10)),
    locations = list(
      cells_column_labels(columns = units),
      cells_body(columns = units)
    )
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_tab_stub_indent_3.png" alt="This image of a table was generated from the third code example in the `tab_stub_indent()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

2-6
}

\section{Function Introduced}{

\code{v0.7.0} (Aug 25, 2022)
}

\seealso{
Other part creation/modification functions: 
\code{\link{tab_caption}()},
\code{\link{tab_footnote}()},
\code{\link{tab_header}()},
\code{\link{tab_info}()},
\code{\link{tab_options}()},
\code{\link{tab_row_group}()},
\code{\link{tab_source_note}()},
\code{\link{tab_spanner}()},
\code{\link{tab_spanner_delim}()},
\code{\link{tab_stubhead}()},
\code{\link{tab_style}()},
\code{\link{tab_style_body}()}
}
\concept{part creation/modification functions}
