#' Text Annotation Tool for R
#'
#' @description A Shiny application for interactive text annotation and analysis
#'
#' @name textAnnotatoR
#' @docType package
#' @author Chao Liu
#'
#' @section Dependencies:
#' This package requires the following packages:
#' \itemize{
#'   \item shiny
#'   \item data.tree
#'   \item jsonlite
#'   \item shinydashboard
#'   \item DT
#'   \item readtext
#'   \item magrittr
#' }
#'
#' @keywords internal
"_PACKAGE"

#' Interactive Text Annotation Interface
#'
#' @title Text Annotation GUI
#' @description Launch an interactive Shiny application for text annotation and analysis.
#' The GUI provides tools for importing text, applying codes, creating memos,
#' and analyzing annotations through various visualizations.
#'
#' @details The annotation interface includes the following features:
#' \itemize{
#'   \item Text import and display
#'   \item Code application and management
#'   \item Memo creation and linking
#'   \item Project management (save/load)
#'   \item Annotation analysis tools
#'   \item Export capabilities
#' }
#'
#' @note This package provides functionality for users to interactively save files
#' through the Shiny interface. All file operations are explicitly initiated by
#' users through file dialogs, and no files are written automatically to the user's
#' system without their direct action and consent.
#'
#' @return Launches a Shiny application in the default web browser
#' @export
#'
#' @examples
#' if(interactive()) {
#'   annotate_gui()
#' }
#'
#' @importFrom shiny runApp shinyApp fluidPage actionButton observeEvent renderUI
#'   showNotification showModal modalDialog removeModal updateTextAreaInput
#'   updateTextInput tabPanel fileInput renderTable renderPlot plotOutput
#'   tableOutput textInput textAreaInput selectInput checkboxGroupInput
#' @importFrom shinyjs useShinyjs toggle runjs
#' @importFrom data.tree Node
#' @importFrom jsonlite fromJSON toJSON write_json
#' @importFrom shinydashboard dashboardPage dashboardHeader dashboardSidebar dashboardBody tabBox
#' @importFrom DT renderDT formatStyle styleInterval datatable
#' @importFrom readtext readtext
#' @importFrom utils write.csv
#'
annotate_gui <- function() {
  if (!interactive()) {
    stop("annotate_gui() is only meant to be used in interactive R sessions")
  }
  # Try to set up resource path safely
  tryCatch({
    # For development, try local path first
    if (dir.exists("inst/www")) {
      addResourcePath("custom", "inst/www")
    } else {
      # Fall back to installed package path
      pkg_www <- system.file("www", package = "textAnnotatoR")
      if (pkg_www != "") {
        addResourcePath("custom", pkg_www)
      }
    }
  }, error = function(e) {
    warning("Could not set up resource path for custom files. Some UI elements might not display correctly.")
  })

  ui <- dashboardPage(
    dashboardHeader(title = "textAnnotatoR", titleWidth = 250),
    dashboardSidebar(disable = TRUE),
    dashboardBody(
      useShinyjs(),
      fluidRow(
        column(8,
               div(style = "margin-bottom: 15px",
                   actionButton("save_project", "Save Project", icon = icon("save")),
                   actionButton("load_project", "Load Project", icon = icon("folder-open")),
                   actionButton("new_project", "New Project", icon = icon("file")),
                   div(style = "display: inline-block; margin-left: 15px; border-left: 1px solid #ddd; padding-left: 15px;",
                       actionButton("export_hierarchy_toolbar", "Export Hierarchy", icon = icon("download")),
                       actionButton("import_hierarchy_toolbar", "Import Hierarchy", icon = icon("upload"))
                   )
               )
        ),
        column(4,
               div(id = "project_status",
                   style = "margin-bottom: 15px; text-align: right; padding-top: 7px;",
                   uiOutput("project_status_display")
               )
        )
      ),
      tags$head(
        # Import Google Font for brand name
        tags$link(rel = "stylesheet",
                  href = "https://fonts.googleapis.com/css2?family=Gamja+Flower&display=swap"),
        tags$script(src = "https://code.jquery.com/ui/1.12.1/jquery-ui.min.js"),
        tags$link(rel = "stylesheet", type = "text/css", href = "https://code.jquery.com/ui/1.12.1/themes/base/jquery-ui.css"),
        tags$style(HTML("
          /* Custom brand text styling */
          .skin-blue .main-header .logo .brand-text {
            font-family: 'Gamja Flower', sans-serif !important;
            font-weight: 800 !important;
            letter-spacing: 1px !important;
            text-transform: none !important;
            color: #FFFFFF !important;
            font-size: 24px !important;
          }
          .skin-blue .main-header .logo:hover {
            background-color: #367fa9 !important;
          }

          body { background-color: #f0f0f0; }
          .char { cursor: pointer; }
          .highlighted {
            background-color: #ffff00 !important;
            box-shadow: 0 0 3px rgba(255, 255, 0, 0.8);
          }

          /* NEW: Selection mode styles */
          .selection-active { cursor: crosshair !important; }
          .selection-active .char { cursor: crosshair !important; }
          .selection-active .char:hover {
            background-color: rgba(255, 255, 0, 0.3);
          }
          .selection-mode-indicator {
            position: fixed;
            top: 10px;
            right: 10px;
            background-color: #28a745;
            color: white;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 12px;
            z-index: 1001;
            display: none;
          }
          .selection-mode-indicator.active {
            display: block;
          }

          /* Toggle button styles */
          .btn-toggle-active {
            background-color: #007bff !important;
            color: white !important;
            border-color: #0056b3 !important;
            box-shadow: inset 0 3px 5px rgba(0,0,0,.125) !important;
          }

          .btn-toggle-inactive {
            background-color: #f8f9fa !important;
            color: #6c757d !important;
            border-color: #dee2e6 !important;
          }

          .code-display {
            padding: 2px 5px;
            margin-right: 5px;
            border-radius: 3px;
            font-weight: bold;
            color: black;
            display: inline-block;
          }
          .code-display br {
            display: block;        /* Make line breaks more prominent */
            content: \"\";           /* Ensure proper spacing */
            margin-top: 0.5em;     /* Add space between lines */
          }
          #annotations_table { margin-top: 20px; overflow-y: auto; max-height: 200px; }
          .content-wrapper { margin-left: 0 !important; }
          .tab-content { padding-top: 20px; }
          #margin_icons {
            position: fixed;
            right: 0;
            top: 50%;
            transform: translateY(-50%);
            z-index: 1000;
            background-color: rgba(248, 249, 250, 0.8);
            padding: 10px 5px;
            border-top-left-radius: 5px;
            border-bottom-left-radius: 5px;
          }
          #left_margin_icons {
            position: fixed;
            left: 0;
            top: 50%;
            transform: translateY(-50%);
            z-index: 1000;
            background-color: rgba(248, 249, 250, 0.8);
            padding: 10px 5px;
            border-top-right-radius: 5px;
            border-bottom-right-radius: 5px;
          }
          #margin_icons .btn, #left_margin_icons .btn {
            display: block;
            width: 40px;
            height: 40px;
            border-radius: 50%;
            margin-bottom: 10px;
            padding: 0;
            background-color: #f8f9fa;
            border: 1px solid #dee2e6;
            transition: background-color 0.3s;
          }
          #margin_icons .btn:hover, #left_margin_icons .btn:hover {
            background-color: #e9ecef;
          }
          #margin_icons .btn i, #left_margin_icons .btn i {
            font-size: 1.4rem;
          }
          #main_content {
            margin-left: 50px;
            margin-right: 50px;
          }
          #text_display, #floating_text_content {
            background-color: white;
            border: 1px solid #dee2e6;
            border-radius: 5px;
            padding: 20px;
            margin-top: 20px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            height: 400px;
            overflow-y: auto;
            white-space: pre-wrap; /* Preserve whitespace and wrap text */
            line-height: 1.5;      /* Improve readability */
          }
          .nav-tabs-custom {
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
          }
          .nav-tabs-custom, #text_display {
            width: 100%;
            max-width: 100%;
            box-sizing: border-box;
          }
          #floating_text_window {
            display: none;
            position: fixed;
            width: 400px;
            height: 300px;
            top: 100px;
            left: 100px;
            background-color: white;
            border: 1px solid #ddd;
            border-radius: 5px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            z-index: 1000;
          }
          #floating_text_window_header {
            padding: 10px;
            cursor: move;
            background-color: #f1f1f1;
            border-bottom: 1px solid #ddd;
          }
          #floating_text_content {
            padding: 10px;
            height: calc(100% - 40px);
            overflow-y: auto;
          }
          .main-header .logo {
            font-family: 'Gamja Flower', sans-serif !important;
            font-weight: 800 !important;
            letter-spacing: 1px !important;
            text-transform: none !important;
            color: #FFFFFF !important;
            font-size: 24px !important;
          }
          .theme-item, .code-item {
            display: inline-block;
            padding: 2px 4px;
            border-radius: 3px;
            transition: background-color 0.2s;
          }

          .theme-item:hover, .code-item:hover {
            background-color: rgba(0, 0, 0, 0.05);
          }

          .theme-item.selected, .code-item.selected {
            background-color: #e6f3ff;
          }

          #hierarchy_pre {
            font-family: monospace;
            line-height: 1.5;
            white-space: pre;
            margin: 10px 0;
            padding: 10px;
            background-color: #f8f9fa;
            border-radius: 4px;
            border: 1px solid #dee2e6;
          }
          /* Styling for the hierarchy container */
        .hierarchy-container {
          font-family: monospace;
          line-height: 1.5;
        }

        /* Selected item styling */
        .theme-item.selected, .code-item.selected {
          background-color: #e6f3ff;
          border-radius: 3px;
          padding: 2px 4px;
        }

        /* Theme item styling */
        .theme-item {
          font-weight: bold;
        }

        /* Code item styling */
        .code-item {
          font-style: normal;
        }

        /* Description preview styling */
        .description-preview {
          color: #666;
          font-style: italic;
          font-size: 0.9em;
        }
        ")),
        tags$head(tags$script(HTML(addJS))),
        tags$script(HTML("
        $(document).keydown(function(e) {
          // Ctrl+S for quick save
          if (e.ctrlKey && e.which === 83) {
            e.preventDefault();
            $('#save_project').click();
          }
        });
      ")),
        tags$script(HTML("
        $(document).ready(function() {
        // Initialize onclick handlers for themes and codes
        initializeThemeHandlers();

        // Function to set up the click handlers
        function initializeThemeHandlers() {
          // Add click handler for theme and code items
          $(document).on('click', '.theme-item, .code-item', function(e) {
            e.preventDefault();
            e.stopPropagation();

            // Remove selection from all items
            $('.theme-item, .code-item').removeClass('selected');

            // Add selection to clicked item
            $(this).addClass('selected');

            // Get the item name - this is the critical part
            var itemName = $(this).data('name');

            // Make sure we have a valid name
            if (!itemName) {
              itemName = $(this).text().trim();
              // Remove any description text if present
              if (itemName.indexOf(' - ') > -1) {
                itemName = itemName.split(' - ')[0].trim();
              }
            }

            // Send the selected item to Shiny
            Shiny.setInputValue('selected_theme', itemName);
          });

          // Add hover effects for better UX
          $(document).on('mouseenter', '.theme-item, .code-item', function() {
            $(this).css({
              'cursor': 'pointer',
              'text-decoration': 'underline'
            });
          }).on('mouseleave', '.theme-item, .code-item', function() {
            $(this).css({
              'cursor': 'default',
              'text-decoration': 'none'
            });
          });
        }
      });
          $(document).ready(function() {
            $('#floating_text_window').draggable({
              handle: '#floating_text_window_header',
              containment: 'window'
            });
            $('#floating_text_window').resizable();
          });
          function toggleTextWindow() {
            $('#floating_text_window').toggle();
          }
        ")),
        tags$script(HTML("
          // Additional refresh handler for import text functionality
          Shiny.addCustomMessageHandler('refreshDisplay', function(message) {
            // Clear any lingering highlights
            $('.char').removeClass('highlighted');

            // Re-initialize selection handlers
            setTimeout(function() {
              if (typeof initializeSelection === 'function') {
                initializeSelection();
              }
            }, 100);

            // Scroll to top of text display
            $('#text_display').scrollTop(0);
            $('#floating_text_content').scrollTop(0);
          });
        "))
      ),
      # Add a button to toggle the floating text window
      actionButton("toggle_text_window", "Toggle Text Window"),

      # Add the floating text window
      div(id = "floating_text_window",
          div(id = "floating_text_window_header", "Text Display"),
          div(id = "floating_text_content", uiOutput("floating_text_display"))
      ),
      div(id = "margin_icons",
          actionButton("select", "",
                       icon = icon("mouse-pointer"),
                       title = "Select text for annotation"),
          actionButton("clear", "",
                       icon = icon("eraser"),
                       title = "Clear current selection")
      ),
      div(id = "left_margin_icons",
          actionButton("undo", "",
                       icon = icon("undo"),
                       title = "Undo last action"),
          actionButton("redo", "",
                       icon = icon("redo"),
                       title = "Redo last undone action")
      ),
      div(id = "main_content",
          tabBox(
            width = NULL,
            id = "tabset",
            tabPanel("File", icon = icon("file"),
                     # Add current file display
                     conditionalPanel(
                       condition = "output.current_file_name",
                       div(style = "margin-bottom: 15px; padding: 10px; background-color: #e8f4e8; border-radius: 5px;",
                           tags$p(
                             icon("file", class = "text-success"),
                             strong("Current file: "),
                             textOutput("current_file_name", inline = TRUE)
                           )
                       )
                     ),
                     fileInput("file_input", "Choose File",
                               accept = c(".txt", ".docx", ".pdf")),
                     actionButton("import_text", "Import Text")
            ),
            tabPanel("Code and Memo", icon = icon("code"),
                     textInput("code", "Code:"),
                     textAreaInput("memo", "Memo:", height = "100px"),
                     actionButton("create_code", "Create Code", icon = icon("plus")),
                     actionButton("apply_code", "Apply Code", icon = icon("check")),
                     actionButton("merge_codes", "Merge Codes", icon = icon("object-group")),
                     actionButton("save_code", "Save Annotated Text", icon = icon("download"))
                     #actionButton("rename_code", "Rename Code", icon = icon("edit")),
                     #actionButton("delete_code", "Delete Code", icon = icon("trash")),
                     #actionButton("display_code", "Display Code", icon = icon("eye"))
            ),
            tabPanel("Codebook", icon = icon("book"),
                     fluidRow(
                       column(6,
                              wellPanel(
                                tags$h4("Code Management"),
                                # Code creation
                                textInput("codebook_new_code", "New Code Name:"),
                                textAreaInput("codebook_code_description", "Description:", height = "80px"),
                                actionButton("codebook_create_code", "Create Code",
                                             icon = icon("plus")),
                                tags$hr(),

                                # Code operations with improved instructions
                                tags$h5("Existing Codes"),
                                tags$p(class = "text-muted",
                                       "Click on rows to select codes. Hold Ctrl/Cmd to select multiple codes for merging."),
                                DTOutput("codebook_codes_table"),
                                tags$br(),

                                # Selection feedback
                                uiOutput("selection_feedback"),
                                tags$br(),

                                # Action buttons
                                div(style = "margin-top: 10px;",
                                    actionButton("codebook_rename_code", "Rename Selected",
                                                 icon = icon("edit")),
                                    actionButton("codebook_delete_code", "Delete Selected",
                                                 icon = icon("trash"), class = "btn-danger"),
                                    actionButton("codebook_merge_codes", "Merge Selected",
                                                 icon = icon("object-group"))
                                )
                              )
                       ),
                       column(6,
                              wellPanel(
                                tags$h4("Code Statistics"),
                                DTOutput("codebook_usage_stats")
                              ),
                              wellPanel(
                                tags$h4("Selected Code Details"),
                                uiOutput("codebook_selected_details")
                              )
                       )
                     )
            ),
            tabPanel("Themes & Codes", icon = icon("folder-tree"),
                     fluidRow(
                       column(4,
                              wellPanel(
                                h4("Organization Management"),
                                tags$p("Organize your codes into hierarchical themes for better analysis."),
                                actionButton("add_theme_btn", "Add Theme", icon = icon("folder-plus")),
                                actionButton("add_code_to_theme_btn", "Add Code to Theme", icon = icon("code")),
                                actionButton("move_item_btn", "Move Item", icon = icon("arrows-alt")),
                                hr(),
                                actionButton("export_hierarchy_btn", "Export Hierarchy", icon = icon("download")),
                                actionButton("import_hierarchy_btn", "Import Hierarchy", icon = icon("upload"))
                              ),
                              wellPanel(
                                h4("Hierarchy Statistics"),
                                tags$p("The hierarchy consists of themes (folders) containing codes (annotations you apply to text)."),
                                verbatimTextOutput("hierarchy_stats")
                              )
                       ),
                       column(8,
                              wellPanel(
                                h4("Theme & Code Hierarchy"),
                                tags$p("This hierarchy shows the organizational structure of your themes and codes:"),
                                tags$ul(
                                  tags$li(HTML(paste0("<strong>Themes</strong> (", "&#128194;", ") - Used to group and organize related codes"))),
                                  tags$li(HTML(paste0("<strong>Codes</strong> (", "&#128196;", ") - Used to annotate and categorize text segments")))
                                ),
                                tags$p("Click on any theme or code to see its details."),
                                uiOutput("hierarchy_view")
                              ),
                              wellPanel(
                                h4("Selected Item Details"),
                                tags$p("Select a theme or code in the hierarchy above to view its details."),
                                uiOutput("theme_details")
                              )
                       )
                     )
            ),
            #tabPanel("Tools", icon = icon("tools"),
            #         actionButton("link_memo", "Link Memo", icon = icon("link")),
            #         actionButton("generate_codebook", "Generate Code Book", icon = icon("book"))
            #),
            tabPanel("Analysis", icon = icon("chart-bar"),
                     fluidRow(
                       column(6,
                              wellPanel(
                                h4("Analysis Settings"),
                                selectInput("cooccurrence_unit",
                                            "Co-occurrence Analysis Unit:",
                                            choices = c("Paragraph" = "paragraph",
                                                        "Sentence" = "sentence",
                                                        "Document" = "document"),
                                            selected = "paragraph"),
                                helpText("Select the analytical unit for co-occurrence analysis:",
                                         tags$br(),
                                         tags$strong("Sentence:"), " Codes appearing in the same sentence",
                                         tags$br(),
                                         tags$strong("Paragraph:"), " Codes appearing in the same paragraph",
                                         tags$br(),
                                         tags$strong("Document:"), " Codes appearing anywhere in the document")
                              )
                       ),
                       column(6,
                              wellPanel(
                                h4("Analysis Tools"),
                                actionButton("code_frequency", "Code Frequency", icon = icon("chart-bar")),
                                br(), br(),
                                actionButton("code_co_occurrence", "Code Co-occurrence", icon = icon("project-diagram")),
                                br(), br(),
                                actionButton("word_cloud", "Word Cloud", icon = icon("cloud")),
                                br(), br(),
                                actionButton("text_summary", "Text Summary", icon = icon("file-alt"))
                              )
                       )
                     )
            ),
            #tabPanel("Import/Export", icon = icon("exchange-alt"),
            #         actionButton("import_annotations", "Import Annotations", icon = icon("file-import")),
            #         actionButton("export_annotations", "Export Annotations", icon = icon("file-export"))
            #),
            tabPanel("Records", icon = icon("table"),
                     div(id = "annotations_table", DTOutput("annotations")),
                     actionButton("save_records", "Save Records", icon = icon("save"))
            ),
            tabPanel("Comparison", icon = icon("balance-scale"),
                     fluidRow(
                       column(4,
                              wellPanel(
                                h4("Import Comparison Data"),
                                # First file upload
                                fileInput("comparison_file1",
                                          "Upload First File (CSV/JSON)",
                                          multiple = FALSE,
                                          accept = c(".csv", ".json")),
                                # Show first file info when uploaded
                                uiOutput("file1_info"),

                                # Second file upload
                                fileInput("comparison_file2",
                                          "Upload Second File (CSV/JSON)",
                                          multiple = FALSE,
                                          accept = c(".csv", ".json")),
                                # Show second file info when uploaded
                                uiOutput("file2_info"),

                                # Add reset button
                                actionButton("reset_comparison", "Reset Files",
                                             icon = icon("trash-alt"),
                                             class = "btn-warning"),

                                br(), br(),

                                # Run comparison button (disabled until both files are uploaded)
                                actionButton("run_comparison", "Run Comparison",
                                             icon = icon("play"))
                              ),
                              wellPanel(
                                h4("Comparison Settings"),
                                checkboxGroupInput("comparison_metrics",
                                                   "Select Analysis Types:",
                                                   choices = c(
                                                     "Coverage Patterns" = "coverage",
                                                     "Code Application" = "application",
                                                     "Code Overlaps" = "overlaps",
                                                     "Code Sequences" = "sequences"
                                                   ),
                                                   selected = c("coverage", "application"))
                              )
                       ),
                       column(8,
                              tabsetPanel(
                                id = "comparison_results_tabs",
                                tabPanel("Summary",
                                         verbatimTextOutput("comparison_summary")),
                                tabPanel("Visualization",
                                         selectInput("plot_type", "Select View:",
                                                     choices = c(
                                                       "Code Distribution" = "distribution",
                                                       "Code Overlaps" = "overlap",
                                                       "Code Sequences" = "sequence"
                                                     )),
                                         plotOutput("comparison_plot", height = "500px")),
                                tabPanel("Detailed Analysis",
                                         h4("Coverage Analysis"),
                                         verbatimTextOutput("coverage_details"),
                                         h4("Code Application Patterns"),
                                         verbatimTextOutput("application_details"),
                                         h4("Pattern Analysis"),
                                         verbatimTextOutput("pattern_details"))
                              )
                       )
                     ))
          ),
          uiOutput("text_display")
      )
    )
  )

  server <- function(input, output, session) {
    # Initialize reactive values
    # Note: Added several new storage-related values
    rv <- reactiveValues(
      data_dir = NULL,
      storage_mode = NULL,
      project_specific_dir = NULL,
      current_project_path = NULL,
      pending_load_path = NULL,
      text = "",
      annotations = data.frame(
        start = integer(),
        end = integer(),
        text = character(),
        code = character(),
        memo = character(),
        source_file = character(),  # NEW: Track source file
        stringsAsFactors = FALSE
      ),
      codes = character(),
      history = list(list(text = "", annotations = data.frame())),
      history_index = 1,
      code_tree = Node$new("Root"),
      code_colors = character(),
      memos = list(),
      code_descriptions = list(),
      project_modified = FALSE,
      current_project = NULL,
      current_file_name = NULL,  # NEW: Track current file name
      action_history = list(),
      action_index = 0,
      merged_codes = list(),
      selected_theme = NULL,
      comparison_data = NULL,
      comparison_results = NULL
    )

    # Initialize roots for shinyFiles
    roots <- c(Home = path.expand("~"))
    if (.Platform$OS.type == "windows") {
      roots <- c(roots, getVolumes()())
    }

    # Initialize directory/file choosing
    shinyDirChoose(input, "directory_select", roots = roots, session = session)
    shinyDirChoose(input, "custom_dir_select", roots = roots, session = session)
    shinyDirChoose(input, "text_directory_select", roots = roots, session = session)
    shinyDirChoose(input, "records_directory_select", roots = roots, session = session)
    shinyFileChoose(input, "file_select", roots = roots, session = session)

    # Initialize directory with user choice
    observe({
      if (is.null(rv$storage_mode)) {
        init_data_dir(session)
      }
    })

    # Initialize selection button state
    observe({
      # Initialize button as inactive on startup
          runjs('
        $(document).ready(function() {
          if ($("#selection-indicator").length === 0) {
            $("body").append("<div id=\\"selection-indicator\\" class=\\"selection-mode-indicator\\">Selection Mode is Inactive</div>");
          }
          $("#select").addClass("btn-toggle-inactive");
        });
      ')
    })

    # Handle storage confirmation - fixed by passing output parameter
    handle_storage_confirmation(input, output, rv, session, roots)

    # Handle save project confirmation - fixed by passing output parameter
    handle_save_project_confirmation(input, output, session, rv, roots)

    # Handle load project confirmation - fixed by passing output parameter
    handle_load_project_confirmation(input, output, session, rv, roots)

    # Load Project handler
    observeEvent(input$load_project, {
      load_project_interactive(rv, input, session, roots)
    })

    output$current_file_name <- renderText({
      rv$current_file_name
    })

    # Initialize code tree properties in an observe block
    observe({
      rv$code_tree$type <- "theme"
      rv$code_tree$description <- "Root of the code hierarchy"
    })

    # Add Theme button handler
    observeEvent(input$add_theme_btn, {
      theme_choices <- get_theme_paths(rv$code_tree)

      showModal(modalDialog(
        title = "Add New Theme",
        textInput("new_theme_name", "Theme Name:"),
        textAreaInput("theme_description", "Description:"),
        selectInput("parent_theme", "Parent Theme:",
                    choices = theme_choices,
                    selected = "Root"),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_add_theme", "Add Theme")
        )
      ))
    })

    # Confirm Add Theme handler
    observeEvent(input$confirm_add_theme, {
      req(input$new_theme_name)

      tryCatch({
        if (input$parent_theme == "Root") {
          # Add theme directly to root
          new_theme <- rv$code_tree$AddChild(input$new_theme_name)
          new_theme$type <- "theme"
          new_theme$description <- input$theme_description
          new_theme$created <- Sys.time()
        } else {
          # Add theme to specified parent
          theme_path <- unlist(strsplit(input$parent_theme, " / "))
          current_node <- rv$code_tree

          # Navigate to parent theme
          for (theme in theme_path) {
            current_node <- current_node$children[[theme]]
            if (is.null(current_node)) {
              stop(paste("Parent theme not found:", theme))
            }
          }

          # Add new theme
          new_theme <- current_node$AddChild(input$new_theme_name)
          new_theme$type <- "theme"
          new_theme$description <- input$theme_description
          new_theme$created <- Sys.time()
        }

        showNotification("Theme added successfully", type = "message")
        removeModal()
      }, error = function(e) {
        showNotification(paste("Error adding theme:", e$message), type = "error")
      })
    })

    # Add Code to Theme button handler
    observeEvent(input$add_code_to_theme_btn, {
      theme_choices <- get_theme_paths(rv$code_tree)

      showModal(modalDialog(
        title = "Add Code to Theme",
        selectInput("code_to_add", "Select Code:",
                    choices = setdiff(rv$codes, get_all_themed_codes(rv$code_tree))),
        selectInput("theme_for_code", "Select Theme:",
                    choices = theme_choices),
        textAreaInput("code_description", "Code Description:"),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_add_code", "Add Code")
        )
      ))
    })

    # Confirm Add Code handler
    observeEvent(input$confirm_add_code, {
      req(input$code_to_add, input$theme_for_code)

      tryCatch({
        if(input$theme_for_code == "Root") {
          # Add directly to root
          rv$code_tree <- add_code_to_theme(rv$code_tree,
                                            input$code_to_add,
                                            character(0),  # empty path for root
                                            input$code_description)
        } else {
          # Add to specified theme
          theme_path <- unlist(strsplit(input$theme_for_code, " / "))
          rv$code_tree <- add_code_to_theme(rv$code_tree,
                                            input$code_to_add,
                                            theme_path,
                                            input$code_description)
        }
        showNotification("Code added to theme successfully", type = "message")
        removeModal()
      }, error = function(e) {
        showNotification(paste("Error adding code:", e$message), type = "error")
      })
    })

    # Move Item button handler
    observeEvent(input$move_item_btn, {
      # Get all items that can be moved (both themes and codes)
      movable_items <- get_all_paths(rv$code_tree)

      # Get possible parent themes
      parent_themes <- get_theme_paths(rv$code_tree)

      showModal(modalDialog(
        title = "Move Item",
        selectInput("item_to_move", "Select Item to Move:",
                    choices = movable_items),
        selectInput("new_parent", "Select New Parent:",
                    choices = parent_themes),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_move", "Move Item")
        )
      ))
    })

    # Confirm Move handler
    observeEvent(input$confirm_move, {
      req(input$item_to_move, input$new_parent)

      tryCatch({
        # Split paths into components
        item_path <- unlist(strsplit(input$item_to_move, " / "))
        new_parent_path <- if(input$new_parent == "Root") {
          character(0)
        } else {
          unlist(strsplit(input$new_parent, " / "))
        }

        # Get the item to move
        current_node <- rv$code_tree
        parent_node <- NULL
        target_node <- NULL

        # Find the item to move
        for (path_component in item_path) {
          parent_node <- current_node
          current_node <- current_node$children[[path_component]]
          if (is.null(current_node)) {
            stop(paste("Cannot find item:", input$item_to_move))
          }
        }
        target_node <- current_node

        # Find the new parent
        new_parent_node <- rv$code_tree
        if (length(new_parent_path) > 0) {
          for (path_component in new_parent_path) {
            new_parent_node <- new_parent_node$children[[path_component]]
            if (is.null(new_parent_node)) {
              stop(paste("Cannot find new parent:", input$new_parent))
            }
          }
        }

        # Check for circular reference
        if (is_ancestor(target_node, new_parent_node)) {
          stop("Cannot move a node to its own descendant")
        }

        # Check if new parent is a theme
        if (!is.null(new_parent_node$type) && new_parent_node$type != "theme") {
          stop("Can only move items to theme nodes")
        }

        # Store item data
        item_data <- list(
          name = target_node$name,
          type = target_node$type,
          description = target_node$description,
          created = target_node$created,
          children = target_node$children
        )

        # Remove from old location
        parent_node$RemoveChild(target_node$name)

        # Add to new location
        new_node <- new_parent_node$AddChild(item_data$name)
        new_node$type <- item_data$type
        new_node$description <- item_data$description
        new_node$created <- item_data$created

        # Restore children if any
        if (length(item_data$children) > 0) {
          for (child in item_data$children) {
            restore_node(new_node, child)
          }
        }

        showNotification("Item moved successfully", type = "message")
        removeModal()
      }, error = function(e) {
        showNotification(paste("Error moving item:", e$message), type = "error")
      })
    })

    # Export hierarchy handler
    observeEvent(input$export_hierarchy_btn, {
      showModal(modalDialog(
        title = "Export Hierarchy",
        downloadButton("download_hierarchy", "Download JSON"),
        footer = modalButton("Close")
      ))
    })

    # Download handler for hierarchy export
    output$download_hierarchy <- downloadHandler(
      filename = function() {
        paste0("code_hierarchy_", format(Sys.time(), "%Y%m%d"), ".json")
      },
      content = function(file) {
        writeLines(export_hierarchy(rv$code_tree), file)
      }
    )

    # Import hierarchy handler
    observeEvent(input$import_hierarchy_btn, {
      showModal(modalDialog(
        title = "Import Hierarchy",
        fileInput("hierarchy_file", "Choose JSON file",
                  accept = c("application/json", ".json")),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_import", "Import")
        )
      ))
    })

    # Confirm Import handler
    observeEvent(input$confirm_import, {
      req(input$hierarchy_file)

      tryCatch({
        json_content <- readLines(input$hierarchy_file$datapath)
        imported_tree <- import_hierarchy(paste(json_content, collapse = "\n"))

        # Validate the imported tree
        if (is.null(imported_tree) || !inherits(imported_tree, "Node")) {
          stop("Invalid hierarchy structure in imported file")
        }

        # Ensure root node has required properties
        if (is.null(imported_tree$type)) {
          imported_tree$type <- "theme"
        }
        if (is.null(imported_tree$description)) {
          imported_tree$description <- "Root of the code hierarchy"
        }

        # Process all nodes to ensure they have required properties
        process_node <- function(node) {
          if (is.null(node$type)) {
            node$type <- "theme"
          }
          if (is.null(node$description)) {
            node$description <- ""
          }
          if (is.null(node$created)) {
            node$created <- Sys.time()
          }

          if (!is.null(node$children)) {
            lapply(node$children, process_node)
          }
          return(node)
        }

        rv$code_tree <- process_node(imported_tree)
        showNotification("Hierarchy imported successfully", type = "message")
        removeModal()

      }, error = function(e) {
        showNotification(paste("Error importing hierarchy:", e$message), type = "error")
      })
    })

    # Hierarchy view output
    output$hierarchy_view <- renderUI({
      # Force reactivity
      input$confirm_add_theme
      input$confirm_add_code
      input$confirm_move
      input$confirm_import

      # Also react to code changes
      all_codes <- rv$codes

      # Generate the hierarchy HTML with all codes information
      hierarchy_html <- visualize_hierarchy(rv$code_tree, all_codes)

      # Wrap the HTML in a div with an ID for easy JavaScript targeting
      html_with_wrapper <- paste0(
        '<div id="hierarchy_container">\n',
        '<pre id="hierarchy_pre">\n',
        hierarchy_html,
        '\n</pre>\n',
        '</div>'
      )

      # Include JavaScript to initialize handlers after the UI renders
      tagList(
        HTML(html_with_wrapper),
        tags$script(HTML("
      $(document).ready(function() {
        // Make sure hierarchy items are clickable
        $('.theme-item, .code-item').on('click', function(e) {
          e.preventDefault();
          e.stopPropagation();

          // Clear existing selections
          $('.theme-item, .code-item').removeClass('selected');
          $(this).addClass('selected');

          // Get item name and send to Shiny
          var itemName = $(this).data('name');
          if (!itemName) {
            itemName = $(this).text().trim();
            if (itemName.indexOf(' - ') > -1) {
              itemName = itemName.split(' - ')[0].trim();
            }
          }

          // This is the critical line to update Shiny
          Shiny.setInputValue('selected_theme', itemName);
          console.log('Selected: ' + itemName);
        });
      });
    "))
      )
    })

    # Hierarchy statistics output
    output$hierarchy_stats <- renderText({
      # Force reactivity
      input$confirm_add_theme
      input$confirm_add_code
      input$confirm_move
      input$confirm_import

      # Also react to code changes in rv$codes
      all_codes <- rv$codes

      # Calculate stats with all codes information
      stats <- calculate_hierarchy_stats(rv$code_tree, all_codes)

      # Format the codes per theme section
      codes_per_theme_text <- if (length(stats$codes_per_theme) > 0) {
        paste("\nCodes per Theme:",
              paste(names(stats$codes_per_theme), ": ",
                    unlist(stats$codes_per_theme),
                    collapse = "\n"),
              sep = "\n")
      } else {
        "\nNo user-created themes with codes yet"
      }

      # Add information about unassigned codes
      unassigned_codes_text <- if (stats$unassigned_codes > 0) {
        paste0("\nCodes not assigned to any theme: ", stats$unassigned_codes,
               " (", paste(head(setdiff(rv$codes, get_all_themed_codes(rv$code_tree)), 5), collapse=", "),
               if(stats$unassigned_codes > 5) "..." else "", ")")
      } else {
        ""
      }

      # Combine all statistics
      paste0(
        "Summary:\n",
        "- Root node is the hierarchy container (not counted as a theme)\n",
        "- User-created Themes: ", stats$total_themes, "\n",
        "- Total Codes: ", length(rv$codes), "\n",
        "- Codes in Hierarchy: ", stats$total_codes, "\n",
        "- Hierarchy Depth: ", stats$max_depth, "\n",
        "- Average Codes per Theme: ",
        sprintf("%.1f", stats$average_codes_per_theme),
        unassigned_codes_text,
        codes_per_theme_text
      )
    })

    # Theme details output
    output$theme_details <- renderUI({
      # React to theme selection
      theme_name <- input$selected_theme

      if (is.null(theme_name)) {
        return(tags$div(
          tags$p("No item selected. Click on a theme or code in the hierarchy to view details."),
          tags$p("In the hierarchy:"),
          tags$ul(
            tags$li(HTML("<strong>&#128194; Folders</strong> represent themes - organizational categories")),
            tags$li(HTML("<strong>&#128196; Files</strong> represent codes - used to annotate text"))
          )
        ))
      }

      # Special handling for "Unassigned Codes" section
      if (theme_name == "Unassigned Codes") {
        # Get unassigned codes
        all_codes <- rv$codes
        themed_codes <- get_all_themed_codes(rv$code_tree)
        unassigned_codes <- setdiff(all_codes, themed_codes)

        return(tags$div(
          tags$h4(HTML(paste0("&#128193; Unassigned Codes Section"))),
          tags$p("These codes have been created but are not organized into any theme."),
          tags$p(HTML(paste0("<strong>Number of unassigned codes:</strong> ", length(unassigned_codes)))),

          if (length(unassigned_codes) > 0) {
            tags$div(
              tags$h5("Unassigned Codes:"),
              tags$ul(
                lapply(sort(unassigned_codes), function(code) {
                  tags$li(HTML(paste0("&#128196; ", code)))
                })
              )
            )
          }
        ))
      }

      # Check if this is a code not in the hierarchy
      selected_node <- find_node_by_name(rv$code_tree, theme_name)

      # If not found in the hierarchy, check if it's one of the unassigned codes
      if (is.null(selected_node) && theme_name %in% rv$codes) {
        # Count annotations using this code
        n_annotations <- sum(rv$annotations$code == theme_name, na.rm = TRUE)

        # For unassigned codes
        return(tags$div(
          tags$h4(HTML(paste0("&#128196; Unassigned Code: ", tags$strong(theme_name)))),
          tags$p(HTML("<strong>Status:</strong> Not organized in any theme")),
          tags$p(HTML(paste0("<strong>Used in:</strong> ", n_annotations, " annotations"))),
          tags$div(
            tags$p("To organize this code, use the 'Add Code to Theme' button.")
          )
        ))
      }

      if (is.null(selected_node)) {
        return(tags$p(paste("Item", theme_name, "not found in hierarchy.")))
      }

      # Determine if this is a theme or a code
      node_type <- if (!is.null(selected_node$type)) selected_node$type else "unknown"

      if (node_type == "theme") {
        # For themes, show theme-specific information

        # Count direct codes and sub-themes with proper error handling
        n_codes <- sum(vapply(selected_node$children, function(x) {
          tryCatch({
            if (is.null(x$type)) return(FALSE)
            x$type == "code"
          }, error = function(e) FALSE)
        }, logical(1)))

        n_subthemes <- sum(vapply(selected_node$children, function(x) {
          tryCatch({
            if (is.null(x$type)) return(FALSE)
            x$type == "theme"
          }, error = function(e) FALSE)
        }, logical(1)))

        # Format the created timestamp safely
        created_time <- tryCatch({
          if (!is.null(selected_node$created)) {
            format(selected_node$created, "%Y-%m-%d %H:%M:%S")
          } else {
            format(Sys.time(), "%Y-%m-%d %H:%M:%S")
          }
        }, error = function(e) format(Sys.time(), "%Y-%m-%d %H:%M:%S"))

        # Safe description handling
        description <- tryCatch({
          if (!is.null(selected_node$description)) {
            selected_node$description
          } else {
            "No description"
          }
        }, error = function(e) "No description")

        # Get the path
        path <- if (selected_node$name == "Root") {
          "Root (top level)"
        } else {
          tryCatch({
            paste(selected_node$path, collapse = " / ")
          }, error = function(e) selected_node$name)
        }

        # For themes, show theme information
        return(tags$div(
          tags$h4(HTML(paste0("&#128194; Theme: ", tags$strong(selected_node$name)))),
          tags$p(HTML(paste0("<strong>Path:</strong> ", path))),
          tags$p(HTML(paste0("<strong>Description:</strong> ", description))),
          tags$p(HTML(paste0("<strong>Created:</strong> ", created_time))),
          tags$p(HTML(paste0("<strong>Contains:</strong> ", n_codes, " codes, ", n_subthemes, " sub-themes"))),

          # Add code list if there are codes
          if (n_codes > 0) {
            codes <- tryCatch({
              vapply(selected_node$children[vapply(selected_node$children, function(x) {
                if (is.null(x$type)) return(FALSE)
                x$type == "code"
              }, logical(1))], function(x) x$name, character(1))
            }, error = function(e) character(0))

            if (length(codes) > 0) {
              tags$div(
                tags$h5("Codes in this theme:"),
                tags$ul(
                  lapply(codes, function(code) {
                    tags$li(HTML(paste0("&#128196; ", code)))
                  })
                )
              )
            }
          },

          # Add sub-themes list if there are sub-themes
          if (n_subthemes > 0) {
            subthemes <- tryCatch({
              vapply(selected_node$children[vapply(selected_node$children, function(x) {
                if (is.null(x$type)) return(FALSE)
                x$type == "theme"
              }, logical(1))], function(x) x$name, character(1))
            }, error = function(e) character(0))

            if (length(subthemes) > 0) {
              tags$div(
                tags$h5("Sub-themes:"),
                tags$ul(
                  lapply(subthemes, function(theme) {
                    tags$li(HTML(paste0("&#128194; ", theme)))
                  })
                )
              )
            }
          }
        ))
      } else if (node_type == "code") {
        # For codes, show code-specific information

        # Format the created timestamp safely
        created_time <- tryCatch({
          if (!is.null(selected_node$created)) {
            format(selected_node$created, "%Y-%m-%d %H:%M:%S")
          } else {
            format(Sys.time(), "%Y-%m-%d %H:%M:%S")
          }
        }, error = function(e) format(Sys.time(), "%Y-%m-%d %H:%M:%S"))

        # Safe description handling
        description <- tryCatch({
          if (!is.null(selected_node$description)) {
            selected_node$description
          } else {
            "No description"
          }
        }, error = function(e) "No description")

        # Get the parent theme
        parent_theme <- tryCatch({
          if (!is.null(selected_node$parent)) {
            selected_node$parent$name
          } else {
            "Unknown"
          }
        }, error = function(e) "Unknown")

        # Get the path
        path <- tryCatch({
          paste(selected_node$path, collapse = " / ")
        }, error = function(e) selected_node$name)

        # Count annotations using this code
        n_annotations <- sum(rv$annotations$code == selected_node$name, na.rm = TRUE)

        # For codes, show code information
        return(tags$div(
          tags$h4(HTML(paste0("&#128196; Code: ", tags$strong(selected_node$name)))),
          tags$p(HTML(paste0("<strong>Path:</strong> ", path))),
          tags$p(HTML(paste0("<strong>Parent Theme:</strong> ", parent_theme))),
          tags$p(HTML(paste0("<strong>Description:</strong> ", description))),
          tags$p(HTML(paste0("<strong>Created:</strong> ", created_time))),
          tags$p(HTML(paste0("<strong>Used in:</strong> ", n_annotations, " annotations")))
        ))
      } else {
        # For unknown types
        return(tags$p(paste("Selected item", theme_name, "has unknown type:", node_type)))
      }
    })

    # Codebook table output
    output$codebook_codes_table <- renderDT({
      # Force reactivity on codes and annotations changes
      rv$codes
      rv$annotations
      rv$code_colors

      if (length(rv$codes) > 0) {
        code_usage <- sapply(rv$codes, function(code) {
          sum(rv$annotations$code == code, na.rm = TRUE)
        })

        colors <- rv$code_colors[rv$codes]
        colors[is.na(colors)] <- "#CCCCCC"  # Default color for codes without colors

        data.frame(
          Code = rv$codes,
          Usage = code_usage,
          Color = colors,
          stringsAsFactors = FALSE
        )
      } else {
        data.frame(Code = character(0), Usage = integer(0), Color = character(0))
      }
    }, options = list(
      pageLength = 10,
      scrollX = TRUE,
      dom = 'Bfrtip',
      buttons = c('copy', 'csv'),
      columnDefs = list(
        list(targets = 2, visible = FALSE)  # Hide the Color column but keep it in data
      )
    ), selection = list(mode = 'multiple', target = 'row'))

    output$codebook_usage_stats <- renderDT({
      # Force reactivity
      rv$annotations
      rv$codes

      if (length(rv$codes) > 0 && nrow(rv$annotations) > 0) {
        # Calculate statistics for each code
        code_stats_list <- lapply(rv$codes, function(code) {
          count <- sum(rv$annotations$code == code, na.rm = TRUE)
          percentage <- if (nrow(rv$annotations) > 0) {
            round(count / nrow(rv$annotations) * 100, 1)
          } else {
            0
          }
          data.frame(
            Code = code,
            Count = count,
            Percentage = percentage,
            stringsAsFactors = FALSE
          )
        })

        # Combine all statistics
        stats_df <- do.call(rbind, code_stats_list)

        # Sort by count descending
        stats_df <- stats_df[order(stats_df$Count, decreasing = TRUE), ]

        return(stats_df)
      } else {
        data.frame(
          Code = character(0),
          Count = integer(0),
          Percentage = numeric(0),
          stringsAsFactors = FALSE
        )
      }
    }, options = list(pageLength = 10, searching = FALSE))

    output$codebook_selected_details <- renderUI({
      # Force reactivity
      rv$codes
      rv$code_tree
      rv$code_colors

      selected_rows <- input$codebook_codes_table_rows_selected

      # Handle no selection
      if (is.null(selected_rows) || length(selected_rows) == 0) {
        return(tags$pre(
          "Click on any row in the 'Existing Codes' table to see detailed information about that code."
        ))
      }

      # Handle single selection
      if (length(selected_rows) == 1 && selected_rows[1] <= length(rv$codes)) {
        selected_code <- rv$codes[selected_rows[1]]

        # Get code info from hierarchy if available
        code_node <- find_node_by_name(rv$code_tree, selected_code)

        description <- if (!is.null(code_node) && !is.null(code_node$description) && code_node$description != "") {
          code_node$description
        } else {
          "No description available"
        }

        created <- if (!is.null(code_node) && !is.null(code_node$created)) {
          format(code_node$created, "%Y-%m-%d %H:%M:%S")
        } else {
          "Unknown"
        }

        # Get usage count
        usage_count <- sum(rv$annotations$code == selected_code, na.rm = TRUE)

        # Get color
        code_color <- rv$code_colors[selected_code]
        if (is.null(code_color) || is.na(code_color)) {
          code_color <- "Not assigned"
          color_display <- "Not assigned"
        } else {
          color_display <- tags$span(
            tags$span(style = paste0("display: inline-block; width: 20px; height: 20px; background-color: ", code_color, "; border: 1px solid #ccc; vertical-align: middle; margin-right: 5px;")),
            code_color
          )
        }

        return(tags$div(
          tags$p(tags$strong("Selected Code: "), selected_code),
          tags$p(tags$strong("Description: "), description),
          tags$p(tags$strong("Usage Count: "), usage_count, " annotation(s)"),
          tags$p(tags$strong("Color: "), color_display),
          tags$p(tags$strong("Created: "), created)
        ))
      }

      # Handle multiple selections
      if (length(selected_rows) > 1) {
        selected_codes <- rv$codes[selected_rows[selected_rows <= length(rv$codes)]]

        if (length(selected_codes) == 0) {
          return(tags$pre("Invalid selection. Please select valid codes from the table."))
        }

        # Calculate statistics for multiple codes
        total_usage <- sum(sapply(selected_codes, function(code) {
          sum(rv$annotations$code == code, na.rm = TRUE)
        }))

        # Get colors for selected codes
        code_colors <- sapply(selected_codes, function(code) {
          color <- rv$code_colors[code]
          if (is.null(color) || is.na(color)) "Not assigned" else color
        })

        # Build summary for multiple codes
        return(tags$div(
          tags$p(tags$strong("Multiple Codes Selected: "), length(selected_codes), " codes"),
          tags$p(tags$strong("Selected Codes:")),
          tags$ul(
            lapply(seq_along(selected_codes), function(i) {
              code <- selected_codes[i]
              usage <- sum(rv$annotations$code == code, na.rm = TRUE)
              color <- code_colors[i]

              color_display <- if (color != "Not assigned") {
                tagList(
                  tags$span(style = paste0("display: inline-block; width: 15px; height: 15px; background-color: ", color, "; border: 1px solid #ccc; vertical-align: middle; margin-right: 5px;")),
                  color
                )
              } else {
                "Not assigned"
              }

              tags$li(
                tags$strong(code), " (", usage, " annotations, color: ", color_display, ")"
              )
            })
          ),
          tags$p(tags$strong("Total Combined Usage: "), total_usage, " annotation(s)"),
          tags$p(tags$strong("Actions Available:")),
          tags$ul(
            tags$li("Rename Selected: Rename one of the selected codes"),
            tags$li("Delete Selected: Delete all selected codes"),
            tags$li("Merge Selected: Combine all codes into a single new code")
          )
        ))
      }
    })

    # Selection feedback output
    output$selection_feedback <- renderUI({
      selected_rows <- input$codebook_codes_table_rows_selected

      if (is.null(selected_rows) || length(selected_rows) == 0) {
        tags$div(
          class = "alert alert-info",
          style = "padding: 8px; margin: 5px 0;",
          icon("info-circle"),
          " No codes selected. Click on table rows to select codes."
        )
      } else if (length(selected_rows) == 1) {
        selected_code <- rv$codes[selected_rows[1]]
        tags$div(
          class = "alert alert-primary",
          style = "padding: 8px; margin: 5px 0;",
          icon("check-circle"),
          paste(" 1 code selected:", selected_code, ". Select more for merging.")
        )
      } else {
        selected_codes <- rv$codes[selected_rows]
        tags$div(
          class = "alert alert-success",
          style = "padding: 8px; margin: 5px 0;",
          icon("check-circle"),
          paste(" ", length(selected_rows), "codes selected:", paste(selected_codes, collapse = ", "), ". Ready for merging!")
        )
      }
    })

    # Add click handler for theme selection
    observeEvent(input$select_theme, {
      rv$selected_theme <- input$select_theme
    })

    # Helper function to get all theme paths
    get_theme_paths <- function(node) {
      paths <- c("Root")  # Start with Root as the first option

      collect_paths <- function(node, current_path = character()) {
        # Check if this node is a theme
        if (!is.null(node$type) && node$type == "theme" && !is.null(node$name)) {
          if (length(current_path) > 0) {
            full_path <- paste(c(current_path, node$name), collapse = " / ")
            paths <<- c(paths, full_path)
          } else if (node$name != "Root") {
            # Add single themes at root level
            paths <<- c(paths, node$name)
          }
        }

        # Recursively process children
        if (!is.null(node$children)) {
          new_path <- if (length(current_path) > 0) {
            c(current_path, node$name)
          } else if (node$name != "Root") {
            node$name
          } else {
            character(0)
          }

          for (child in node$children) {
            collect_paths(child, new_path)
          }
        }
      }

      collect_paths(node)
      return(unique(paths))
    }

    # Helper function to get all themed codes
    get_all_themed_codes <- function(node) {
      codes <- character()

      traverse_node <- function(node) {
        if (node$type == "code") {
          codes <<- c(codes, node$name)
        }
        if (length(node$children) > 0) {
          lapply(node$children, traverse_node)
        }
      }

      traverse_node(node)
      return(codes)
    }

    # Helper function to get all paths (themes and codes)
    get_all_paths <- function(node) {
      paths <- character()

      collect_paths <- function(node, current_path = character()) {
        if (!is.null(node$name) && node$name != "Root") {
          # Create the full path for this node
          full_path <- if (length(current_path) > 0) {
            paste(c(current_path, node$name), collapse = " / ")
          } else {
            node$name
          }
          # Add the path only if it's a code or theme
          if (!is.null(node$type) && (node$type == "theme" || node$type == "code")) {
            paths <<- c(paths, full_path)
          }
        }

        # Process children if they exist
        if (!is.null(node$children) && length(node$children) > 0) {
          for (child in node$children) {
            new_path <- if (length(current_path) > 0 && node$name != "Root") {
              c(current_path, node$name)
            } else if (node$name != "Root") {
              node$name
            } else {
              character(0)
            }
            collect_paths(child, new_path)
          }
        }
      }

      collect_paths(node)
      return(unique(paths))
    }

    # Initialize roots for shinyFiles
    roots <- c(Home = path.expand("~"))
    if (.Platform$OS.type == "windows") {
      roots <- c(roots, getVolumes()())
    }

    # Initialize directory/file choosing
    shinyDirChoose(input, "directory_select", roots = roots, session = session)
    shinyDirChoose(input, "text_directory_select", roots = roots, session = session)
    shinyDirChoose(input, "records_directory_select", roots = roots, session = session)
    shinyFileChoose(input, "file_select", roots = roots, session = session)

    # Save Project handler
    observeEvent(input$save_project, {
      # Check if we already have a current project with a known save location
      if (!is.null(rv$current_project) && !is.null(rv$current_project_path)) {
        # Quick save to existing location
        tryCatch({
          # Create project state
          project_state <- list(
            text = rv$text,
            annotations = rv$annotations,
            codes = rv$codes,
            code_tree = rv$code_tree,
            code_colors = rv$code_colors,
            memos = rv$memos,
            code_descriptions = rv$code_descriptions,
            history = rv$history,
            history_index = rv$history_index
          )

          # Add project directory info if in project-specific mode
          if (!is.null(rv$storage_mode) && rv$storage_mode == "project" && !is.null(rv$project_specific_dir)) {
            project_state$project_dir <- rv$project_specific_dir
          }

          # Save to existing location
          saveRDS(project_state, file = rv$current_project_path)
          rv$project_modified <- FALSE
          showNotification(paste("Project saved to", rv$current_project_path), type = "message")

        }, error = function(e) {
          showNotification(paste("Error saving project:", e$message), type = "error")
          # If quick save fails, fall back to save dialog
          save_project_interactive(rv, input, session, roots)
        })
      } else {
        # No existing save location, show save dialog
        save_project_interactive(rv, input, session, roots)
      }
    })

    observeEvent(input$remove_annotation, {
      rv <- remove_annotation(rv,
                              input$remove_annotation$start,
                              input$remove_annotation$end,
                              input$remove_annotation$code)

      # Update UI
      output$text_display <- renderUI({
        HTML(update_text_display())
      })

      showNotification("Annotation removed", type = "message")
    })

    observeEvent(input$save_as_project, {
      req(input$project_name)

      # Determine save location based on storage mode
      if (!is.null(rv$storage_mode) && rv$storage_mode == "custom") {
        project_dir <- get_project_dir(rv)
      } else {
        # Default location
        project_dir <- get_project_dir()
      }

      # Construct full filepath
      filename <- if (!grepl("\\.rds$", input$project_name)) {
        paste0(input$project_name, ".rds")
      } else {
        input$project_name
      }
      filepath <- file.path(project_dir, filename)

      # Create project state
      project_state <- list(
        text = rv$text,
        annotations = rv$annotations,
        codes = rv$codes,
        code_tree = rv$code_tree,
        code_colors = rv$code_colors,
        memos = rv$memos,
        code_descriptions = rv$code_descriptions,
        history = rv$history,
        history_index = rv$history_index
      )

      # Save project
      tryCatch({
        saveRDS(project_state, file = filepath)
        rv$current_project <- input$project_name
        rv$current_project_path <- filepath
        rv$project_modified <- FALSE
        showNotification(paste("Project saved as new copy to", filepath), type = "message")
      }, error = function(e) {
        showNotification(paste("Error saving project:", e$message), type = "error")
      })

      removeModal()
    })

    # Display selected save directory
    output$selected_dir <- renderText({
      if (!is.null(input$directory_select)) {
        parseDirPath(roots, input$directory_select)
      }
    })

    # Display selected text save directory
    output$selected_text_dir <- renderText({
      if (!is.null(input$text_directory_select)) {
        parseDirPath(roots, input$text_directory_select)
      }
    })

    # Display selected records directory
    output$selected_records_dir <- renderText({
      if (!is.null(input$records_directory_select)) {
        parseDirPath(roots, input$records_directory_select)
      }
    })

    # Project status display
    output$project_status_display <- renderUI({
      if (is.null(rv$current_project) || rv$current_project == "") {
        # No project open
        tags$span(
          style = "color: #999; font-style: italic;",
          icon("circle", style = "color: #ccc; margin-right: 5px;"),
          "No project open"
        )
      } else {
        # Project is open
        status_color <- if (rv$project_modified) "#f39c12" else "#27ae60"  # Orange if modified, green if saved
        status_icon <- if (rv$project_modified) "circle" else "check-circle"
        status_text <- if (rv$project_modified) "unsaved changes" else "saved"

        tags$span(
          style = paste0("color: ", status_color, "; font-weight: 500;"),
          icon(status_icon, style = paste0("color: ", status_color, "; margin-right: 5px;")),
          paste("Project:", rv$current_project),
          tags$br(),
          tags$small(
            style = paste0("color: ", status_color, "; font-style: italic;"),
            status_text
          )
        )
      }
    })

    # This ensures that when input$selected_theme changes, it gets stored in rv$selected_theme
    observeEvent(input$selected_theme, {
      if (!is.null(input$selected_theme) && input$selected_theme != "") {
        rv$selected_theme <- input$selected_theme
        # Debug message to console
        message(paste("Theme selected:", input$selected_theme))
      }
    })

    # Merge codes functionality
    observeEvent(input$merge_codes, {
      showModal(modalDialog(
        title = "Merge Codes",
        checkboxGroupInput("codes_to_merge", "Select codes to merge:",
                           choices = rv$codes,
                           selected = NULL
        ),
        textInput("new_code_name", "New code name:"),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_merge_codes", "Merge")
        )
      ))
    })

    # Add this observer for handling code merging
    observeEvent(input$confirm_merge_codes, {
      req(input$codes_to_merge, input$new_code_name)

      # Create merge action
      merge_action <- create_action(
        type = "merge_codes",
        data = list(
          old_codes = input$codes_to_merge,
          new_code = input$new_code_name
        )
      )

      # Store old colors before merge
      old_colors <- list()
      for (code in input$codes_to_merge) {
        old_colors[[code]] <- rv$code_colors[code]
      }
      merge_action$data$old_colors <- old_colors

      # Apply and record the action
      apply_action(rv, merge_action)
      add_action(rv, merge_action)

      # Update codes list
      rv$codes <- unique(c(setdiff(rv$codes, input$codes_to_merge), input$new_code_name))

      # Assign color to the new merged code
      used_colors <- as.character(rv$code_colors)
      new_color <- get_next_palette_color(used_colors)

      # If color already exists, generate a new readable one
      if (new_color %in% used_colors) {
        new_color <- generate_readable_color()

        attempts <- 0
        while (!is_color_readable(new_color) && attempts < 10) {
          new_color <- generate_readable_color()
          attempts <- attempts + 1
        }

        if (!is_color_readable(new_color)) {
          safe_colors <- c("#FFE6CC", "#E6F3FF", "#E6FFE6", "#FFE6F3", "#F3E6FF")
          new_color <- sample(safe_colors, 1)
        }
      }

      rv$code_colors[input$new_code_name] <- new_color

      # Remove old code colors
      rv$code_colors <- rv$code_colors[!names(rv$code_colors) %in% input$codes_to_merge]

      removeModal()
      showNotification(paste("Codes merged into", input$new_code_name), type = "message")

      # Update UI
      output$text_display <- renderUI({
        HTML(update_text_display())
      })
    })

    observe({
      if (length(rv$code_colors) > 0) {
        rv <- update_dark_colors(rv)
      }
    }, priority = 999)  # High priority to run early

    # Codebook merge codes handler
    observeEvent(input$codebook_merge_codes, {
      selected_rows <- input$codebook_codes_table_rows_selected

      if (is.null(selected_rows) || length(selected_rows) == 0) {
        showNotification("Please select codes from the table first. Click on table rows to select them.",
                         type = "warning", duration = 5)
        return()
      }

      if (length(selected_rows) < 2) {
        showNotification("Please select at least 2 codes to merge. Hold Ctrl/Cmd and click to select multiple rows.",
                         type = "warning", duration = 5)
        return()
      }

      # Get the selected codes from the table
      selected_codes <- rv$codes[selected_rows]

      showModal(modalDialog(
        title = "Merge Selected Codes",
        size = "m",
        tags$div(
          tags$h5("Codes to be merged:"),
          tags$div(
            style = "background-color: #f8f9fa; padding: 10px; border-radius: 5px; margin: 10px 0;",
            lapply(selected_codes, function(code) {
              usage_count <- sum(rv$annotations$code == code, na.rm = TRUE)
              tags$div(
                style = "margin: 5px 0;",
                tags$strong(code),
                tags$span(class = "text-muted", paste0(" (", usage_count, " annotations)"))
              )
            })
          ),
          tags$p(class = "text-info",
                 icon("info-circle"),
                 " All annotations with these codes will be updated to use the new merged code name."),
          textInput("codebook_new_code_name", "New merged code name:",
                    placeholder = "Enter name for the merged code"),
          textAreaInput("codebook_merged_description", "Description for merged code:",
                        height = "80px", placeholder = "Optional description for the new code")
        ),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_codebook_merge", "Merge Codes",
                       class = "btn-primary", icon = icon("object-group"))
        )
      ))
    })

    # Handle codebook merge confirmation
    observeEvent(input$confirm_codebook_merge, {
      req(input$codebook_new_code_name)

      selected_rows <- input$codebook_codes_table_rows_selected
      if (length(selected_rows) < 2) {
        showNotification("Please select at least 2 codes to merge", type = "error")
        return()
      }

      selected_codes <- rv$codes[selected_rows]
      new_code_name <- trimws(input$codebook_new_code_name)

      if (nchar(new_code_name) == 0) {
        showNotification("Please enter a valid code name", type = "warning")
        return()
      }

      if (new_code_name %in% rv$codes && !new_code_name %in% selected_codes) {
        showNotification("A code with that name already exists", type = "error")
        return()
      }

      tryCatch({
        # Create merge action for undo/redo system
        merge_action <- create_action(
          type = "merge_codes",
          data = list(
            old_codes = selected_codes,
            new_code = new_code_name
          )
        )

        # Store old colors before merge
        old_colors <- list()
        for (code in selected_codes) {
          old_colors[[code]] <- rv$code_colors[code]
        }
        merge_action$data$old_colors <- old_colors

        # Update all annotations that use the old codes
        for (old_code in selected_codes) {
          rv$annotations$code[rv$annotations$code == old_code] <- new_code_name
        }

        # Update codes list - remove old codes and add new one
        rv$codes <- unique(c(setdiff(rv$codes, selected_codes), new_code_name))

        # Assign color to the new merged code
        used_colors <- as.character(rv$code_colors)
        new_color <- get_next_palette_color(used_colors)

        # If color already exists, generate a new readable one
        if (new_color %in% used_colors) {
          new_color <- generate_readable_color()

          attempts <- 0
          while (!is_color_readable(new_color) && attempts < 10) {
            new_color <- generate_readable_color()
            attempts <- attempts + 1
          }

          if (!is_color_readable(new_color)) {
            safe_colors <- c("#FFE6CC", "#E6F3FF", "#E6FFE6", "#FFE6F3", "#F3E6FF")
            new_color <- sample(safe_colors, 1)
          }
        }

        rv$code_colors[new_code_name] <- new_color

        # Remove old code colors
        rv$code_colors <- rv$code_colors[!names(rv$code_colors) %in% selected_codes]

        # Update hierarchy - remove old codes and add new one
        for (old_code in selected_codes) {
          old_code_node <- find_node_by_name(rv$code_tree, old_code)
          if (!is.null(old_code_node) && !is.null(old_code_node$parent)) {
            old_code_node$parent$RemoveChild(old_code)
          }
        }

        # Add new merged code to root if not already in hierarchy
        if (!code_exists_in_hierarchy(rv$code_tree, new_code_name)) {
          new_code_node <- rv$code_tree$AddChild(new_code_name)
          new_code_node$type <- "code"
          new_code_node$description <- input$codebook_merged_description %||% ""
          new_code_node$created <- Sys.time()
        }

        # Record the action for undo/redo
        add_action(rv, merge_action)

        # Mark project as modified
        rv$project_modified <- TRUE

        removeModal()
        showNotification(paste("Successfully merged", length(selected_codes), "codes into", new_code_name), type = "message")

        # Update text display to reflect the merge
        output$text_display <- renderUI({
          HTML(update_text_display())
        })

      }, error = function(e) {
        showNotification(paste("Error merging codes:", e$message), type = "error")
      })
    })

    # Update code to handle color assignment during code creation in the codebook
    # Codebook code creation
    observeEvent(input$codebook_create_code, {
      req(input$codebook_new_code)

      new_code <- trimws(input$codebook_new_code)

      if (nchar(new_code) == 0) {
        showNotification("Please enter a valid code name", type = "warning")
        return()
      }

      if (!new_code %in% rv$codes) {
        # Add to codes list
        rv$codes <- c(rv$codes, new_code)

        # Assign color
        used_colors <- as.character(rv$code_colors)
        new_color <- get_next_palette_color(used_colors)
        rv$code_colors[new_code] <- new_color

        # Add to hierarchy if not exists
        if (!code_exists_in_hierarchy(rv$code_tree, new_code)) {
          new_code_node <- rv$code_tree$AddChild(new_code)
          new_code_node$type <- "code"
          new_code_node$description <- input$codebook_code_description %||% ""
          new_code_node$created <- Sys.time()
        }

        # Clear inputs
        updateTextInput(session, "codebook_new_code", value = "")
        updateTextAreaInput(session, "codebook_code_description", value = "")

        showNotification(paste("Code '", new_code, "' created successfully"), type = "message")
      } else {
        showNotification("Code already exists", type = "warning")
      }
    })

    observeEvent(input$codebook_rename_code, {
      selected_rows <- input$codebook_codes_table_rows_selected

      if (is.null(selected_rows) || length(selected_rows) == 0) {
        showNotification("Please select a code to rename", type = "warning")
        return()
      }

      if (length(selected_rows) > 1) {
        showNotification("Please select only one code to rename. Use 'Merge Selected' to combine multiple codes.", type = "warning")
        return()
      }

      if (selected_rows[1] > length(rv$codes)) {
        showNotification("Invalid selection", type = "error")
        return()
      }

      selected_code <- rv$codes[selected_rows[1]]

      showModal(modalDialog(
        title = "Rename Code",
        textInput("new_code_name_codebook", "New code name:", value = selected_code),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_rename_codebook", "Rename")
        )
      ))
    })

    # Handle rename confirmation
    observeEvent(input$confirm_rename_codebook, {
      req(input$new_code_name_codebook)

      selected_row <- input$codebook_codes_table_rows_selected
      if (length(selected_row) > 0 && length(rv$codes) >= selected_row) {
        old_code <- rv$codes[selected_row]
        new_code <- trimws(input$new_code_name_codebook)

        if (new_code != old_code && !new_code %in% rv$codes) {
          # Update codes list
          rv$codes[selected_row] <- new_code

          # Update annotations
          rv$annotations$code[rv$annotations$code == old_code] <- new_code

          # Update colors
          rv$code_colors[new_code] <- rv$code_colors[old_code]
          rv$code_colors <- rv$code_colors[names(rv$code_colors) != old_code]

          # Update hierarchy
          code_node <- find_node_by_name(rv$code_tree, old_code)
          if (!is.null(code_node)) {
            code_node$name <- new_code
          }

          showNotification(paste("Code renamed from '", old_code, "' to '", new_code, "'"), type = "message")
        } else if (new_code %in% rv$codes) {
          showNotification("A code with that name already exists", type = "error")
        }
      }

      removeModal()
    })

    observeEvent(input$codebook_delete_code, {
      selected_rows <- input$codebook_codes_table_rows_selected

      if (is.null(selected_rows) || length(selected_rows) == 0) {
        showNotification("Please select code(s) to delete", type = "warning")
        return()
      }

      # Filter out invalid selections
      valid_rows <- selected_rows[selected_rows <= length(rv$codes)]
      if (length(valid_rows) == 0) {
        showNotification("Invalid selection", type = "error")
        return()
      }

      selected_codes <- rv$codes[valid_rows]

      # Calculate total usage count
      total_usage <- sum(sapply(selected_codes, function(code) {
        sum(rv$annotations$code == code, na.rm = TRUE)
      }))

      # Create appropriate warning message
      if (length(selected_codes) == 1) {
        warning_text <- paste("Are you sure you want to delete the code '", selected_codes[1], "'?")
        if (total_usage > 0) {
          warning_text <- paste0(warning_text, "\n\nWarning: This code is used in ", total_usage, " annotation(s). All these annotations will be removed.")
        } else {
          warning_text <- paste0(warning_text, "\n\nThis code is not currently used in any annotations.")
        }
      } else {
        warning_text <- paste0("Are you sure you want to delete ", length(selected_codes), " codes?\n\nCodes to delete: ", paste(selected_codes, collapse = ", "))
        if (total_usage > 0) {
          warning_text <- paste0(warning_text, "\n\nWarning: These codes are used in a total of ", total_usage, " annotation(s). All these annotations will be removed.")
        } else {
          warning_text <- paste0(warning_text, "\n\nThese codes are not currently used in any annotations.")
        }
      }

      showModal(modalDialog(
        title = if (length(selected_codes) == 1) "Delete Code" else "Delete Multiple Codes",
        tags$p(warning_text),
        if (total_usage > 0) {
          tags$p(tags$strong("This action cannot be undone!"), style = "color: red;")
        },
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_delete_codebook", "Delete", class = "btn-danger")
        )
      ))
    })

    # Handle delete confirmation
    observeEvent(input$confirm_delete_codebook, {
      selected_rows <- input$codebook_codes_table_rows_selected

      if (!is.null(selected_rows) && length(selected_rows) > 0) {
        # Filter out invalid selections
        valid_rows <- selected_rows[selected_rows <= length(rv$codes)]

        if (length(valid_rows) > 0) {
          codes_to_delete <- rv$codes[valid_rows]

          # Remove from annotations
          rv$annotations <- rv$annotations[!rv$annotations$code %in% codes_to_delete, ]

          # Remove from codes list
          rv$codes <- rv$codes[-valid_rows]

          # Remove colors
          rv$code_colors <- rv$code_colors[!names(rv$code_colors) %in% codes_to_delete]

          # Remove from hierarchy
          for (code_to_delete in codes_to_delete) {
            code_node <- find_node_by_name(rv$code_tree, code_to_delete)
            if (!is.null(code_node) && !is.null(code_node$parent)) {
              code_node$parent$RemoveChild(code_to_delete)
            }
          }

          # Mark project as modified
          rv$project_modified <- TRUE

          showNotification(paste(length(codes_to_delete), "code(s) deleted successfully"), type = "message")

          # Update text display
          output$text_display <- renderUI({
            HTML(update_text_display())
          })
        }
      }

      removeModal()
    })

    #update_text_display function
    update_text_display <- function() {
      if (nrow(rv$annotations) == 0) {
        return(paste0("<span class='char' id='char_", 1:nchar(rv$text), "'>", strsplit(rv$text, "")[[1]], "</span>", collapse = ""))
      }

      sorted_annotations <- rv$annotations[order(rv$annotations$start), ]
      displayed_text <- ""
      last_end <- 0

      for (i in 1:nrow(sorted_annotations)) {
        if (sorted_annotations$start[i] > last_end + 1) {
          displayed_text <- paste0(displayed_text,
                                   paste0("<span class='char' id='char_", (last_end + 1):(sorted_annotations$start[i] - 1), "'>",
                                          strsplit(substr(rv$text, last_end + 1, sorted_annotations$start[i] - 1), "")[[1]],
                                          "</span>", collapse = ""))
        }

        # Get color for this code with improved fallback
        code_color <- rv$code_colors[sorted_annotations$code[i]]
        if (is.null(code_color) || length(code_color) == 0) {
          code_color <- "#FFE6CC"  # Use a readable default color if not found
        }

        # Check if the color is readable, if not use a safe default
        if (!is_color_readable(code_color)) {
          code_color <- "#FFE6CC"
        }

        displayed_text <- paste0(displayed_text,
                                 "<span class='code-display' style='background-color: ", code_color, ";' data-code='", sorted_annotations$code[i], "' data-start='", sorted_annotations$start[i], "' data-end='", sorted_annotations$end[i], "'>",
                                 "[", sorted_annotations$code[i], "]",
                                 paste0("<span class='char' id='char_", sorted_annotations$start[i]:sorted_annotations$end[i], "'>",
                                        strsplit(substr(rv$text, sorted_annotations$start[i], sorted_annotations$end[i]), "")[[1]],
                                        "</span>", collapse = ""),
                                 "</span>")
        last_end <- sorted_annotations$end[i]
      }

      if (last_end < nchar(rv$text)) {
        displayed_text <- paste0(displayed_text,
                                 paste0("<span class='char' id='char_", (last_end + 1):nchar(rv$text), "'>",
                                        strsplit(substr(rv$text, last_end + 1, nchar(rv$text)), "")[[1]],
                                        "</span>", collapse = ""))
      }

      return(displayed_text)
    }

    observeEvent(input$replace_code, {
      showModal(modalDialog(
        title = "Replace Code",
        selectInput("new_code", "Select new code:", choices = rv$codes),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_replace_code", "Replace")
        )
      ))
    })

    observeEvent(input$confirm_replace_code, {
      removeModal()
      code_to_replace <- input$replace_code$code
      new_code <- input$new_code
      start <- input$replace_code$start
      end <- input$replace_code$end

      idx <- which(rv$annotations$start == start & rv$annotations$end == end & rv$annotations$code == code_to_replace)
      if (length(idx) > 0) {
        rv$annotations$code[idx] <- new_code
        save_state()
        output$text_display <- renderUI({
          HTML(update_text_display())
        })
      }
    })

    observeEvent(input$rename_code, {
      showModal(modalDialog(
        title = "Rename Code",
        textInput("new_code_name", "Enter new code name:"),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_rename_code", "Rename")
        )
      ))
    })

    observeEvent(input$confirm_rename_code, {
      removeModal()
      old_code <- input$rename_code$code
      new_code <- input$new_code_name

      rv$codes <- unique(c(setdiff(rv$codes, old_code), new_code))
      rv$annotations$code[rv$annotations$code == old_code] <- new_code
      rv$code_colors[new_code] <- rv$code_colors[old_code]
      rv$code_colors <- rv$code_colors[names(rv$code_colors) != old_code]
      save_state()
      output$text_display <- renderUI({
        HTML(update_text_display())
      })
    })

    observeEvent(input$delete_code, {
      showModal(modalDialog(
        title = "Delete Code",
        p("Are you sure you want to delete this code?"),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_delete_code", "Delete")
        )
      ))
    })

    observeEvent(input$confirm_delete_code, {
      removeModal()
      code_to_delete <- input$delete_code$code
      start <- input$delete_code$start
      end <- input$delete_code$end

      idx <- which(rv$annotations$start == start & rv$annotations$end == end & rv$annotations$code == code_to_delete)
      if (length(idx) > 0) {
        rv$annotations <- rv$annotations[-idx, ]
        if (!(code_to_delete %in% rv$annotations$code)) {
          rv$codes <- setdiff(rv$codes, code_to_delete)
          rv$code_colors <- rv$code_colors[names(rv$code_colors) != code_to_delete]
        }
        save_state()
        output$text_display <- renderUI({
          HTML(update_text_display())
        })
      }
    })

    # Save Annotated Text handler
    observeEvent(input$save_code, {
      save_annotated_text_interactive(rv, input, session, roots)
    })

    # Save annotated text confirmation handler
    observeEvent(input$confirm_save_annotations, {
      req(input$save_filename)
      req(input$text_directory_select)

      # Get selected directory path
      dir_path <- parseDirPath(roots, input$text_directory_select)

      filename <- input$save_filename
      if (!grepl(paste0("\\.", input$save_format, "$"), filename)) {
        filename <- paste0(filename, ".", input$save_format)
      }
      filepath <- file.path(dir_path, filename)

      tryCatch({
        dir.create(dirname(filepath), recursive = TRUE, showWarnings = FALSE)
        if (input$save_format == "html") {
          save_as_html(filepath)
        } else if (input$save_format == "txt") {
          save_as_text(filepath)
        }
        showNotification(paste("Annotated text saved to", filepath), type = "message")
      }, error = function(e) {
        showNotification(paste("Error saving annotated text:", e$message), type = "error")
      })

      removeModal()
    })

    # Function to save as HTML
    save_as_html <- function(filename) {
      # Get the current state of the text display
      html_content <- update_text_display()

      # Create a complete HTML document
      full_html <- paste0(
        "<!DOCTYPE html>\n<html>\n<head>\n",
        "<style>\n",
        ".code-display { padding: 2px 5px; margin-right: 5px; border-radius: 3px; font-weight: bold; color: black; }\n",
        "</style>\n",
        "</head>\n<body>\n",
        "<h1>Annotated Text</h1>\n",
        "<div id='annotated_text'>\n",
        html_content,
        "\n</div>\n",
        "</body>\n</html>"
      )

      # Write the HTML content to a file
      writeLines(full_html, filename)
    }

    # Function to save as text file
    save_as_text <- function(filename) {
      # Get the annotated text
      annotated_text <- create_plain_text_annotations()

      # Write the content to a file
      writeLines(annotated_text, filename)
    }

    # Helper function to create plain text with annotations
    create_plain_text_annotations <- function() {
      if (nrow(rv$annotations) == 0) {
        return(rv$text)
      }

      sorted_annotations <- rv$annotations[order(rv$annotations$start), ]
      plain_text <- ""
      last_end <- 0

      for (i in 1:nrow(sorted_annotations)) {
        if (sorted_annotations$start[i] > last_end + 1) {
          plain_text <- paste0(plain_text, substr(rv$text, last_end + 1, sorted_annotations$start[i] - 1))
        }
        plain_text <- paste0(plain_text,
                             "[", sorted_annotations$code[i], ": ",
                             substr(rv$text, sorted_annotations$start[i], sorted_annotations$end[i]),
                             "]")
        last_end <- sorted_annotations$end[i]
      }

      if (last_end < nchar(rv$text)) {
        plain_text <- paste0(plain_text, substr(rv$text, last_end + 1, nchar(rv$text)))
      }

      return(plain_text)
    }

    # Modify the text display output
    output$text_display <- renderUI({
      HTML(update_text_display())
    })

    # Create the floating text content once when the text changes
    observe({
      # Preserve line breaks in the floating text content
      text_with_linebreaks <- gsub("\n", "<br>", rv$text)

      # Split by <br> tags
      text_parts <- strsplit(text_with_linebreaks, "<br>")[[1]]
      result <- character(length(text_parts))

      # Process each line separately
      for (i in seq_along(text_parts)) {
        line_chars <- strsplit(text_parts[i], "")[[1]]
        # Calculate the character offset for this line
        char_offset <- ifelse(i == 1, 0, sum(nchar(text_parts[1:(i-1)])) + (i-1))
        char_indices <- char_offset + 1:length(line_chars)

        # Create spans for each character with correct indices
        result[i] <- paste0("<span class='char' id='float_char_", char_indices, "'>",
                            line_chars, "</span>", collapse = "")
      }

      # Join the lines with <br> tags
      rv$floating_text_content <- paste(result, collapse = "<br>")
    })

    # Update the floating text display
    output$floating_text_display <- renderUI({
      HTML(update_text_display())
    })

    # Toggle floating text window
    observeEvent(input$toggle_text_window, {
      toggle("floating_text_window")
      runjs("initializeSelection();")
    })



    # Save project confirmation handler
    observeEvent(input$confirm_save_project, {
      req(input$project_name)

      # ... existing save logic ...

      # After successful save, check if we need to import
      if (!is.null(rv$import_after_save) && rv$import_after_save) {
        rv$import_after_save <- NULL  # Clear the flag
        perform_text_import()
      }
    })

    # Display selected load file
    output$selected_file <- renderText({
      if (!is.null(input$file_select)) {
        selected <- parseFilePaths(roots, input$file_select)
        if (nrow(selected) > 0) {
          as.character(selected$datapath)
        }
      }
    })

    # Load project confirmation handler
    observeEvent(input$confirm_load_project, {
      req(input$file_select)

      # Get selected file path
      selected <- parseFilePaths(roots, input$file_select)
      if (nrow(selected) == 0) return()
      filepath <- as.character(selected$datapath[1])

      tryCatch({
        project_state <- readRDS(filepath)

        # Update all reactive values with loaded state
        rv$text <- project_state$text
        rv$annotations <- project_state$annotations

        # Handle annotations that don't have source_file column (backward compatibility)
        if (!"source_file" %in% colnames(rv$annotations) && nrow(rv$annotations) > 0) {
          rv$annotations$source_file <- "Legacy Import"
        }

        rv$codes <- project_state$codes
        rv$code_tree <- project_state$code_tree
        rv$code_colors <- project_state$code_colors
        rv$memos <- project_state$memos
        rv$code_descriptions <- project_state$code_descriptions
        rv$history <- project_state$history
        rv$history_index <- project_state$history_index
        rv$current_project <- basename(filepath)
        rv$current_file_name <- project_state$current_file_name  # Restore current file name
        rv$project_modified <- FALSE

        # SYNC CODES WITH HIERARCHY AFTER LOADING
        rv <- sync_codes_with_hierarchy(rv)

        # UPDATE ANY DARK COLORS AFTER LOADING
        rv <- update_dark_colors(rv)

        # Update UI elements
        updateTextAreaInput(session, "text_input", value = rv$text)
        session$sendCustomMessage("clearSelection", list())

        showNotification("Project loaded successfully", type = "message")
      }, error = function(e) {
        showNotification(paste("Error loading project:", e$message), type = "error")
      })

      removeModal()
    })

    observeEvent(input$load_without_saving, {
      removeModal()
      load_selected_project()
    })

    load_selected_project <- function() {
      project_state <- load_project_state(paste0(input$project_to_load, ".rds"))
      if (!is.null(project_state)) {
        # Update all reactive values with loaded state
        rv$text <- project_state$text
        rv$annotations <- project_state$annotations
        rv$codes <- project_state$codes
        rv$code_tree <- project_state$code_tree
        rv$code_colors <- project_state$code_colors
        rv$memos <- project_state$memos
        rv$code_descriptions <- project_state$code_descriptions
        rv$history <- project_state$history
        rv$history_index <- project_state$history_index
        rv$current_project <- input$project_to_load
        rv$project_modified <- FALSE

        # Update UI elements
        updateTextAreaInput(session, "text_input", value = rv$text)
        session$sendCustomMessage("clearSelection", list())
      }
      removeModal()
    }

    # Add modified state tracking
    observe({
      # Only mark as modified if we actually have a project open and content changes
      if (!is.null(rv$current_project) && rv$current_project != "") {
        rv$project_modified <- TRUE
      }
    }, priority = 1000)

    # Add New Project functionality
    observeEvent(input$new_project, {
      if (rv$project_modified) {
        showModal(modalDialog(
          title = "Save Current Project?",
          "Would you like to save the current project before creating a new one?",
          footer = tagList(
            actionButton("save_before_new", "Save First"),
            actionButton("new_without_saving", "Don't Save"),
            modalButton("Cancel")
          )
        ))
      } else {
        confirm_new_project()
      }
    })

    confirm_new_project <- function() {
      showModal(modalDialog(
        title = "Confirm New Project",
        "Are you sure you want to create a new project? This will clear all current work.",
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_new_project", "Create New Project")
        )
      ))
    }

    # Handle new project confirmation
    observeEvent(input$confirm_new_project, {
      removeModal()
      create_new_project(rv, session)
    })

    observeEvent(input$save_before_new, {
      removeModal()
      showModal(modalDialog(
        title = "Save Project",
        textInput("project_name", "Project Name:",
                  value = rv$current_project %||% ""),
        selectInput("save_location", "Save Location:",
                    choices = c("Default Location" = "default",
                                "Custom Location" = "custom")),
        conditionalPanel(
          condition = "input.save_location == 'custom'",
          textInput("custom_save_path", "Custom Save Path:",
                    value = getwd())
        ),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_save_before_new", "Save")
        )
      ))
    })

    observeEvent(input$confirm_save_before_new, {
      req(input$project_name)

      # Save current project
      if (input$save_location == "default") {
        save_path <- get_project_dir()
      } else {
        save_path <- input$custom_save_path
        if (!dir.exists(save_path)) {
          dir.create(save_path, recursive = TRUE)
        }
      }

      filename <- if (!grepl("\\.rds$", input$project_name)) {
        paste0(input$project_name, ".rds")
      } else {
        input$project_name
      }

      filepath <- file.path(save_path, filename)

      project_state <- list(
        text = rv$text,
        annotations = rv$annotations,
        codes = rv$codes,
        code_tree = rv$code_tree,
        code_colors = rv$code_colors,
        memos = rv$memos,
        code_descriptions = rv$code_descriptions,
        history = rv$history,
        history_index = rv$history_index
      )

      handle_error(
        expr = {
          saveRDS(project_state, file = filepath)
          rv$current_project <- input$project_name
          rv$project_modified <- FALSE  # Reset modified flag after successful save
          showNotification(paste("Project saved successfully to", filepath),
                           type = "message")
          # Proceed to create new project after successful save
          removeModal()
          create_new_project(rv, session)
        },
        error_msg = paste("Failed to save project to", filepath)
      )
    })

    observeEvent(input$new_without_saving, {
      removeModal()
      create_new_project(rv, session)
    })

    # Helper function to create new project
    create_new_project <- function(rv, session) {
      rv$text <- ""
      rv$annotations <- data.frame(
        start = integer(),
        end = integer(),
        text = character(),
        code = character(),
        memo = character(),
        stringsAsFactors = FALSE
      )
      rv$codes <- character()
      rv$code_tree <- Node$new("Root")
      rv$code_colors <- character()
      rv$memos <- list()
      rv$code_descriptions <- list()
      rv$history <- list(list(text = "", annotations = data.frame()))
      rv$history_index <- 1
      rv$current_project <- NULL
      rv$project_modified <- FALSE
      rv$action_history <- list()
      rv$action_index <- 0
      rv$merged_codes <- list()

      # Clear UI elements
      updateTextAreaInput(session, "text_input", value = "")
      session$sendCustomMessage("clearSelection", list())

      showNotification("New project created", type = "message")
    }

    # Enhance error handling for file operations
    observeEvent(input$import_text, {
      req(input$file_input)

      # Check if there's existing work that might be lost
      has_existing_work <- (
        nchar(rv$text) > 0 ||
          nrow(rv$annotations) > 0 ||
          length(rv$codes) > 0
      )

      if (has_existing_work) {
        # Show confirmation dialog
        showModal(modalDialog(
          title = "Import New Text",
          size = "m",
          tags$div(
            tags$h4("Warning: This will clear all current work!", style = "color: #d9534f;"),
            tags$p("Importing a new text file will:"),
            tags$ul(
              tags$li("Replace the current text"),
              tags$li("Remove all existing annotations"),
              tags$li("Clear all codes and their colors"),
              tags$li("Reset the code hierarchy"),
              tags$li("Clear all memos")
            ),
            tags$br(),
            tags$p(
              style = "background-color: #f8f9fa; padding: 10px; border-radius: 4px; border-left: 4px solid #007bff;",
              icon("info-circle"),
              " ",
              strong("Recommendation: "),
              "Save your current project before proceeding if you want to keep your work."
            ),
            tags$br(),
            tags$p("Do you want to continue with importing the new text file?")
          ),
          footer = tagList(
            actionButton("save_before_import", "Save Project First",
                         class = "btn-primary", icon = icon("save")),
            modalButton("Cancel"),
            actionButton("confirm_import_text", "Import Without Saving",
                         class = "btn-warning", icon = icon("upload"))
          )
        ))
      } else {
        # No existing work, proceed directly
        perform_text_import()
      }
    })

    # Add handler for saving before import
    observeEvent(input$save_before_import, {
      removeModal()

      # Trigger save project functionality
      if (!is.null(rv$current_project) && !is.null(rv$current_project_path)) {
        # Quick save to existing location
        tryCatch({
          project_state <- list(
            text = rv$text,
            annotations = rv$annotations,
            codes = rv$codes,
            code_tree = rv$code_tree,
            code_colors = rv$code_colors,
            memos = rv$memos,
            code_descriptions = rv$code_descriptions,
            history = rv$history,
            history_index = rv$history_index,
            current_file_name = rv$current_file_name
          )

          saveRDS(project_state, file = rv$current_project_path)
          rv$project_modified <- FALSE
          showNotification("Project saved successfully. Proceeding with import...", type = "message")

          # Proceed with import after successful save
          perform_text_import()

        }, error = function(e) {
          showNotification(paste("Error saving project:", e$message), type = "error")
          # Show save dialog as fallback
          save_project_interactive(rv, input, session, roots)
          # Store flag to import after save
          rv$import_after_save <- TRUE
        })
      } else {
        # Show save dialog for new project
        save_project_interactive(rv, input, session, roots)
        # Store flag to import after save
        rv$import_after_save <- TRUE
      }
    })

    # Add handler for confirming import without saving
    observeEvent(input$confirm_import_text, {
      removeModal()
      perform_text_import()
    })

    perform_text_import <- function() {
      req(input$file_input)

      handle_error(
        expr = {
          # Show progress notification
          showNotification("Importing text file...", type = "message", duration = 2)

          # Get file extension and original file name
          file_ext <- tolower(tools::file_ext(input$file_input$name))
          rv$current_file_name <- basename(input$file_input$name)

          # Read the file based on its type
          if (file_ext == "txt") {
            # For plain text files, read directly with line breaks preserved
            text_content <- readLines(input$file_input$datapath, warn = FALSE)
            new_text <- paste(text_content, collapse = "\n")
          } else {
            # For other formats (docx, pdf), use readtext
            imported_text <- readtext(input$file_input$datapath)
            new_text <- imported_text$text
          }

          # COMPLETE RESET OF ALL APPLICATION STATE
          rv$text <- new_text
          rv$annotations <- data.frame(
            start = integer(),
            end = integer(),
            text = character(),
            code = character(),
            memo = character(),
            source_file = character(),
            stringsAsFactors = FALSE
          )
          rv$codes <- character()
          rv$code_colors <- character()
          rv$memos <- list()
          rv$code_descriptions <- list()

          # Reset code hierarchy to fresh state
          rv$code_tree <- Node$new("Root")
          rv$code_tree$type <- "theme"
          rv$code_tree$description <- "Root of the code hierarchy"

          # Reset history with the new clean state
          rv$history <- list(list(text = rv$text, annotations = rv$annotations))
          rv$history_index <- 1

          # Clear action history for undo/redo
          rv$action_history <- list()
          rv$action_index <- 0

          # Clear any selection state
          rv$selected_start <- NULL
          rv$selected_end <- NULL
          rv$selected_theme <- NULL

          # Mark project as modified since we've imported new content
          rv$project_modified <- TRUE

          # COMPREHENSIVE UI CLEANUP
          # Clear all input fields
          updateTextInput(session, "code", value = "")
          updateTextAreaInput(session, "memo", value = "")
          updateTextInput(session, "codebook_new_code", value = "")
          updateTextAreaInput(session, "codebook_code_description", value = "")

          # Clear any text selection
          session$sendCustomMessage("clearSelection", list())

          # Force immediate update of text display
          output$text_display <- renderUI({
            HTML(update_text_display())
          })

          # Also update floating text display
          output$floating_text_display <- renderUI({
            HTML(update_text_display())
          })

          # Use a slight delay to ensure UI updates are processed
          session$sendCustomMessage("refreshDisplay", list())

          # Success message
          showNotification(
            paste("Text imported successfully from", rv$current_file_name,
                  "- All previous work has been cleared."),
            type = "message",
            duration = 4
          )

        },
        error_msg = "Failed to import text file"
      )
    }

    # Add observer to handle import after save completion
    observeEvent(rv$project_modified, {
      # Check if we should import after saving
      if (!is.null(rv$import_after_save) && rv$import_after_save && !rv$project_modified) {
        rv$import_after_save <- NULL  # Clear the flag
        perform_text_import()
      }
    })

    observeEvent(input$create_code, {
      if (!is.null(rv$selected_start) && !is.null(rv$selected_end)) {
        new_annotation <- data.frame(
          start = rv$selected_start,
          end = rv$selected_end,
          text = substr(rv$text, rv$selected_start, rv$selected_end),
          code = input$code,
          memo = input$memo,
          source_file = rv$current_file_name %||% "Unknown",  # Include source file
          stringsAsFactors = FALSE
        )

        # Create and apply the action
        add_action <- create_action(
          type = "add_annotation",
          data = new_annotation,
          reverse_data = new_annotation  # Same data used for reverse action
        )

        apply_action(rv, add_action)
        add_action(rv, add_action)

        # Update codes list
        rv$codes <- unique(c(rv$codes, input$code))

        # AUTO-ADD CODE TO ROOT IF NOT ALREADY IN HIERARCHY
        if (!code_exists_in_hierarchy(rv$code_tree, input$code)) {
          new_code_node <- rv$code_tree$AddChild(input$code)
          new_code_node$type <- "code"
          new_code_node$description <- ""
          new_code_node$created <- Sys.time()
        }

        # Assign color if needed using readable color palette
        if (!(input$code %in% names(rv$code_colors))) {
          used_colors <- as.character(rv$code_colors)
          new_color <- get_next_palette_color(used_colors)

          # If all palette colors are used, generate a random readable color
          if (new_color %in% used_colors) {
            new_color <- generate_readable_color()

            # Ensure the generated color is actually readable
            attempts <- 0
            while (!is_color_readable(new_color) && attempts < 10) {
              new_color <- generate_readable_color()
              attempts <- attempts + 1
            }

            # If we still don't have a readable color after 10 attempts, use a safe default
            if (!is_color_readable(new_color)) {
              safe_colors <- c("#FFE6CC", "#E6F3FF", "#E6FFE6", "#FFE6F3", "#F3E6FF")
              new_color <- sample(safe_colors, 1)
            }
          }

          rv$code_colors[input$code] <- new_color
        }

        # Clear inputs
        updateTextInput(session, "code", value = "")
        updateTextAreaInput(session, "memo", value = "")
        rv$selected_start <- NULL
        rv$selected_end <- NULL
        session$sendCustomMessage("clearSelection", list())

        # Update UI
        output$text_display <- renderUI({
          HTML(update_text_display())
        })
      }
    })

    # Save Records handler
    observeEvent(input$save_records, {
      save_records_interactive(rv, input, session, roots)
    })

    # Save records confirmation handler
    observeEvent(input$confirm_save_records, {
      req(input$save_filename)
      req(input$records_directory_select)

      # Get selected directory path
      dir_path <- parseDirPath(roots, input$records_directory_select)

      # Construct full filepath
      filename <- input$save_filename
      if (!grepl(paste0("\\.", input$save_format, "$"), filename)) {
        filename <- paste0(filename, ".", input$save_format)
      }
      filepath <- file.path(dir_path, filename)

      # Save records
      tryCatch({
        dir.create(dirname(filepath), recursive = TRUE, showWarnings = FALSE)

        if (input$save_format == "csv") {
          # Save as CSV
          write.csv(rv$annotations, file = filepath, row.names = FALSE)
        } else {
          # Save as JSON
          write_json(rv$annotations, filepath)
        }

        showNotification(paste("Records saved successfully to", filepath), type = "message")
      }, error = function(e) {
        showNotification(paste("Error saving records:", e$message), type = "error")
      })

      removeModal()
    })

    observeEvent(input$export_hierarchy_toolbar, {
      # Call the existing export_hierarchy_btn function
      showModal(modalDialog(
        title = "Export Hierarchy",
        downloadButton("download_hierarchy", "Download JSON"),
        footer = modalButton("Close")
      ))
    })

    # Import hierarchy toolbar button handler
    observeEvent(input$import_hierarchy_toolbar, {
      # Call the existing import_hierarchy_btn function
      showModal(modalDialog(
        title = "Import Hierarchy",
        fileInput("hierarchy_file", "Choose JSON file",
                  accept = c("application/json", ".json")),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_import", "Import")
        )
      ))
    })

    # Undo functionality
    observeEvent(input$undo, {
      if (rv$action_index > 0) {
        action <- rv$action_history[[rv$action_index]]
        apply_action(rv, action, reverse = TRUE)
        rv$action_index <- rv$action_index - 1

        # Update UI
        output$text_display <- renderUI({
          HTML(update_text_display())
        })
      }
    })

    # Redo functionality
    observeEvent(input$redo, {
      if (rv$action_index < length(rv$action_history)) {
        rv$action_index <- rv$action_index + 1
        action <- rv$action_history[[rv$action_index]]
        apply_action(rv, action, reverse = FALSE)

        # Update UI
        output$text_display <- renderUI({
          HTML(update_text_display())
        })
      }
    })

    # Helper function to save state
    save_state <- function() {
      current_state <- list(
        text = rv$text,
        annotations = rv$annotations,
        codes = rv$codes,
        code_tree = rv$code_tree,
        code_colors = rv$code_colors,
        memos = rv$memos,
        code_descriptions = rv$code_descriptions
      )

      # Remove any future states if we're not at the end of the history
      if (rv$history_index < length(rv$history)) {
        rv$history <- rv$history[1:rv$history_index]
      }

      rv$history <- c(rv$history, list(current_state))
      rv$history_index <- length(rv$history)
    }

    # Modify the JavaScript for text selection
    observeEvent(input$select, {
      runjs('
        // Global variables for selection state
        var selectionState = {
          isActive: false,
          startChar: null,
          endChar: null,
          isSelecting: false
        };

        // Check current state and toggle
        var currentlyActive = $("#text_display, #floating_text_content").hasClass("selection-active");
        var newState = !currentlyActive;

        // Initialize selection functionality
        function initializeTextSelection() {
          // Remove any existing event listeners to prevent duplicates
          $(document).off("mousedown.textselection mousemove.textselection mouseup.textselection");
          $("#text_display, #floating_text_content").off("mousedown.textselection mousemove.textselection");

          // Add namespaced event listeners
          $(document).on("mousedown.textselection", "#text_display .char, #floating_text_content .char", function(e) {
            if (!selectionState.isActive) return;

            e.preventDefault();
            e.stopPropagation();

            try {
              var charId = $(this).attr("id");
              if (!charId) return;

              var charNum = parseInt(charId.replace(/^(char_|float_char_)/, ""));
              if (isNaN(charNum)) return;

              selectionState.startChar = charNum;
              selectionState.endChar = charNum;
              selectionState.isSelecting = true;

              updateHighlight();
            } catch (error) {
              console.error("Error in mousedown handler:", error);
            }
          });

          $(document).on("mousemove.textselection", "#text_display .char, #floating_text_content .char", function(e) {
            if (!selectionState.isActive || !selectionState.isSelecting) return;

            try {
              var charId = $(this).attr("id");
              if (!charId) return;

              var charNum = parseInt(charId.replace(/^(char_|float_char_)/, ""));
              if (isNaN(charNum)) return;

              selectionState.endChar = charNum;
              updateHighlight();
            } catch (error) {
              console.error("Error in mousemove handler:", error);
            }
          });

          $(document).on("mouseup.textselection", function(e) {
            if (!selectionState.isActive || !selectionState.isSelecting) return;

            try {
              selectionState.isSelecting = false;

              if (selectionState.startChar !== null && selectionState.endChar !== null) {
                updateShinySelection();
              }
            } catch (error) {
              console.error("Error in mouseup handler:", error);
            }
          });
        }

        function updateHighlight() {
          try {
            // Clear all existing highlights
            $(".char").removeClass("highlighted");

            if (selectionState.startChar === null || selectionState.endChar === null) {
              return;
            }

            var start = Math.min(selectionState.startChar, selectionState.endChar);
            var end = Math.max(selectionState.startChar, selectionState.endChar);

            // Apply highlights to the range
            for (var i = start; i <= end; i++) {
              $("#char_" + i + ", #float_char_" + i).addClass("highlighted");
            }
          } catch (error) {
            console.error("Error updating highlights:", error);
          }
        }

        function updateShinySelection() {
          try {
            if (selectionState.startChar === null || selectionState.endChar === null) {
              return;
            }

            var start = Math.min(selectionState.startChar, selectionState.endChar);
            var end = Math.max(selectionState.startChar, selectionState.endChar);

            // Get the selected text by collecting text from highlighted characters
            var selectedText = "";
            for (var i = start; i <= end; i++) {
              var charElement = $("#char_" + i);
              if (charElement.length > 0) {
                selectedText += charElement.text();
              }
            }

            // Send selection to Shiny
            Shiny.setInputValue("selected_text", {
              text: selectedText,
              start: start,
              end: end
            });

            console.log("Selection sent to Shiny:", {text: selectedText, start: start, end: end});
          } catch (error) {
            console.error("Error updating Shiny selection:", error);
          }
        }

        function toggleSelectionMode(active) {
          selectionState.isActive = active;

          if (active) {
            $("#text_display, #floating_text_content").addClass("selection-active");
            $("#selection-indicator").addClass("active");
            $("#selection-indicator").text("Selection Mode Active - Click and drag to select text");
            $("#select").addClass("btn-toggle-active").removeClass("btn-toggle-inactive");
          } else {
            $("#text_display, #floating_text_content").removeClass("selection-active");
            $("#selection-indicator").removeClass("active");
            $("#selection-indicator").text("Selection Mode is Inactive");
            $("#select").addClass("btn-toggle-inactive").removeClass("btn-toggle-active");
            clearSelection();
          }
        }

        function clearSelection() {
          $(".char").removeClass("highlighted");
          selectionState.startChar = null;
          selectionState.endChar = null;
          selectionState.isSelecting = false;
          Shiny.setInputValue("selected_text", null);
        }

        // Add selection indicator if it doesnt exist
        if ($("#selection-indicator").length === 0) {
          $("body").append("<div id=\\"selection-indicator\\" class=\\"selection-mode-indicator\\">Selection Mode is Inactive</div>");
        }

        // Initialize and toggle selection mode
        initializeTextSelection();
        toggleSelectionMode(newState);

        // Custom message handlers for Shiny integration
        Shiny.addCustomMessageHandler("startSelecting", function(message) {
          toggleSelectionMode(true);
        });

        Shiny.addCustomMessageHandler("clearSelection", function(message) {
          clearSelection();
        });

        Shiny.addCustomMessageHandler("refreshDisplay", function(message) {
          setTimeout(function() {
            initializeTextSelection();
          }, 100);
        });
      ')
    })

    # Handle text selection
    observeEvent(input$selected_text, {
      if (!is.null(input$selected_text)) {
        rv$selected_start <- input$selected_text$start
        rv$selected_end <- input$selected_text$end
      }
    })

    # Clear button
    observeEvent(input$clear, {
      rv$selected_start <- NULL
      rv$selected_end <- NULL
      updateTextInput(session, "code", value = "")
      updateTextAreaInput(session, "memo", value = "")
      runjs('
        $(".char").removeClass("highlighted");
        $("#text_display, #floating_text_content").removeClass("selection-active");
        $("#selection-indicator").removeClass("active");
        $("#selection-indicator").text("Selection Mode is Inactive");
        $("#select").addClass("btn-toggle-inactive").removeClass("btn-toggle-active");
        if (typeof selectionState !== "undefined") {
          selectionState.isActive = false;
          selectionState.startChar = null;
          selectionState.endChar = null;
          selectionState.isSelecting = false;
        }
      ')
      session$sendCustomMessage("clearSelection", list())
    })

    # Update the save function
    observeEvent(input$create_code, {
      if (!is.null(rv$selected_start) && !is.null(rv$selected_end)) {
        new_annotation <- data.frame(
          start = rv$selected_start,
          end = rv$selected_end,
          text = substr(rv$text, rv$selected_start, rv$selected_end),
          code = input$code,
          memo = input$memo,
          source_file = rv$current_file_name %||% "Unknown",  # Include source file
          stringsAsFactors = FALSE
        )

        # Create and apply the action
        add_action <- create_action(
          type = "add_annotation",
          data = new_annotation,
          reverse_data = new_annotation  # Same data used for reverse action
        )

        apply_action(rv, add_action)
        add_action(rv, add_action)

        # Update codes list
        rv$codes <- unique(c(rv$codes, input$code))

        # AUTO-ADD CODE TO ROOT IF NOT ALREADY IN HIERARCHY
        if (!code_exists_in_hierarchy(rv$code_tree, input$code)) {
          new_code_node <- rv$code_tree$AddChild(input$code)
          new_code_node$type <- "code"
          new_code_node$description <- ""
          new_code_node$created <- Sys.time()
        }

        # Assign color if needed using readable color palette
        if (!(input$code %in% names(rv$code_colors))) {
          used_colors <- as.character(rv$code_colors)
          new_color <- get_next_palette_color(used_colors)

          # If all palette colors are used, generate a random readable color
          if (new_color %in% used_colors) {
            new_color <- generate_readable_color()

            # Ensure the generated color is actually readable
            attempts <- 0
            while (!is_color_readable(new_color) && attempts < 10) {
              new_color <- generate_readable_color()
              attempts <- attempts + 1
            }

            # If we still don't have a readable color after 10 attempts, use a safe default
            if (!is_color_readable(new_color)) {
              safe_colors <- c("#FFE6CC", "#E6F3FF", "#E6FFE6", "#FFE6F3", "#F3E6FF")
              new_color <- sample(safe_colors, 1)
            }
          }

          rv$code_colors[input$code] <- new_color
        }

        # Clear inputs
        updateTextInput(session, "code", value = "")
        updateTextAreaInput(session, "memo", value = "")
        rv$selected_start <- NULL
        rv$selected_end <- NULL
        session$sendCustomMessage("clearSelection", list())

        # Update UI
        output$text_display <- renderUI({
          HTML(update_text_display())
        })
      }
    })

    # Apply Code button
    observeEvent(input$apply_code, {
      if (!is.null(rv$selected_start) && !is.null(rv$selected_end)) {
        showModal(modalDialog(
          title = "Apply Code",
          selectInput("code_to_apply", "Select a code to apply:", choices = rv$codes),
          footer = tagList(
            modalButton("Cancel"),
            actionButton("confirm_apply_code", "Apply")
          )
        ))
      } else {
        showNotification("Please select text before applying a code.", type = "warning")
      }
    })

    # Confirm Apply Code
    observeEvent(input$confirm_apply_code, {
      removeModal()
      if (!is.null(rv$selected_start) && !is.null(rv$selected_end)) {
        new_annotation <- data.frame(
          start = rv$selected_start,
          end = rv$selected_end,
          text = substr(rv$text, rv$selected_start, rv$selected_end),
          code = input$code_to_apply,
          memo = "",
          source_file = rv$current_file_name %||% "Unknown",  # Include source file
          stringsAsFactors = FALSE
        )
        rv$annotations <- rbind(rv$annotations, new_annotation)

        # Update codes list
        rv$codes <- unique(c(rv$codes, input$code_to_apply))

        # AUTO-ADD CODE TO ROOT IF NOT ALREADY IN HIERARCHY
        if (!code_exists_in_hierarchy(rv$code_tree, input$code_to_apply)) {
          new_code_node <- rv$code_tree$AddChild(input$code_to_apply)
          new_code_node$type <- "code"
          new_code_node$description <- ""
          new_code_node$created <- Sys.time()
        }

        # Assign color if needed using improved color system
        if (!(input$code_to_apply %in% names(rv$code_colors))) {
          used_colors <- as.character(rv$code_colors)
          new_color <- get_next_palette_color(used_colors)

          # If all palette colors are used, generate a random readable color
          if (new_color %in% used_colors) {
            new_color <- generate_readable_color()

            # Ensure the generated color is actually readable
            attempts <- 0
            while (!is_color_readable(new_color) && attempts < 10) {
              new_color <- generate_readable_color()
              attempts <- attempts + 1
            }

            # If we still don't have a readable color after 10 attempts, use a safe default
            if (!is_color_readable(new_color)) {
              safe_colors <- c("#FFE6CC", "#E6F3FF", "#E6FFE6", "#FFE6F3", "#F3E6FF")
              new_color <- sample(safe_colors, 1)
            }
          }

          rv$code_colors[input$code_to_apply] <- new_color
        }

        updateTextInput(session, "code", value = input$code_to_apply)
        showNotification("Code applied successfully!", type = "message")
        save_state()

        # Update the text display
        output$text_display <- renderUI({
          HTML(update_text_display())
        })
      }
    })

    # Display Code button
    observeEvent(input$display_code, {
      sorted_annotations <- rv$annotations[order(rv$annotations$start), ]
      displayed_text <- ""
      last_end <- 0

      for (i in 1:nrow(sorted_annotations)) {
        if (sorted_annotations$start[i] > last_end + 1) {
          displayed_text <- paste0(displayed_text,
                                   substr(rv$text, last_end + 1, sorted_annotations$start[i] - 1))
        }
        displayed_text <- paste0(displayed_text,
                                 "<span class='code-display'>[", sorted_annotations$code[i], "]</span>",
                                 substr(rv$text, sorted_annotations$start[i], sorted_annotations$end[i]))
        last_end <- sorted_annotations$end[i]
      }

      if (last_end < nchar(rv$text)) {
        displayed_text <- paste0(displayed_text, substr(rv$text, last_end + 1, nchar(rv$text)))
      }

      showModal(modalDialog(
        title = "Coded Text Display",
        tags$div(style = "white-space: pre-wrap; line-height: 1.5;", HTML(displayed_text)),
        size = "l",
        easyClose = TRUE
      ))
    })

    # Export Codes
    observeEvent(input$export_codes, {
      codes_df <- data.frame(
        code = get_code_names(rv$code_tree),
        path = get_code_paths(rv$code_tree),
        color = rv$code_colors[get_code_names(rv$code_tree)]
      )
      write.csv(codes_df, file = "exported_codes.csv", row.names = FALSE)
      showNotification("Codes exported successfully", type = "message")
    })

    # Helper function to get code paths
    get_code_paths <- function(node) {
      if (node$isRoot) {
        return(character(0))
      } else {
        return(c(paste(node$path, collapse = "/"), unlist(lapply(node$children, get_code_paths))))
      }
    }

    # Update the display of annotations in the Records tab
    output$annotations <- renderDT({
      datatable(rv$annotations, options = list(pageLength = 5))
    })

    # Import Annotations
    observeEvent(input$import_annotations, {
      showModal(modalDialog(
        title = "Import Annotations",
        fileInput("annotation_file", "Choose JSON File", accept = c("application/json", ".json")),
        footer = tagList(
          modalButton("Cancel"),
          actionButton("confirm_import_annotations", "Import")
        )
      ))
    })

    observeEvent(input$confirm_import_annotations, {
      req(input$annotation_file)
      imported_annotations <- fromJSON(input$annotation_file$datapath)
      rv$annotations <- rbind(rv$annotations, imported_annotations)
      removeModal()
      save_state()
    })

    # Export Annotations
    observeEvent(input$export_annotations, {
      write_json(rv$annotations, "exported_annotations.json")
      showNotification("Annotations exported successfully", type = "message")
    })

    # Event handler for Code Frequency button
    observeEvent(input$code_frequency, {
      req(nrow(rv$annotations) > 0)
      plot <- generate_code_frequency_plot(rv$annotations)
      output$code_freq_plot <- renderPlot({ plot })
      showModal(modalDialog(
        title = "Code Frequency",
        plotOutput("code_freq_plot"),
        size = "l",
        easyClose = TRUE
      ))
    })

    # Event handler for Code Co-occurrence button
    observeEvent(input$code_co_occurrence, {
      req(nrow(rv$annotations) > 0)

      # Get the selected analytical unit
      unit <- input$cooccurrence_unit %||% "paragraph"

      # Generate the enhanced analysis with unit-based methodology
      analysis_results <- generate_code_co_occurrence_analysis(
        annotations = rv$annotations,
        text = rv$text,
        unit = unit
      )

      # Store results in reactive values for access across multiple outputs
      rv$co_occurrence_results <- analysis_results

      # Create the modal dialog with multiple tabs including unit information
      showModal(modalDialog(
        title = paste("Code Co-occurrence Analysis -",
                      tools::toTitleCase(analysis_results$unit_info$analytical_unit), "Level"),

        tabsetPanel(
          id = "co_occurrence_tabs",

          # Unit Information tab
          tabPanel("Analysis Info",
                   fluidRow(
                     column(6,
                            h4("Analysis Settings"),
                            tags$div(
                              tags$p(tags$strong("Analytical Unit: "),
                                     tools::toTitleCase(analysis_results$unit_info$analytical_unit)),
                              tags$p(tags$strong("Total Units Analyzed: "),
                                     analysis_results$unit_info$total_units),
                              tags$p(tags$strong("Mean Codes per Unit: "),
                                     sprintf("%.2f", analysis_results$unit_info$codes_per_unit$mean_codes_per_unit)),
                              tags$p(tags$strong("Max Codes in Single Unit: "),
                                     analysis_results$unit_info$codes_per_unit$max_codes_per_unit),
                              tags$p(tags$strong("Units with Multiple Codes: "),
                                     analysis_results$unit_info$codes_per_unit$units_with_multiple_codes)
                            )
                     ),
                     column(6,
                            h4("Co-occurrence Definition"),
                            tags$div(
                              switch(analysis_results$unit_info$analytical_unit,
                                     "sentence" = tags$div(
                                       tags$p("Codes are considered co-occurring when they appear within the same sentence."),
                                       tags$p(tags$em("Example: 'Theme A' and 'Theme B' codes applied to different parts of the same sentence will be counted as co-occurring."))
                                     ),
                                     "paragraph" = tags$div(
                                       tags$p("Codes are considered co-occurring when they appear within the same paragraph."),
                                       tags$p(tags$em("Example: 'Theme A' and 'Theme B' codes applied to different sentences within the same paragraph will be counted as co-occurring."))
                                     ),
                                     "document" = tags$div(
                                       tags$p("Codes are considered co-occurring when they appear anywhere within the same document."),
                                       tags$p(tags$em("Example: 'Theme A' and 'Theme B' codes applied to any parts of the document will be counted as co-occurring."))
                                     )
                              )
                            )
                     )
                   )),

          # Network visualization tab
          tabPanel("Network View",
                   plotOutput("code_co_occurrence_network", height = "500px"),
                   hr(),
                   helpText("Node size reflects total co-occurrences. Line thickness indicates Jaccard similarity strength. Line opacity shows phi coefficient magnitude.")),

          # Heatmap visualization tab
          tabPanel("Heatmap View",
                   plotOutput("code_co_occurrence_heatmap", height = "500px"),
                   hr(),
                   helpText("Darker colors indicate stronger co-occurrence relationships based on Jaccard similarity coefficients.")),

          # Statistics tab
          tabPanel("Statistics",
                   h4("Summary Statistics"),
                   tableOutput("co_occurrence_summary"),
                   hr(),
                   h4("Detailed Co-occurrence Matrix"),
                   DTOutput("co_occurrence_table"))
        ),

        size = "l",
        easyClose = TRUE,
        footer = modalButton("Close")
      ))
    })

    output$code_co_occurrence_network <- renderPlot({
      req(rv$co_occurrence_results)
      print(rv$co_occurrence_results$network_plot)
    })

    output$code_co_occurrence_heatmap <- renderPlot({
      req(rv$co_occurrence_results)
      print(rv$co_occurrence_results$heatmap_plot)
    })

    output$co_occurrence_summary <- renderTable({
      req(rv$co_occurrence_results)
      summary_df <- data.frame(
        Metric = c("Analytical Unit",
                   "Total Units Analyzed",
                   "Units with Multiple Codes",
                   "Total Number of Codes",
                   "Maximum Co-occurrence Count",
                   "Maximum Jaccard Similarity",
                   "Mean Jaccard Similarity",
                   "Significant Pairs (|phi| > 0.3)"),
        Value = c(tools::toTitleCase(rv$co_occurrence_results$unit_info$analytical_unit),
                  rv$co_occurrence_results$unit_info$total_units,
                  rv$co_occurrence_results$unit_info$codes_per_unit$units_with_multiple_codes,
                  rv$co_occurrence_results$summary$total_codes,
                  rv$co_occurrence_results$summary$max_co_occurrence,
                  round(rv$co_occurrence_results$summary$max_jaccard, 3),
                  round(rv$co_occurrence_results$summary$mean_jaccard, 3),
                  rv$co_occurrence_results$summary$significant_pairs)
      )
      summary_df
    })

    output$co_occurrence_table <- renderDT({
      req(rv$co_occurrence_results)
      # Convert matrices to data frames for display
      co_df <- as.data.frame(rv$co_occurrence_results$co_occurrence)
      jaccard_df <- as.data.frame(round(rv$co_occurrence_results$jaccard_similarity, 3))
      phi_df <- as.data.frame(round(rv$co_occurrence_results$phi_coefficient, 3))

      # Add row names as a column
      co_df$Code <- rownames(co_df)
      jaccard_df$Code <- rownames(jaccard_df)
      phi_df$Code <- rownames(phi_df)

      # Move Code column to front
      co_df <- co_df[, c(ncol(co_df), 1:(ncol(co_df)-1))]

      datatable(co_df,
                options = list(
                  pageLength = 10,
                  scrollX = TRUE,
                  dom = 'Bfrtip',
                  buttons = c('copy', 'csv', 'excel')
                ),
                caption = "Raw Co-occurrence Counts") %>%
        DT::formatStyle(names(co_df)[-1],
                        background = styleInterval(c(0, 2, 5, 10),
                                                   c('white', '#f7fbff', '#deebf7', '#9ecae1', '#3182bd')))
    })

    # Word Cloud button
    observeEvent(input$word_cloud, {
      word_cloud <- generate_word_cloud(rv$text)
      output$word_cloud_plot <- renderPlot({ word_cloud })
      showModal(modalDialog(
        title = "Word Cloud",
        plotOutput("word_cloud_plot", height = "500px"),
        size = "l",
        easyClose = TRUE
      ))
    })

    # Text Summary button
    observeEvent(input$text_summary, {
      summary <- generate_text_summary(rv$text, rv$annotations)
      output$text_summary_table <- renderTable({
        data.frame(Metric = names(summary), Value = unlist(summary))
      })
      showModal(modalDialog(
        title = "Text Summary",
        tableOutput("text_summary_table"),
        size = "m",
        easyClose = TRUE
      ))
    })

    # Memo Linking
    #observeEvent(input$link_memo, {
    #  showModal(modalDialog(
    #    title = "Link Memo",
    #    selectInput("memo_link_type", "Link to:",
    #                choices = c("Code", "Document")),
    #    uiOutput("memo_link_options"),
    #    textAreaInput("memo_text", "Memo:"),
    #    footer = tagList(
    #      modalButton("Cancel"),
    #      actionButton("confirm_link_memo", "Link")
    #    )
    # ))
    #})

    output$memo_link_options <- renderUI({
      if (input$memo_link_type == "Code") {
        selectInput("memo_link_code", "Select Code:",
                    choices = rv$codes)  # Use rv$codes instead of get_code_names(rv$code_tree)
      } else {
        # For document-level memo, no additional input is needed
        NULL
      }
    })

    # Update the Link Memo functionality
    #observeEvent(input$confirm_link_memo, {
    #  new_memo <- list(
    #    type = input$memo_link_type,
    #    link = if (input$memo_link_type == "Code") input$memo_link_code else "Document",
    #    text = input$memo_text
    #  )

    #  if (new_memo$type == "Code") {
    #    # Find all annotations with the selected code and update their memos
    #    code_indices <- which(rv$annotations$code == new_memo$link)
    #    if (length(code_indices) > 0) {
    #      rv$annotations$memo[code_indices] <- sapply(rv$annotations$memo[code_indices],
    #                                                  function(memo) concatenate_memos(memo, new_memo$text))
    #    }
    #  } else {
    #    # For document-level memos, we'll add them to all annotations
    #    rv$annotations$memo <- sapply(rv$annotations$memo,
    #                                  function(memo) concatenate_memos(memo, new_memo$text))
    #  }

    #  rv$memos <- c(rv$memos, list(new_memo))
    #  removeModal()
    #  save_state()
    #})

    # Helper function to concatenate memos without extra semicolons
    concatenate_memos <- function(existing_memo, new_memo) {
      if (existing_memo == "") {
        return(new_memo)
      } else {
        return(paste(existing_memo, new_memo, sep = "; "))
      }
    }

    # Code Book Generation
    #observeEvent(input$generate_codebook, {
    #  codebook <- lapply(get_code_names(rv$code_tree), function(code) {
    #    code_annotations <- rv$annotations[rv$annotations$code == code, ]
    #   list(
    #      code = code,
    #      description = rv$code_descriptions[[code]] %||% "",
    #      example_quotes = if (nrow(code_annotations) > 0) {
    #        sample(code_annotations$text, min(3, nrow(code_annotations)))
    #      } else {
    #        character(0)
    #      }
    #    )
    #  })

    #  # Generate Markdown for the codebook
    #  codebook_md <- sapply(codebook, function(entry) {
    #    paste0(
    #      "## ", entry$code, "\n\n",
    #      "**Description:** ", entry$description, "\n\n",
    #      "**Example Quotes:**\n",
    #      paste0("- ", entry$example_quotes, collapse = "\n"), "\n\n"
    #    )
    #  })

    #  # Write the codebook to a file
    #  writeLines(c("# Code Book\n\n", codebook_md), "codebook.md")
    # showNotification("Code book generated successfully", type = "message")
    #})

    # Handle file uploads for comparison
    observeEvent(input$comparison_file, {
      req(input$comparison_file)

      # Load all uploaded files
      comparison_data <- lapply(input$comparison_file$datapath, function(path) {
        tryCatch({
          ext <- tolower(tools::file_ext(path))

          if (ext == "csv") {
            # Handle CSV files
            df <- read.csv(path, stringsAsFactors = FALSE)
          } else if (ext == "json") {
            # Handle JSON files
            df <- fromJSON(path)
          } else {
            showNotification(paste("Unsupported file format:", ext), type = "error")
            return(NULL)
          }

          # Ensure proper column types and structure
          if (is.data.frame(df)) {
            # Check required columns
            required_cols <- c("start", "end", "code")
            if (!all(required_cols %in% colnames(df))) {
              missing_cols <- setdiff(required_cols, colnames(df))
              showNotification(paste("Missing required columns:",
                                     paste(missing_cols, collapse = ", ")),
                               type = "error")
              return(NULL)
            }

            # Convert columns to proper types
            df$start <- as.numeric(as.character(df$start))
            df$end <- as.numeric(as.character(df$end))
            df$code <- as.character(df$code)

            # Remove rows with NA values in required columns
            df <- df[complete.cases(df[, required_cols]), ]

            # Ensure data frame has at least one row
            if (nrow(df) == 0) {
              showNotification("No valid annotations found after cleaning", type = "warning")
              return(NULL)
            }

            return(df)
          }
          return(NULL)

        }, error = function(e) {
          showNotification(paste("Error loading file:", e$message), type = "error")
          return(NULL)
        })
      })

      # Remove any NULL entries from failed loads
      comparison_data <- comparison_data[!sapply(comparison_data, is.null)]

      if (length(comparison_data) >= 2) {
        rv$comparison_data <- comparison_data
        showNotification("Comparison data loaded successfully", type = "message")
      } else {
        showNotification("Need at least two valid annotation sets for comparison",
                         type = "warning")
      }
    })

    observeEvent(input$comparison_file1, {
      req(input$comparison_file1)

      # Process the first file
      df1 <- tryCatch({
        process_comparison_file(input$comparison_file1$datapath)
      }, error = function(e) {
        showNotification(paste("Error processing first file:", e$message), type = "error")
        return(NULL)
      })

      if (!is.null(df1)) {
        rv$comparison_file1 <- df1
        showNotification("First file loaded successfully", type = "message")
      }
    })

    observeEvent(input$comparison_file2, {
      req(input$comparison_file2)

      # Process the second file
      df2 <- tryCatch({
        process_comparison_file(input$comparison_file2$datapath)
      }, error = function(e) {
        showNotification(paste("Error processing second file:", e$message), type = "error")
        return(NULL)
      })

      if (!is.null(df2)) {
        rv$comparison_file2 <- df2
        showNotification("Second file loaded successfully", type = "message")
      }
    })

    # Helper function to process comparison files
    process_comparison_file <- function(filepath) {
      ext <- tolower(tools::file_ext(filepath))

      df <- if (ext == "csv") {
        read.csv(filepath, stringsAsFactors = FALSE)
      } else if (ext == "json") {
        fromJSON(filepath)
      } else {
        stop("Unsupported file format")
      }

      # If the file is from Records tab (has text and memo columns), reformat it
      if (all(c("start", "end", "text", "code", "memo") %in% colnames(df))) {
        df <- df[, c("start", "end", "code")]
      }

      # Check required columns
      required_cols <- c("start", "end", "code")
      if (!all(required_cols %in% colnames(df))) {
        missing_cols <- setdiff(required_cols, colnames(df))
        stop(paste("Missing required columns:", paste(missing_cols, collapse = ", ")))
      }

      # Convert columns to proper types
      df$start <- as.numeric(as.character(df$start))
      df$end <- as.numeric(as.character(df$end))
      df$code <- as.character(df$code)

      # Remove rows with NA values
      df <- df[complete.cases(df[, required_cols]), ]

      if (nrow(df) == 0) {
        stop("No valid annotations found after cleaning")
      }

      return(df)
    }

    # Add file info displays
    output$file1_info <- renderUI({
      if (!is.null(rv$comparison_file1)) {
        div(
          style = "margin: 10px 0; padding: 10px; background-color: #e8f4e8; border-radius: 5px;",
          tags$p(
            icon("check-circle", class = "text-success"),
            strong("File 1 loaded:"),
            br(),
            paste("Number of annotations:", nrow(rv$comparison_file1))
          )
        )
      }
    })

    output$file2_info <- renderUI({
      if (!is.null(rv$comparison_file2)) {
        div(
          style = "margin: 10px 0; padding: 10px; background-color: #e8f4e8; border-radius: 5px;",
          tags$p(
            icon("check-circle", class = "text-success"),
            strong("File 2 loaded:"),
            br(),
            paste("Number of annotations:", nrow(rv$comparison_file2))
          )
        )
      }
    })

    # Add reset functionality
    observeEvent(input$reset_comparison, {
      rv$comparison_file1 <- NULL
      rv$comparison_file2 <- NULL
      rv$comparison_data <- NULL
      rv$comparison_results <- NULL

      # Reset file inputs (this requires a bit of JavaScript)
      runjs("
    document.getElementById('comparison_file1').value = '';
    document.getElementById('comparison_file2').value = '';
  ")

      showNotification("Comparison files have been reset", type = "message")
    })

    # Run comparison analysis
    observeEvent(input$run_comparison, {
      if (is.null(rv$comparison_file1) || is.null(rv$comparison_file2)) {
        showNotification("Please upload both files before running comparison",
                         type = "warning")
        return()
      }

      # Run comparison with the two files
      rv$comparison_data <- list(rv$comparison_file1, rv$comparison_file2)

      withProgress(message = 'Running comparison analysis...', {
        rv$comparison_results <- tryCatch({
          results <- generate_comparison_analysis(rv$comparison_data)
          plots <- generate_comparison_plots(results)
          c(results, list(plots = plots))
        }, error = function(e) {
          showNotification(paste("Error in comparison analysis:", e$message),
                           type = "error")
          NULL
        })
      })
    })

    observe({
      req(rv$comparison_results)

      tryCatch({
        # Update the UI elements based on comparison results
        output$comparison_summary <- renderText({
          req(rv$comparison_results)

          # Summarize key differences in coding approaches
          differences <- rv$comparison_results$pattern_comparison

          # Format coverage differences
          coverage_text <- format_coverage_differences(differences$coverage_differences)

          # Format code application differences
          code_text <- format_code_differences(differences$code_differences)

          # Format overlap pattern differences
          overlap_text <- format_overlap_differences(differences$combination_differences)

          # Format sequence differences
          sequence_text <- format_sequence_differences(differences$sequence_differences)

          # Combine all text
          paste0(
            "Qualitative Comparison Summary\n",
            "===========================\n\n",
            "Coverage Patterns:\n", coverage_text, "\n\n",
            "Code Application Patterns:\n", code_text, "\n\n",
            "Code Overlap Patterns:\n", overlap_text, "\n\n",
            "Sequence Patterns:\n", sequence_text
          )
        })

        output$comparison_plot <- renderPlot({
          req(input$plot_type)
          if (is.null(rv$comparison_results$plots[[input$plot_type]])) {
            plot(0, 0, type = "n",
                 main = "No data to display",
                 xlab = "", ylab = "")
          } else {
            print(rv$comparison_results$plots[[input$plot_type]])
          }
        })
      }, error = function(e) {
        showNotification(paste("Error updating comparison results:", e$message),
                         type = "error")
      })
    })

    # Render comparison summary
    output$comparison_summary <- renderText({
      req(rv$comparison_results)

      # Summarize key differences in coding approaches
      differences <- rv$comparison_results$pattern_comparison

      # Format coverage differences
      coverage_text <- format_coverage_differences(differences$coverage_differences)

      # Format code application differences
      code_text <- format_code_differences(differences$code_differences)

      # Format overlap pattern differences
      overlap_text <- format_overlap_differences(differences$combination_differences)

      # Format sequence differences
      sequence_text <- format_sequence_differences(differences$sequence_differences)

      # Combine all text
      paste0(
        "Qualitative Comparison Summary\n",
        "===========================\n\n",
        "Coverage Patterns:\n", coverage_text, "\n\n",
        "Code Application Patterns:\n", code_text, "\n\n",
        "Code Overlap Patterns:\n", overlap_text, "\n\n",
        "Sequence Patterns:\n", sequence_text
      )
    })

    # Render visualizations
    output$comparison_plot <- renderPlot({
      req(rv$comparison_results, input$plot_type)

      # Get the appropriate plot based on selected type
      plot_result <- NULL
      if (!is.null(rv$comparison_results$plots)) {
        plot_result <- switch(input$plot_type,
                              "distribution" = rv$comparison_results$plots$distribution,
                              "overlap" = rv$comparison_results$plots$overlap,
                              "sequence" = rv$comparison_results$plots$sequence)
      }

      # If no plot is available, show empty plot with message
      if (is.null(plot_result)) {
        plot.new()
        title(main = "No data available for selected visualization")
      } else {
        print(plot_result)
      }
    }, height = function() {
      # Dynamically adjust height based on number of coders
      if (!is.null(rv$comparison_results)) {
        n_coders <- length(rv$comparison_results$coding_strategies)
        return(200 * n_coders)  # 200 pixels per coder
      }
      return(400)  # Default height
    })

    # Update the detailed analysis output
    output$coverage_details <- renderText({
      req(rv$comparison_results)

      coverage <- rv$comparison_results$comparison$coverage_differences
      paste0(
        "Total Codes Range: ", paste(coverage$total_codes_range, collapse=" - "), "\n",
        "Unique Codes Range: ", paste(coverage$unique_codes_range, collapse=" - ")
      )
    })

    output$application_details <- renderText({
      req(rv$comparison_results)

      codes <- rv$comparison_results$comparison$code_differences
      paste0(
        "Shared Codes: ", paste(codes$shared_codes, collapse=", "), "\n",
        "Usage Patterns:\n",
        apply(codes$usage_matrix, 1, function(row) {
          paste0("  ", names(row)[1], ": ", paste(row, collapse=" vs "), "\n")
        })
      )
    })

    output$pattern_details <- renderText({
      req(rv$comparison_results)

      overlaps <- rv$comparison_results$comparison$overlap_differences
      paste0(
        "Total Overlaps Range: ", paste(overlaps$total_overlaps_range, collapse=" - "), "\n",
        "Unique Pairs Range: ", paste(overlaps$unique_pairs_range, collapse=" - ")
      )
    })

    # Update plot type choices
    observeEvent(input$comparison_metrics, {
      req(rv$comparison_results)

      # Update available plot types based on selected metrics
      plot_choices <- list(
        "Code Distribution" = "distribution",
        "Code Overlaps" = "overlap",
        "Code Sequences" = "sequence"
      )

      updateSelectInput(session, "plot_type",
                        choices = plot_choices)
    })

  }

  runApp(shinyApp(ui, server))
}

#' @importFrom utils write.csv packageVersion
#' @importFrom stats runif
#' @importFrom grDevices rgb rainbow recordPlot
#' @importFrom graphics par barplot plot points text lines
#' @importFrom shiny runApp shinyApp fluidPage actionButton observeEvent renderUI
#'   showNotification showModal modalDialog removeModal updateTextAreaInput
#'   updateTextInput tabPanel fileInput renderTable renderPlot plotOutput
#'   tableOutput textInput textAreaInput selectInput checkboxGroupInput
#'   tags icon reactive reactiveValues isolate req
#' @importFrom shinyjs useShinyjs toggle runjs
#' @importFrom data.tree Node as.Node
#' @importFrom jsonlite fromJSON toJSON write_json
#' @importFrom shinydashboard dashboardPage dashboardHeader dashboardSidebar
#'   dashboardBody tabBox box
#' @importFrom DT renderDT datatable DTOutput
#' @importFrom readtext readtext
#' @importFrom tools R_user_dir
#' @importFrom shinyFiles shinyFileChoose shinyFilesButton shinyDirButton shinyDirChoose parseDirPath parseFilePaths
NULL

#' \%||\% operator
#'
#' @name grapes-or-or-grapes
#' @aliases %||%
#' @title Null coalescing operator
#' @description Provides null coalescing functionality, returning the first non-NULL argument
#' @param a First value to check
#' @param b Second value (default) to use if first is NULL
#' @return Returns \code{a} if not NULL, otherwise returns \code{b}
#' @keywords internal
`%||%` <- function(a, b) if (!is.null(a)) a else b

#' Handle errors with custom messages
#'
#' @description
#' Provides error handling with customizable success, error, and completion messages.
#' Wraps expressions in a tryCatch block and displays appropriate notifications.
#'
#' @param expr Expression to evaluate
#' @param success_msg Optional character string for success notification
#' @param error_msg Optional character string for error notification
#' @param finally_msg Optional character string for completion notification
#'
#' @return Result of the expression or NULL if error occurs
#'
#' @importFrom shiny showNotification
#' @keywords internal
handle_error <- function(expr, success_msg = NULL, error_msg = NULL, finally_msg = NULL) {
  is_shiny <- requireNamespace("shiny", quietly = TRUE) &&
    exists("session") &&
    !is.null(get0("session"))

  notify <- function(msg, type = "message") {
    if (is_shiny) {
      shiny::showNotification(msg, type = type)
    } else {
      message(msg)
    }
  }

  tryCatch({
    result <- expr
    if (!is.null(success_msg)) {
      notify(success_msg, "message")
    }
    return(result)
  }, error = function(e) {
    msg <- if (is.null(error_msg)) paste("Error:", e$message) else error_msg
    notify(msg, "error")
    return(NULL)
  }, finally = {
    if (!is.null(finally_msg)) {
      notify(finally_msg, "message")
    }
  })
}

#' Display interactive dialog for saving annotated text
#'
#' @description
#' Creates and displays a modal dialog that allows users to save their annotated text
#' in either HTML or plain text format. Provides options for filename and directory selection.
#'
#' @param rv ReactiveValues object containing the application state
#' @param input Shiny input object
#' @param session Shiny session object
#' @param volumes List of available storage volumes for directory selection
#'
#' @return Invisible NULL, called for side effects
#'
#' @importFrom shiny showModal modalDialog textInput selectInput modalButton actionButton
#' @importFrom shiny verbatimTextOutput
#' @keywords internal
save_annotated_text_interactive <- function(rv, input, session, volumes) {
  showModal(modalDialog(
    title = "Save Annotated Text",
    textInput("save_filename", "Enter filename:"),
    selectInput("save_format", "Select file format:",
                choices = c("HTML" = "html", "Text File" = "txt")),
    div(style = "margin: 10px 0;",
        shinyDirButton("text_directory_select",
                       label = "Choose Directory",
                       title = "Select Directory to Save Text")
    ),
    verbatimTextOutput("selected_text_dir"),
    footer = tagList(
      modalButton("Cancel"),
      actionButton("confirm_save_annotations", "Save")
    )
  ))
}

#' Display interactive dialog for saving annotation records
#'
#' @description
#' Creates and displays a modal dialog that allows users to save their annotation records
#' in either CSV or JSON format. Provides options for filename and directory selection.
#'
#' @param rv ReactiveValues object containing the application state
#' @param input Shiny input object
#' @param session Shiny session object
#' @param volumes List of available storage volumes for directory selection
#'
#' @return Invisible NULL, called for side effects
#'
#' @importFrom shiny showModal modalDialog textInput selectInput modalButton actionButton
#' @importFrom shiny verbatimTextOutput
#' @keywords internal
save_records_interactive <- function(rv, input, session, volumes) {
  showModal(modalDialog(
    title = "Save Records",
    textInput("save_filename", "Enter filename:"),
    selectInput("save_format", "Select file format:",
                choices = c("CSV" = "csv", "JSON" = "json")),
    div(style = "margin: 10px 0;",
        shinyDirButton("records_directory_select",
                       label = "Choose Directory",
                       title = "Select Directory to Save Records")
    ),
    verbatimTextOutput("selected_records_dir"),
    footer = tagList(
      modalButton("Cancel"),
      actionButton("confirm_save_records", "Save")
    )
  ))
}

#' Get available storage volumes on Windows
#'
#' @description
#' Creates a closure that returns a named vector of available Windows drive letters
#' and their corresponding paths. Checks for the existence of drives from A: to Z:
#' (excluding C: which is handled separately).
#'
#' @return Function that returns named character vector of available drives
#'
#' @importFrom stats setNames
#' @keywords internal
getVolumes <- function() {
  function() {
    volumes <- c("C:" = "C:/")
    for (letter in LETTERS[-3]) {
      drive <- paste0(letter, ":/")
      if (dir.exists(drive)) {
        volumes <- c(volumes, stats::setNames(drive, toupper(paste0(letter, ":"))))
      }
    }
    return(volumes)
  }
}

#' Create and manage undo/redo action
#'
#' @description
#' Creates an action object for the undo/redo system, containing information about
#' the type of action, the data involved, and how to reverse the action.
#'
#' @param type Character string specifying the type of action
#' @param data List containing the action data
#' @param reverse_data Optional list containing data for reversing the action
#'
#' @return List containing:
#'   \itemize{
#'     \item type: Action type identifier
#'     \item data: Action data
#'     \item reverse_data: Data for reversing the action
#'     \item timestamp: Time the action was created
#'   }
#'
#' @keywords internal
create_action <- function(type, data, reverse_data = NULL) {
  list(
    type = type,
    data = data,
    reverse_data = reverse_data,
    timestamp = Sys.time()
  )
}

#' Add action to history
#'
#' @param rv Reactive values object
#' @param action Action to add
#' @keywords internal
add_action <- function(rv, action) {
  # Remove any future actions if we're not at the end
  if (rv$action_index < length(rv$action_history)) {
    rv$action_history <- rv$action_history[1:rv$action_index]
  }

  # Add the new action
  rv$action_history[[rv$action_index + 1]] <- action
  rv$action_index <- rv$action_index + 1
}

#' Apply or reverse an action (Updated version)
#'
#' @description
#' Applies or reverses an action in the undo/redo system. Handles different types of
#' actions including adding/removing annotations and merging/unmerging codes.
#'
#' @param rv ReactiveValues object containing application state
#' @param action List containing action information
#' @param reverse Logical indicating whether to reverse the action
#'
#' @return Invisible rv (ReactiveValues object)
#'
#' @keywords internal
apply_action <- function(rv, action, reverse = FALSE) {
  data <- if (reverse) action$reverse_data else action$data

  switch(action$type,
         "add_annotation" = {
           if (reverse) {
             # Remove annotation
             if(nrow(rv$annotations) > 0) {
               rv$annotations <- rv$annotations[-which(
                 rv$annotations$start == data$start &
                   rv$annotations$end == data$end &
                   rv$annotations$code == data$code
               ), ]
             }
           } else {
             # Add annotation
             if(is.null(rv$annotations)) {
               rv$annotations <- data.frame(
                 start = integer(),
                 end = integer(),
                 text = character(),
                 code = character(),
                 memo = character(),
                 source_file = character(),
                 stringsAsFactors = FALSE
               )
             }

             # Ensure data has source_file column
             if (!"source_file" %in% colnames(data)) {
               data$source_file <- rv$current_file_name %||% "Unknown"
             }

             rv$annotations <- rbind(rv$annotations, data)
           }
         },
         "merge_codes" = {
           if (reverse) {
             # Reverse merge: split merged code back into original codes
             new_code <- data$new_code
             old_codes <- data$old_codes
             old_colors <- data$old_colors

             # Update annotations back to original codes
             # For simplicity, we'll assign all instances to the first old code
             # (A more sophisticated approach would track which annotations had which original codes)
             rv$annotations$code[rv$annotations$code == new_code] <- old_codes[1]

             # Update codes list
             rv$codes <- unique(c(setdiff(rv$codes, new_code), old_codes))

             # Restore old colors
             if (!is.null(old_colors)) {
               for (code in names(old_colors)) {
                 rv$code_colors[code] <- old_colors[[code]]
               }
             }

             # Remove merged code color
             rv$code_colors <- rv$code_colors[names(rv$code_colors) != new_code]

           } else {
             # Apply merge: combine old codes into new code
             old_codes <- data$old_codes
             new_code <- data$new_code

             # Update all annotations that use old codes
             for (old_code in old_codes) {
               rv$annotations$code[rv$annotations$code == old_code] <- new_code
             }

             # Update codes list
             rv$codes <- unique(c(setdiff(rv$codes, old_codes), new_code))

             # Assign color to new code if not already assigned
             if (!(new_code %in% names(rv$code_colors))) {
               used_colors <- as.character(rv$code_colors)
               new_color <- get_next_palette_color(used_colors)
               rv$code_colors[new_code] <- new_color
             }

             # Remove old code colors
             rv$code_colors <- rv$code_colors[!names(rv$code_colors) %in% old_codes]
           }
         })

  invisible(rv)
}

#' Get all code names from hierarchy
#'
#' @description
#' Recursively extracts all code names from a code hierarchy tree structure,
#' traversing through all nodes and collecting their names.
#'
#' @param node Root node of the code hierarchy (data.tree Node object)
#'
#' @return Character vector containing all code names in the hierarchy
#' @keywords internal
get_code_names <- function(node) {
  if (node$isLeaf) {
    return(node$name)
  } else {
    return(c(node$name, unlist(lapply(node$children, get_code_names))))
  }
}

#' Save and manage project state
#'
#' @description
#' Saves the current state of a text annotation project, including annotations,
#' codes, and memos. Creates necessary directories and handles file operations
#' safely.
#'
#' @param state List containing project components:
#'   \itemize{
#'     \item text: Original text content
#'     \item annotations: Data frame of annotations
#'     \item codes: Vector of code names
#'     \item code_tree: Hierarchical organization of codes
#'     \item code_colors: Color assignments for codes
#'     \item memos: List of annotation memos
#'   }
#' @param filename Character string specifying the output file name
#'
#' @return Invisible NULL, called for side effect of saving project state
#'
#' @importFrom tools file_path_sans_ext
#'
#' @keywords internal
save_project_state <- function(state, filename) {
  # Create the projects directory if it doesn't exist
  project_dir <- get_project_dir()
  if (is.null(project_dir)) return(invisible(NULL))

  # Add .rds extension if not present
  if (!grepl("\\.rds$", filename)) {
    filename <- paste0(filename, ".rds")
  }

  # Clean the path and get full filepath
  filepath <- file.path(project_dir, basename(filename))

  # Add version information
  state$version <- utils::packageVersion("textAnnotatoR")

  # Save state to RDS file
  handle_error(
    expr = saveRDS(state, file = filepath),
    success_msg = paste("Project saved successfully to", filepath),
    error_msg = "Failed to save project"
  )

  invisible(NULL)
}

#' Load project state from file
#'
#' @description
#' Loads a previously saved project state from an RDS file. Performs version checking
#' and data structure validation during the loading process.
#'
#' @param filename Character string specifying the filename to load
#'
#' @return List containing the loaded project state, or NULL if loading fails
#'
#' @importFrom data.tree as.Node
#' @importFrom utils packageVersion
#' @keywords internal
load_project_state <- function(filename) {
  # Add .rds extension if not present
  if (!grepl("\\.rds$", filename)) {
    filename <- paste0(filename, ".rds")
  }

  # Get the projects directory and create full filepath
  project_dir <- get_project_dir()
  filepath <- file.path(project_dir, basename(filename))

  if (!file.exists(filepath)) {
    showNotification(paste("Project file not found:", filepath), type = "error")
    return(NULL)
  }

  handle_error(
    expr = {
      state <- readRDS(filepath)

      # Version check
      current_version <- utils::packageVersion("textAnnotatoR")
      if (!is.null(state$version) && state$version > current_version) {
        warning("Project was created with a newer version of textAnnotatoR")
      }

      # Convert list back to data.tree object if necessary
      if (!is.null(state$code_tree) && !inherits(state$code_tree, "Node")) {
        state$code_tree <- as.Node(state$code_tree)
      }

      return(state)
    },
    error_msg = paste("Failed to load project from", filepath)
  )
}

#' Load project from a selected recent project
#'
#' @description
#' Loads a project from the list of recent projects
#'
#' @param rv ReactiveValues object containing application state
#' @param input Shiny input object
#' @param session Shiny session object
#' @keywords internal
load_selected_project <- function(rv, input, session) {
  # Determine the filepath based on storage mode
  if (!is.null(rv$storage_mode) && rv$storage_mode != "project") {
    project_dir <- get_project_dir(rv)
    filepath <- file.path(project_dir, paste0(input$project_to_load, ".rds"))
  } else {
    # For project-specific mode, this shouldn't be called directly
    showNotification("Cannot load from recent projects in project-specific mode", type = "warning")
    return()
  }

  # Use the existing load_project_from_path function
  load_project_from_path(rv, filepath, session)
}

#' Load project from filepath
#'
#' @description
#' Loads a project from a specified filepath
#'
#' @param rv ReactiveValues object containing application state
#' @param filepath Path to the project file
#' @param session Shiny session object
#' @keywords internal
load_project_from_path <- function(rv, filepath, session) {
  tryCatch({
    project_state <- readRDS(filepath)

    # Check if this is a project-specific storage project
    if (!is.null(project_state$project_dir)) {
      # Update storage mode for this session
      rv$storage_mode <- "project"
      rv$project_specific_dir <- project_state$project_dir
    }

    # Update all reactive values with loaded state
    rv$text <- project_state$text
    rv$annotations <- project_state$annotations
    rv$codes <- project_state$codes
    rv$code_tree <- project_state$code_tree
    rv$code_colors <- project_state$code_colors
    rv$memos <- project_state$memos
    rv$code_descriptions <- project_state$code_descriptions
    rv$history <- project_state$history
    rv$history_index <- project_state$history_index
    rv$current_project <- basename(tools::file_path_sans_ext(filepath))
    rv$current_project_path <- filepath
    rv$project_modified <- FALSE  # Reset modified flag after successful load

    # Update UI elements
    updateTextAreaInput(session, "text_input", value = rv$text)
    session$sendCustomMessage("clearSelection", list())

    showNotification("Project loaded successfully", type = "message")
    removeModal()
  }, error = function(e) {
    showNotification(paste("Error loading project:", e$message), type = "error")
  })
}

#' Update text display with annotations
#'
#' @description
#' Creates an HTML representation of the text with annotations, highlighting codes
#' with their assigned colors and preserving line breaks.
#'
#' @param rv ReactiveValues object containing text and annotations
#' @return HTML string containing the formatted text with annotations
#' @keywords internal
update_text_display <- function(rv) {
  # First, convert newlines in the original text to HTML <br> tags
  # We need to do this before processing character by character
  text_with_linebreaks <- gsub("\n", "<br>", rv$text)

  if (nrow(rv$annotations) == 0) {
    # If no annotations, still need to preserve line breaks
    # First split by <br> tags we just added
    text_parts <- strsplit(text_with_linebreaks, "<br>")[[1]]
    result <- character(length(text_parts))

    # Process each line separately
    for (i in seq_along(text_parts)) {
      line_chars <- strsplit(text_parts[i], "")[[1]]
      # Calculate the character offset for this line
      char_offset <- ifelse(i == 1, 0, sum(nchar(text_parts[1:(i-1)])) + (i-1))
      char_indices <- char_offset + 1:length(line_chars)

      # Create spans for each character with correct indices
      result[i] <- paste0("<span class='char' id='char_", char_indices, "'>",
                          line_chars, "</span>", collapse = "")
    }

    # Join the lines with <br> tags
    return(paste(result, collapse = "<br>"))
  }

  # For text with annotations, we need a more complex approach
  sorted_annotations <- rv$annotations[order(rv$annotations$start), ]
  displayed_text <- ""
  last_end <- 0

  for (i in 1:nrow(sorted_annotations)) {
    # Handle text before current annotation
    if (sorted_annotations$start[i] > last_end + 1) {
      before_text <- substr(rv$text, last_end + 1, sorted_annotations$start[i] - 1)
      # Replace newlines with <br> tags in this segment
      before_text <- gsub("\n", "<br>", before_text)

      # Split by <br> and process each part separately
      before_parts <- strsplit(before_text, "<br>")[[1]]
      before_result <- character(length(before_parts))

      for (j in seq_along(before_parts)) {
        if (before_parts[j] == "") next

        part_chars <- strsplit(before_parts[j], "")[[1]]
        # Calculate character offset
        char_offset <- ifelse(j == 1, last_end,
                              last_end + sum(nchar(before_parts[1:(j-1)])) + (j-1))
        char_indices <- char_offset + 1:length(part_chars)

        before_result[j] <- paste0("<span class='char' id='char_", char_indices, "'>",
                                   part_chars, "</span>", collapse = "")
      }

      displayed_text <- paste0(displayed_text, paste(before_result, collapse = "<br>"))
    }

    # Handle the annotated text
    code_color <- rv$code_colors[sorted_annotations$code[i]]
    if (is.null(code_color)) {
      code_color <- "#CCCCCC"  # Default color if not found
    }

    # Get the annotated text and handle line breaks
    annotated_text <- substr(rv$text, sorted_annotations$start[i], sorted_annotations$end[i])
    # Replace newlines with <br> tags
    annotated_text <- gsub("\n", "<br>", annotated_text)

    # Split by <br> and process each part
    annotated_parts <- strsplit(annotated_text, "<br>")[[1]]
    annotated_result <- character(length(annotated_parts))

    for (j in seq_along(annotated_parts)) {
      if (annotated_parts[j] == "") {
        annotated_result[j] <- ""
        next
      }

      part_chars <- strsplit(annotated_parts[j], "")[[1]]
      # Calculate character offset
      char_offset <- sorted_annotations$start[i] - 1
      if (j > 1) {
        char_offset <- char_offset + sum(nchar(annotated_parts[1:(j-1)])) + (j-1)
      }
      char_indices <- char_offset + 1:length(part_chars)

      annotated_result[j] <- paste0("<span class='char' id='char_", char_indices, "'>",
                                    part_chars, "</span>", collapse = "")
    }

    annotated_spans <- paste(annotated_result, collapse = "<br>")

    displayed_text <- paste0(displayed_text,
                             "<span class='code-display' style='background-color: ", code_color,
                             ";' data-code='", sorted_annotations$code[i],
                             "' data-start='", sorted_annotations$start[i],
                             "' data-end='", sorted_annotations$end[i], "'>",
                             "[", sorted_annotations$code[i], "]",
                             annotated_spans,
                             "</span>")

    last_end <- sorted_annotations$end[i]
  }

  # Handle text after the last annotation
  if (last_end < nchar(rv$text)) {
    after_text <- substr(rv$text, last_end + 1, nchar(rv$text))
    # Replace newlines with <br> tags
    after_text <- gsub("\n", "<br>", after_text)

    # Split by <br> and process each part
    after_parts <- strsplit(after_text, "<br>")[[1]]
    after_result <- character(length(after_parts))

    for (j in seq_along(after_parts)) {
      if (after_parts[j] == "") next

      part_chars <- strsplit(after_parts[j], "")[[1]]
      # Calculate character offset
      char_offset <- last_end
      if (j > 1) {
        char_offset <- char_offset + sum(nchar(after_parts[1:(j-1)])) + (j-1)
      }
      char_indices <- char_offset + 1:length(part_chars)

      after_result[j] <- paste0("<span class='char' id='char_", char_indices, "'>",
                                part_chars, "</span>", collapse = "")
    }

    displayed_text <- paste0(displayed_text, paste(after_result, collapse = "<br>"))
  }

  return(displayed_text)
}

#' Concatenate memo texts
#'
#' @description
#' Combines existing and new memo texts with proper separators,
#' handling empty memos appropriately.
#'
#' @param existing_memo Character string containing current memo text
#' @param new_memo Character string containing memo text to append
#'
#' @return Character string of combined memo text
#' @keywords internal
concatenate_memos <- function(existing_memo, new_memo) {
  if (existing_memo == "") {
    return(new_memo)
  } else {
    return(paste(existing_memo, new_memo, sep = "; "))
  }
}

#' Save annotated text as HTML document
#'
#' @description
#' Creates an HTML document containing the annotated text with proper styling
#' for code highlights and formatting, preserving line breaks.
#'
#' @param filename Character string specifying output file path
#' @param rv ReactiveValues object containing:
#'   \itemize{
#'     \item text: Original text content
#'     \item annotations: Data frame of annotations
#'     \item code_colors: Named character vector of code colors
#'   }
#'
#' @return Invisible NULL, called for side effect
#' @keywords internal
save_as_html <- function(filename, rv) {
  # Get the current state of the text display
  html_content <- update_text_display(rv)

  # Create a complete HTML document
  full_html <- paste0(
    "<!DOCTYPE html>\n<html>\n<head>\n",
    "<style>\n",
    ".code-display { padding: 2px 5px; margin-right: 5px; border-radius: 3px; font-weight: bold; color: black; display: inline-block; }\n",
    "#annotated_text { white-space: pre-wrap; line-height: 1.5; }\n",
    ".code-display br { display: block; content: \"\"; margin-top: 0.5em; }\n",
    "</style>\n",
    "</head>\n<body>\n",
    "<h1>Annotated Text</h1>\n",
    "<div id='annotated_text'>\n",
    html_content,
    "\n</div>\n",
    "</body>\n</html>"
  )

  # Write the HTML content to a file
  writeLines(full_html, filename)
}

#' Save annotated text as plain text
#'
#' @description
#' Creates a plain text file containing the annotated text with code markers.
#'
#' @param filename Character string specifying output file path
#' @param rv ReactiveValues object containing:
#'   \itemize{
#'     \item text: Original text content
#'     \item annotations: Data frame of annotations
#'   }
#'
#' @return Invisible NULL, called for side effect
#' @keywords internal
save_as_text <- function(filename, rv) {
  # Get the annotated text
  annotated_text <- create_plain_text_annotations(rv$text, rv$annotations)

  # Write the content to a file
  writeLines(annotated_text, filename)
}

#' Create plain text version of annotations
#'
#' @description
#' Converts annotated text to plain text format with code markers. Each annotation
#' is represented as a code identifier and annotated text wrapped in square brackets.
#' Multiple annotations are preserved and shown in order of appearance in the text.
#'
#' @param text Character string containing the original text
#' @param annotations Data frame of annotations with columns:
#'   \itemize{
#'     \item start: Numeric vector of starting positions
#'     \item end: Numeric vector of ending positions
#'     \item code: Character vector of code names
#'   }
#'
#' @return Character string containing formatted text with code markers
#' @keywords internal
create_plain_text_annotations <- function(text, annotations) {
  if (nrow(annotations) == 0) {
    return(text)
  }

  sorted_annotations <- annotations[order(annotations$start), ]
  plain_text <- ""
  last_end <- 0

  for (i in 1:nrow(sorted_annotations)) {
    if (sorted_annotations$start[i] > last_end + 1) {
      # Extract text before current annotation, preserving line breaks
      plain_text <- paste0(plain_text, substr(text, last_end + 1, sorted_annotations$start[i] - 1))
    }

    # Extract the annotated text segment, preserving line breaks
    annotated_segment <- substr(text, sorted_annotations$start[i], sorted_annotations$end[i])

    plain_text <- paste0(plain_text,
                         "[", sorted_annotations$code[i], ": ",
                         annotated_segment,
                         "]")
    last_end <- sorted_annotations$end[i]
  }

  if (last_end < nchar(text)) {
    # Extract text after last annotation, preserving line breaks
    plain_text <- paste0(plain_text, substr(text, last_end + 1, nchar(text)))
  }

  return(plain_text)
}

#' Initialize new project
#'
#' @description
#' Creates new project by resetting all reactive values to defaults
#' and clearing UI elements.
#'
#' @param rv ReactiveValues object to reset containing:
#'   \itemize{
#'     \item text: Text content
#'     \item annotations: Annotation data frame
#'     \item codes: Vector of codes
#'     \item code_tree: Hierarchy Node object
#'     \item All other project state values
#'   }
#' @param session Shiny session object
#'
#' @return Invisible NULL, called for side effect
#' @keywords internal
create_new_project <- function(rv, session) {
  rv$text <- ""
  rv$annotations <- data.frame(
    start = integer(),
    end = integer(),
    text = character(),
    code = character(),
    memo = character(),
    source_file = character(),  # Include source_file column
    stringsAsFactors = FALSE
  )
  rv$codes <- character()
  rv$code_tree <- Node$new("Root")
  rv$code_colors <- character()
  rv$memos <- list()
  rv$code_descriptions <- list()
  rv$history <- list(list(text = "", annotations = data.frame()))
  rv$history_index <- 1
  rv$current_project <- NULL
  rv$current_file_name <- NULL  # Reset current file name
  rv$project_modified <- FALSE
  rv$action_history <- list()
  rv$action_index <- 0
  rv$merged_codes <- list()

  # Clear UI elements
  updateTextAreaInput(session, "text_input", value = "")
  session$sendCustomMessage("clearSelection", list())

  showNotification("New project created", type = "message")
}

#' Generate code frequency visualization
#'
#' @description
#' Creates a barplot visualization showing the frequency of each code in the annotations.
#' The plot displays codes on the x-axis and their frequency counts on the y-axis.
#'
#' @param annotations Data frame containing text annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @return A recordedplot object containing the code frequency visualization
#'
#' @importFrom graphics par barplot
#' @importFrom grDevices recordPlot
#' @keywords internal
generate_code_frequency_plot <- function(annotations) {
  # Save current par settings and restore on exit
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar))

  code_freq <- table(annotations$code)
  code_freq_sorted <- sort(code_freq, decreasing = TRUE)

  par(mar = c(8, 4, 2, 2))
  barplot(code_freq_sorted,
          main = "Code Frequency",
          xlab = "",
          ylab = "Frequency",
          col = "steelblue",
          las = 2)
  return(recordPlot())
}

#' Generate code co-occurrence statistics and visualization
#'
#' @description
#' Performs a comprehensive analysis of code co-occurrences in the text, including
#' calculation of various similarity metrics and generation of network and heatmap
#' visualizations.
#'
#' @param annotations Data frame containing text annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#' @param text Character string containing the original text (optional)
#' @param unit Character string specifying the analytical unit: "sentence", "paragraph", or "document"
#'
#' @return List containing:
#'   \itemize{
#'     \item co_occurrence: Matrix of raw co-occurrence counts
#'     \item jaccard_similarity: Matrix of Jaccard similarity coefficients
#'     \item phi_coefficient: Matrix of Phi coefficients
#'     \item network_plot: Network visualization of code relationships
#'     \item heatmap_plot: Heatmap visualization of code co-occurrences
#'     \item summary: List of summary statistics
#'   }
#'
#' @importFrom graphics par plot points text lines image axis
#' @importFrom grDevices rgb colorRampPalette recordPlot
#' @importFrom stats cor
#' @keywords internal
generate_code_co_occurrence_analysis <- function(annotations, text = NULL, unit = "paragraph") {
  # Validate input parameters
  unit <- match.arg(unit, choices = c("sentence", "paragraph", "document"))

  if (is.null(annotations) || nrow(annotations) == 0) {
    return(create_empty_cooccurrence_result(unit))
  }

  # Get unique codes
  codes <- unique(annotations$code)
  n_codes <- length(codes)

  if (n_codes <= 1) {
    return(create_empty_cooccurrence_result(unit))
  }

  # Parse text into analytical units if text is provided
  if (!is.null(text) && nchar(text) > 0) {
    text_units <- parse_text_into_units(text, unit)
    unit_assignments <- assign_annotations_to_units(annotations, text_units)
  } else {
    # Fallback: use position-based units if no text provided
    unit_assignments <- assign_annotations_to_position_units(annotations, unit)
  }

  # Calculate co-occurrence matrices
  co_matrix <- calculate_unit_cooccurrence_matrix(codes, unit_assignments)
  jaccard_matrix <- calculate_jaccard_similarity_matrix(codes, unit_assignments)
  phi_matrix <- calculate_phi_coefficient_matrix(codes, unit_assignments)

  # Generate visualizations
  network_plot <- generate_network_visualization(co_matrix, jaccard_matrix, phi_matrix, codes)
  heatmap_plot <- generate_heatmap_visualization(jaccard_matrix, codes)

  # Calculate summary statistics
  summary_stats <- calculate_cooccurrence_summary(co_matrix, jaccard_matrix, phi_matrix, codes)

  # Add unit information
  unit_info <- list(
    analytical_unit = unit,
    total_units = length(unique(unit_assignments$unit_id)),
    codes_per_unit = calculate_codes_per_unit_stats(unit_assignments)
  )

  return(list(
    co_occurrence = co_matrix,
    jaccard_similarity = jaccard_matrix,
    phi_coefficient = phi_matrix,
    network_plot = network_plot,
    heatmap_plot = heatmap_plot,
    summary = summary_stats,
    unit_info = unit_info
  ))
}

# Helper function to create empty result structure
create_empty_cooccurrence_result <- function(unit) {
  empty_plot <- function() {
    plot(0, 0, type = "n", xlim = c(0, 1), ylim = c(0, 1),
         main = "No co-occurrence data available", xlab = "", ylab = "")
    recordPlot()
  }

  list(
    co_occurrence = matrix(0, nrow = 0, ncol = 0),
    jaccard_similarity = matrix(0, nrow = 0, ncol = 0),
    phi_coefficient = matrix(0, nrow = 0, ncol = 0),
    network_plot = empty_plot(),
    heatmap_plot = empty_plot(),
    summary = list(
      total_codes = 0,
      max_co_occurrence = 0,
      max_jaccard = 0,
      mean_jaccard = 0,
      significant_pairs = 0
    ),
    unit_info = list(
      analytical_unit = unit,
      total_units = 0,
      codes_per_unit = list(
        mean_codes_per_unit = 0,
        max_codes_per_unit = 0,
        units_with_multiple_codes = 0
      )
    )
  )
}

#' Generate word cloud visualization
#'
#' @description
#' Creates a simple word cloud visualization from the input text, showing the most
#' frequent words with size proportional to their frequency.
#'
#' @param text Character string containing the text to visualize
#'
#' @return A plot object containing the word cloud visualization
#'
#' @importFrom graphics plot text
#'
#' @keywords internal
generate_word_cloud <- function(text) {
  words <- unlist(strsplit(tolower(text), "\\W+"))
  word_freq <- sort(table(words[nchar(words) > 3]), decreasing = TRUE)
  word_freq <- word_freq[1:min(100, length(word_freq))]

  plot(1, type = "n", xlab = "", ylab = "", xlim = c(0, 1), ylim = c(0, 1),
       main = "Word Cloud")

  n <- length(word_freq)
  angles <- runif(n, 0, 2 * pi)
  x <- 0.5 + 0.4 * cos(angles)
  y <- 0.5 + 0.4 * sin(angles)

  sizes <- 1 + 3 * (word_freq - min(word_freq)) / (max(word_freq) - min(word_freq))
  text(x, y, labels = names(word_freq), cex = sizes,
       col = rainbow(n, s = 0.7, v = 0.7))

  return(recordPlot())
}

#' Generate text summary statistics
#'
#' @description
#' Calculates basic summary statistics for the annotated text, including word counts,
#' character counts, annotation counts, and unique code counts.
#'
#' @param text Character string containing the text being analyzed
#' @param annotations Data frame of annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @return A list containing summary statistics:
#'   \itemize{
#'     \item total_words: total number of words in the text
#'     \item total_characters: total number of characters
#'     \item total_sentences: number of sentences (approximated by punctuation)
#'     \item total_paragraphs: number of paragraphs (non-empty lines)
#'     \item total_annotations: number of annotations
#'     \item unique_codes: number of unique codes used
#'   }
#'
#' @keywords internal
generate_text_summary <- function(text, annotations) {
  # Count paragraphs (sequences separated by blank lines)
  paragraphs <- strsplit(text, "\n")[[1]]
  # Count actual paragraphs (non-empty lines)
  paragraph_count <- sum(nzchar(trimws(paragraphs)))

  list(
    total_words = length(unlist(strsplit(text, "\\W+"))),
    total_characters = nchar(text),
    total_sentences = length(unlist(strsplit(text, "[.!?]+\\s+"))),
    total_paragraphs = paragraph_count,
    total_annotations = nrow(annotations),
    unique_codes = length(unique(annotations$code))
  )
}

#' Generate network visualization for code co-occurrence
#'
#' @description
#' Creates a network plot showing relationships between codes based on co-occurrence
#' patterns. Node sizes reflect total co-occurrences, edge thickness shows Jaccard
#' similarity, and edge opacity indicates phi coefficient magnitude.
#'
#' @param co_matrix Matrix of raw co-occurrence counts between codes
#' @param jaccard_matrix Matrix of Jaccard similarity coefficients
#' @param phi_matrix Matrix of phi coefficients
#' @param codes Character vector of code names
#'
#' @return A recordedplot object containing the network visualization
#'
#' @importFrom graphics plot points text segments par
#' @importFrom grDevices recordPlot rgb
#' @keywords internal
generate_network_visualization <- function(co_matrix, jaccard_matrix, phi_matrix, codes) {
  # Save current par settings and restore on exit
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar))

  if (length(codes) < 2) {
    plot(0, 0, type = "n", xlim = c(0, 1), ylim = c(0, 1),
         main = "Network Visualization",
         xlab = "Insufficient codes for network visualization",
         ylab = "")
    return(recordPlot())
  }

  # Calculate node positions in a circle
  n_codes <- length(codes)
  angles <- seq(0, 2 * pi, length.out = n_codes + 1)[1:n_codes]
  x_pos <- 0.5 + 0.3 * cos(angles)
  y_pos <- 0.5 + 0.3 * sin(angles)

  # Calculate node sizes based on total co-occurrences
  node_totals <- rowSums(co_matrix, na.rm = TRUE)
  max_total <- max(node_totals, na.rm = TRUE)
  if (max_total == 0) max_total <- 1  # Avoid division by zero
  node_sizes <- 0.5 + 2 * (node_totals / max_total)  # Scale between 0.5 and 2.5

  # Set up plot with adjusted margins to remove space for legend
  par(mar = c(2, 2, 3, 2))
  plot(0, 0, type = "n", xlim = c(0, 1), ylim = c(0, 1),
       main = "Code Co-occurrence Network",
       xlab = "", ylab = "", axes = FALSE)

  # Draw edges for significant relationships
  max_jaccard <- max(jaccard_matrix, na.rm = TRUE)
  if (max_jaccard > 0) {
    for (i in 1:(n_codes-1)) {
      for (j in (i+1):n_codes) {
        jaccard_sim <- jaccard_matrix[i, j]
        phi_coeff <- abs(phi_matrix[i, j])

        # Only draw edges for meaningful relationships
        if (!is.na(jaccard_sim) && jaccard_sim > 0.1) {
          # Edge thickness based on Jaccard similarity
          edge_width <- 1 + 3 * (jaccard_sim / max_jaccard)

          # Edge opacity based on phi coefficient magnitude
          edge_alpha <- max(0.3, min(1, phi_coeff * 2))
          edge_color <- rgb(0.5, 0.5, 0.5, alpha = edge_alpha)

          segments(x_pos[i], y_pos[i], x_pos[j], y_pos[j],
                   lwd = edge_width, col = edge_color)
        }
      }
    }
  }

  # Draw nodes
  for (i in 1:n_codes) {
    points(x_pos[i], y_pos[i], pch = 19, cex = node_sizes[i],
           col = "lightblue", lwd = 2)
    points(x_pos[i], y_pos[i], pch = 1, cex = node_sizes[i],
           col = "darkblue", lwd = 2)
  }

  # Add labels
  for (i in 1:n_codes) {
    # Position labels outside the nodes
    label_offset <- 0.08
    label_x <- x_pos[i] + label_offset * cos(angles[i])
    label_y <- y_pos[i] + label_offset * sin(angles[i])

    text(label_x, label_y, codes[i], cex = 0.8,
         adj = c(0.5, 0.5), font = 2)
  }

  # Legend removed as requested

  return(recordPlot())
}

#' Generate heatmap visualization for code co-occurrence
#'
#' @description
#' Creates a heatmap showing the strength of relationships between codes based on
#' Jaccard similarity coefficients. Darker colors indicate stronger relationships.
#'
#' @param jaccard_matrix Matrix of Jaccard similarity coefficients between codes
#' @param codes Character vector of code names
#'
#' @return A recordedplot object containing the heatmap visualization
#'
#' @importFrom graphics image axis par
#' @importFrom grDevices colorRampPalette recordPlot
#' @keywords internal
generate_heatmap_visualization <- function(jaccard_matrix, codes) {
  # Save current par settings and restore on exit
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar))

  if (length(codes) < 2) {
    plot(0, 0, type = "n", xlim = c(0, 1), ylim = c(0, 1),
         main = "Co-occurrence Heatmap",
         xlab = "Insufficient codes for heatmap",
         ylab = "")
    return(recordPlot())
  }

  # Create color palette
  colors <- colorRampPalette(c("white", "lightblue", "blue", "darkblue"))(100)

  # Set up margins with more space on the right for compact legend
  par(mar = c(8, 8, 4, 6))

  # Create the heatmap
  image(1:ncol(jaccard_matrix), 1:nrow(jaccard_matrix),
        t(jaccard_matrix),
        col = colors,
        main = "Code Co-occurrence Heatmap\n(Jaccard Similarity)",
        xlab = "", ylab = "", axes = FALSE)

  # Add axes with code names
  axis(1, at = 1:length(codes), labels = codes, las = 2, cex.axis = 0.8)
  axis(2, at = 1:length(codes), labels = codes, las = 2, cex.axis = 0.8)

  # Add grid lines
  abline(h = (1:length(codes)) + 0.5, col = "lightgray", lwd = 0.5)
  abline(v = (1:length(codes)) + 0.5, col = "lightgray", lwd = 0.5)

  # Add text values in cells for small matrices
  if (length(codes) <= 8) {
    for (i in 1:nrow(jaccard_matrix)) {
      for (j in 1:ncol(jaccard_matrix)) {
        if (!is.na(jaccard_matrix[i, j]) && jaccard_matrix[i, j] > 0) {
          text(j, i, sprintf("%.2f", jaccard_matrix[i, j]),
               cex = 0.7, col = ifelse(jaccard_matrix[i, j] > 0.5, "white", "black"))
        }
      }
    }
  }

  # Add a more compact color scale legend
  # Create a vertical color bar in the right margin
  legend_x <- par("usr")[2] + 0.15 * (par("usr")[2] - par("usr")[1])
  legend_y_seq <- seq(par("usr")[3], par("usr")[4], length.out = 5)

  # Create legend with smaller, more compact design
  legend_values <- c("1.0", "0.75", "0.5", "0.25", "0.0")
  legend_colors <- colors[c(100, 75, 50, 25, 1)]

  # Add a compact legend that doesn't overlap with the heatmap
  par(xpd = TRUE)  # Allow drawing outside plot region

  # Create legend box coordinates
  legend_width <- 0.03 * (par("usr")[2] - par("usr")[1])
  legend_height <- 0.6 * (par("usr")[4] - par("usr")[3])
  legend_x_start <- par("usr")[2] + 0.05 * (par("usr")[2] - par("usr")[1])
  legend_y_start <- par("usr")[3] + 0.2 * (par("usr")[4] - par("usr")[3])

  # Draw color boxes for legend
  legend_box_height <- legend_height / 5
  for (i in 1:5) {
    y_bottom <- legend_y_start + (i - 1) * legend_box_height
    y_top <- y_bottom + legend_box_height
    rect(legend_x_start, y_bottom, legend_x_start + legend_width, y_top,
         col = legend_colors[6 - i], border = "black", lwd = 0.5)

    # Add text labels
    text(legend_x_start + legend_width + 0.01 * (par("usr")[2] - par("usr")[1]),
         (y_bottom + y_top) / 2,
         legend_values[6 - i],
         cex = 0.6, adj = 0)
  }

  # Add legend title
  text(legend_x_start + legend_width / 2,
       legend_y_start + legend_height + 0.05 * (par("usr")[4] - par("usr")[3]),
       "Jaccard\nSimilarity",
       cex = 0.7, adj = 0.5, font = 2)

  par(xpd = FALSE)  # Reset to default

  return(recordPlot())
}

#' Add theme to code hierarchy
#'
#' @description
#' Adds a new theme to the code hierarchy tree. Themes can be used to organize and
#' group related codes in a hierarchical structure.
#'
#' @param node Root node of the hierarchy tree
#' @param theme_name Character string specifying the name of the new theme
#' @param description Optional character string providing a description of the theme
#'
#' @return Updated node with new theme added
#'
#' @importFrom data.tree Node
#' @keywords internal
add_theme <- function(node, theme_name, description = "") {
  # Check if theme already exists
  if (!is.null(node$children[[theme_name]])) {
    stop("Theme already exists")
  }

  # Create new theme node
  new_theme <- node$AddChild(theme_name)
  new_theme$description <- description
  new_theme$type <- "theme"
  new_theme$created <- Sys.time()

  return(node)
}

#' Process comparison file
#'
#' @description
#' Processes uploaded comparison files, handling different file formats (CSV, JSON)
#' and ensuring proper data structure and types for comparison analysis.
#'
#' @param filepath Character string specifying the path to the comparison file
#'
#' @return Data frame containing processed annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @importFrom jsonlite fromJSON
#' @importFrom utils read.csv
#' @keywords internal
process_comparison_file <- function(filepath) {
  ext <- tolower(tools::file_ext(filepath))

  df <- if (ext == "csv") {
    read.csv(filepath, stringsAsFactors = FALSE)
  } else if (ext == "json") {
    fromJSON(filepath)
  } else {
    stop("Unsupported file format")
  }

  # If the file is from Records tab (has text and memo columns), reformat it
  if (all(c("start", "end", "text", "code", "memo") %in% colnames(df))) {
    df <- df[, c("start", "end", "code")]
  }

  # Check required columns
  required_cols <- c("start", "end", "code")
  if (!all(required_cols %in% colnames(df))) {
    missing_cols <- setdiff(required_cols, colnames(df))
    stop(paste("Missing required columns:", paste(missing_cols, collapse = ", ")))
  }

  # Convert columns to proper types
  df$start <- as.numeric(as.character(df$start))
  df$end <- as.numeric(as.character(df$end))
  df$code <- as.character(df$code)

  # Remove rows with NA values
  df <- df[complete.cases(df[, required_cols]), ]

  if (nrow(df) == 0) {
    stop("No valid annotations found after cleaning")
  }

  return(df)
}

#' Compare coding patterns between different documents or coders
#'
#' @description
#' Performs a comprehensive comparison of coding patterns between different sets of
#' annotations, analyzing differences in coverage, code application, overlaps, and
#' code sequences.
#'
#' @param annotations_list A list of data frames, where each data frame contains
#'        annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @return A list containing comparison results and analysis:
#'   \itemize{
#'     \item coding_strategies: list of analyzed coding patterns for each input
#'     \item comparison: list of comparative analyses between coding patterns
#'   }
#'
#' @keywords internal
generate_comparison_analysis <- function(annotations_list) {
  if (!is.list(annotations_list) || length(annotations_list) < 2) {
    stop("Need at least two annotation sets for comparison")
  }

  # Process each annotation set with error handling
  coding_strategies <- lapply(annotations_list, function(annotations) {
    tryCatch({
      if (!is.data.frame(annotations)) {
        stop("Invalid data format: input must be a data frame")
      }

      # Validate and clean data
      required_cols <- c("start", "end", "code")
      if (!all(required_cols %in% colnames(annotations))) {
        stop(paste("Missing required columns:",
                   paste(setdiff(required_cols, colnames(annotations)), collapse = ", ")))
      }

      # Ensure proper types and handle NA values
      annotations$start <- as.numeric(as.character(annotations$start))
      annotations$end <- as.numeric(as.character(annotations$end))
      annotations$code <- as.character(annotations$code)

      # Remove invalid rows
      valid_rows <- !is.na(annotations$start) &
        !is.na(annotations$end) &
        !is.na(annotations$code) &
        annotations$start <= annotations$end

      if (sum(valid_rows) == 0) {
        stop("No valid annotations found after cleaning")
      }

      annotations <- annotations[valid_rows, ]

      # Calculate coverage statistics with error handling
      coverage <- tryCatch({
        list(
          distribution = list(
            frequencies = table(annotations$code)
          )
        )
      }, error = function(e) {
        list(distribution = list(frequencies = table(character(0))))
      })

      # Calculate co-occurrence statistics with error handling
      co_occurrences <- tryCatch({
        list(
          combinations = list(
            frequencies = calculate_co_occurrences(annotations)
          )
        )
      }, error = function(e) {
        list(combinations = list(frequencies = table(character(0))))
      })

      # Calculate sequence statistics with error handling
      sequences <- tryCatch({
        list(
          transitions = calculate_transitions(annotations[order(annotations$start), ])
        )
      }, error = function(e) {
        list(transitions = list())
      })

      return(list(
        coverage = coverage,
        co_occurrences = co_occurrences,
        sequences = sequences
      ))
    }, error = function(e) {
      # Return empty results if processing fails
      list(
        coverage = list(distribution = list(frequencies = table(character(0)))),
        co_occurrences = list(combinations = list(frequencies = table(character(0)))),
        sequences = list(transitions = list())
      )
    })
  })

  # Calculate comparison metrics with error handling
  comparison <- tryCatch({
    list(
      coverage_differences = compare_coverage(coding_strategies),
      code_differences = compare_codes(coding_strategies),
      overlap_differences = compare_overlaps(coding_strategies)
    )
  }, error = function(e) {
    list(
      coverage_differences = "Error calculating differences",
      code_differences = "Error comparing codes",
      overlap_differences = "Error analyzing overlaps"
    )
  })

  return(list(
    coding_strategies = coding_strategies,
    comparison = comparison
  ))
}

#' Calculate code co-occurrences in annotations
#'
#' @description
#' Analyzes text annotations to identify and count instances where different codes
#' overlap or co-occur in the same text regions. Handles edge cases and provides
#' error-safe operation.
#'
#' @param annotations Data frame containing annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code identifier
#'   }
#'
#' @return Table object containing frequencies of code pairs that co-occur,
#'         with code pair names as "code1 & code2"
#'
#' @details
#' Co-occurrences are identified by finding overlapping text regions between
#' different code annotations. The function sorts annotations by position and
#' checks for overlaps between each pair of annotations.
#'
#' @keywords internal
calculate_co_occurrences <- function(annotations) {
  if (!is.data.frame(annotations) || nrow(annotations) <= 1) {
    return(table(character(0)))
  }

  tryCatch({
    co_occurrences <- c()

    # Sort annotations by start position
    annotations <- annotations[order(annotations$start), ]

    # Find overlapping annotations
    for (i in 1:(nrow(annotations)-1)) {
      for (j in (i+1):nrow(annotations)) {
        if (annotations$start[j] <= annotations$end[i]) {
          pair <- sort(c(annotations$code[i], annotations$code[j]))
          co_occurrences <- c(co_occurrences, paste(pair, collapse=" & "))
        } else {
          break  # No more overlaps possible with current i
        }
      }
    }

    return(table(co_occurrences))
  }, error = function(e) {
    return(table(character(0)))
  })
}

#' Calculate transitions between consecutive codes
#'
#' @description
#' Analyzes the sequence of code applications to identify transitions between
#' consecutive codes in the text. Creates a list of code pairs representing
#' each transition from one code to another.
#'
#' @param annotations Data frame containing annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code identifier
#'   }
#'
#' @return List where each element is a named vector containing:
#'   \itemize{
#'     \item from: Character string of the source code
#'     \item to: Character string of the target code
#'   }
#'
#' @details
#' Transitions are identified by sorting annotations by position and then
#' analyzing consecutive pairs of codes. The function handles edge cases
#' and provides error-safe operation.
#'
#' @keywords internal
calculate_transitions <- function(annotations) {
  if (!is.data.frame(annotations) || nrow(annotations) <= 1) {
    return(list())
  }

  tryCatch({
    # Sort annotations by start position
    annotations <- annotations[order(annotations$start), ]

    # Create transitions list
    transitions <- vector("list", nrow(annotations) - 1)
    for (i in 1:(nrow(annotations)-1)) {
      transitions[[i]] <- c(
        from = annotations$code[i],
        to = annotations$code[i+1]
      )
    }

    return(transitions)
  }, error = function(e) {
    return(list())
  })
}

#' Compare code usage between coders
#'
#' @description
#' Compares how different coders use codes by analyzing shared codes and their
#' usage patterns across coding strategies.
#'
#' @param coding_strategies List of coding strategies, where each strategy contains
#'        code frequency information
#'
#' @return List containing:
#'   \itemize{
#'     \item shared_codes: Character vector of codes used across strategies
#'     \item usage_matrix: Matrix showing code usage across strategies
#'   }
#' @keywords internal
compare_codes <- function(coding_strategies) {
  tryCatch({
    all_codes <- unique(unlist(lapply(coding_strategies, function(strategy) {
      names(strategy$coverage$distribution$frequencies)
    })))

    if (length(all_codes) == 0) {
      return(list(
        shared_codes = character(0),
        usage_matrix = matrix(0, nrow = 0, ncol = 0)
      ))
    }

    code_usage <- sapply(coding_strategies, function(strategy) {
      freqs <- strategy$coverage$distribution$frequencies
      sapply(all_codes, function(code) {
        if (code %in% names(freqs)) freqs[code] else 0
      })
    })

    return(list(
      shared_codes = all_codes,
      usage_matrix = code_usage
    ))
  }, error = function(e) {
    return(list(
      shared_codes = character(0),
      usage_matrix = matrix(0, nrow = 0, ncol = 0)
    ))
  })
}

#' Compare overlap patterns between coders
#'
#' @description
#' Analyzes how different coders overlap in their code applications by comparing
#' overlap patterns and frequencies across coding strategies.
#'
#' @param coding_strategies List of coding strategies, where each strategy contains
#'        overlap information
#'
#' @return List containing:
#'   \itemize{
#'     \item total_overlaps_range: Range of total overlaps across strategies
#'     \item unique_pairs_range: Range of unique code pairs across strategies
#'   }
#' @keywords internal
compare_overlaps <- function(coding_strategies) {
  tryCatch({
    overlap_stats <- lapply(coding_strategies, function(strategy) {
      freqs <- strategy$co_occurrences$combinations$frequencies
      list(
        total_overlaps = if(length(freqs) > 0) sum(freqs) else 0,
        unique_pairs = length(freqs)
      )
    })

    total_overlaps <- sapply(overlap_stats, `[[`, "total_overlaps")
    unique_pairs <- sapply(overlap_stats, `[[`, "unique_pairs")

    return(list(
      total_overlaps_range = if(length(total_overlaps) > 0) range(total_overlaps) else c(0, 0),
      unique_pairs_range = if(length(unique_pairs) > 0) range(unique_pairs) else c(0, 0)
    ))
  }, error = function(e) {
    return(list(
      total_overlaps_range = c(0, 0),
      unique_pairs_range = c(0, 0)
    ))
  })
}


#' Analyze coverage patterns in annotations
#'
#' @description
#' Analyzes how codes are distributed throughout the text, including clustering
#' patterns and coding density.
#'
#' @param annotations Data frame containing text annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item clusters: List of annotation clusters
#'     \item density: List containing overall density metrics
#'     \item distribution: List containing code frequencies and positions
#'   }
#'
#' @keywords internal
analyze_coverage <- function(annotations) {
  if (!is.data.frame(annotations) || nrow(annotations) == 0) {
    return(list(
      clusters = list(),
      density = list(overall_density = 0),
      distribution = list(
        frequencies = integer(0),
        positions = list()
      )
    ))
  }

  # Ensure numeric values and handle NAs
  annotations$start <- as.numeric(as.character(annotations$start))
  annotations$end <- as.numeric(as.character(annotations$end))
  annotations$code <- as.character(annotations$code)

  valid_rows <- !is.na(annotations$start) &
    !is.na(annotations$end) &
    !is.na(annotations$code) &
    annotations$start <= annotations$end

  annotations <- annotations[valid_rows, ]

  if (nrow(annotations) == 0) {
    return(list(
      clusters = list(),
      density = list(overall_density = 0),
      distribution = list(
        frequencies = integer(0),
        positions = list()
      )
    ))
  }

  # Sort annotations by position
  sorted_anns <- annotations[order(annotations$start), ]

  # Calculate code frequencies
  code_freq <- table(sorted_anns$code)

  # Calculate code positions with error handling
  code_pos <- tryCatch({
    tapply(sorted_anns$start, sorted_anns$code, function(x) list(positions = x))
  }, error = function(e) {
    list()
  })

  # Calculate density with error handling
  total_length <- max(sorted_anns$end) - min(sorted_anns$start)
  total_coded <- sum(sorted_anns$end - sorted_anns$start + 1)
  density <- if (total_length > 0) total_coded / total_length else 0

  return(list(
    clusters = find_annotation_clusters(sorted_anns),
    density = list(overall_density = density),
    distribution = list(
      frequencies = code_freq,
      positions = code_pos
    )
  ))
}

#' Analyze code application patterns
#'
#' @description
#' Analyzes patterns in how codes are applied in the annotations.
#'
#' @param annotations Data frame containing code annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position
#'     \item end: numeric, ending position
#'     \item code: character, code identifier
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item patterns: List of code patterns
#'     \item summary: Summary statistics
#'   }
#'
#' @keywords internal
analyze_code_patterns <- function(annotations) {
  if (!is.data.frame(annotations) || nrow(annotations) == 0) {
    return(list(
      patterns = list(),
      summary = list(total_codes = 0)
    ))
  }

  # Group annotations by code with error handling
  code_groups <- tryCatch({
    split(annotations, annotations$code)
  }, error = function(e) {
    list()
  })

  # Analyze patterns for each code
  code_patterns <- lapply(code_groups, function(code_anns) {
    tryCatch({
      lengths <- code_anns$end - code_anns$start + 1
      list(
        typical_length = mean(lengths, na.rm = TRUE),
        length_variation = stats::sd(lengths, na.rm = TRUE),
        code_count = nrow(code_anns)
      )
    }, error = function(e) {
      list(
        typical_length = 0,
        length_variation = 0,
        code_count = 0
      )
    })
  })

  return(list(
    patterns = code_patterns,
    summary = list(
      total_codes = length(code_patterns),
      unique_codes = length(unique(annotations$code))
    )
  ))
}

#' Analyze code co-occurrence patterns
#'
#' @description
#' Analyzes how different codes co-occur within the annotated text by examining overlapping
#' annotations and calculating various metrics of co-occurrence strength.
#'
#' @param annotations A data frame containing text annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @return A list containing co-occurrence analysis results:
#'   \itemize{
#'     \item combinations: list containing frequency table of code co-occurrences
#'     \item characteristics: list with average overlap length and total overlap count
#'   }
#'
#' @keywords internal
analyze_co_occurrences <- function(annotations) {
  if (!is.data.frame(annotations) || nrow(annotations) <= 1) {
    return(list(
      combinations = list(frequencies = integer(0)),
      characteristics = list(avg_length = 0, total_overlaps = 0)
    ))
  }

  # Find overlapping annotations with error handling
  overlaps <- find_overlapping_codes(annotations)

  # Analyze overlap patterns
  if (length(overlaps) > 0) {
    combinations <- tryCatch({
      table(sapply(overlaps, function(x) {
        if (!is.null(x$code1) && !is.null(x$code2)) {
          paste(sort(c(x$code1, x$code2)), collapse = "-")
        } else {
          NA
        }
      }))
    }, error = function(e) {
      table(character(0))
    })

    lengths <- sapply(overlaps, function(x) {
      if (!is.null(x$overlap_start) && !is.null(x$overlap_end)) {
        x$overlap_end - x$overlap_start + 1
      } else {
        NA
      }
    })

    avg_length <- mean(lengths, na.rm = TRUE)
    if (is.nan(avg_length)) avg_length <- 0
  } else {
    combinations <- table(character(0))
    avg_length <- 0
  }

  return(list(
    combinations = list(frequencies = combinations),
    characteristics = list(
      avg_length = avg_length,
      total_overlaps = length(overlaps)
    )
  ))
}

#' Analyze sequences and transitions between codes
#'
#' @description
#' Analyzes how codes are sequenced in the text by examining transitions
#' between consecutive codes and identifying repeated patterns.
#'
#' @param annotations Data frame of text annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item transitions: List of transitions between consecutive codes
#'     \item patterns: List of identified repeated code sequences
#'   }
#' @keywords internal
analyze_sequences <- function(annotations) {
  if (!is.data.frame(annotations) || nrow(annotations) <= 1) {
    return(list(
      transitions = list(),
      patterns = list()
    ))
  }

  # Ensure proper types
  annotations$start <- as.numeric(annotations$start)
  annotations$end <- as.numeric(annotations$end)
  annotations$code <- as.character(annotations$code)

  # Remove any rows with NA values
  valid_rows <- stats::complete.cases(annotations[, c("start", "end", "code")])
  annotations <- annotations[valid_rows, ]

  if (nrow(annotations) <= 1) {
    return(list(
      transitions = list(),
      patterns = list()
    ))
  }

  # Sort annotations by position
  sorted_anns <- annotations[order(annotations$start), ]

  # Find transitions
  transitions <- list()
  for (i in 1:(nrow(sorted_anns)-1)) {
    transitions <- c(transitions,
                     list(c(from = sorted_anns$code[i],
                            to = sorted_anns$code[i+1])))
  }

  return(list(
    transitions = transitions,
    patterns = find_repeated_sequences(sorted_anns)
  ))
}

#' Compare coding patterns between different coders
#'
#' @description
#' Analyzes and compares coding patterns between different coders by examining
#' various aspects including coverage, code application patterns, combinations,
#' and sequences.
#'
#' @param coding_strategies List of coding strategies, where each strategy contains:
#'   \itemize{
#'     \item coverage: List containing density and distribution information
#'     \item code_patterns: List of code application patterns
#'     \item combinations: List of code combination patterns
#'     \item sequences: List of code sequence patterns
#'   }
#'
#' @return List containing comparison results:
#'   \itemize{
#'     \item coverage_differences: Analysis of coding density variations
#'     \item code_differences: Analysis of code application differences
#'     \item combination_differences: Analysis of code combination patterns
#'     \item sequence_differences: Analysis of code sequence patterns
#'   }
#'
#' @details
#' The function performs multiple comparisons with error handling for each aspect
#' of coding patterns. Returns descriptive messages when analysis cannot be
#' performed due to insufficient data.
#'
#' @keywords internal
compare_patterns <- function(coding_strategies) {
  if (length(coding_strategies) < 2) {
    return(list(
      coverage_differences = "Insufficient data for comparison",
      code_differences = "Insufficient data for comparison",
      combination_differences = "Insufficient data for comparison",
      sequence_differences = "Insufficient data for comparison"
    ))
  }

  # Compare coverage patterns
  coverage_diff <- tryCatch({
    # Extract densities safely
    densities <- sapply(coding_strategies, function(x) {
      if (!is.null(x$coverage$density$overall_density)) {
        x$coverage$density$overall_density
      } else {
        0
      }
    })

    list(
      density_variation = diff(range(densities)),
      density_summary = sprintf("Coverage density varies from %.2f to %.2f",
                                min(densities), max(densities))
    )
  }, error = function(e) {
    list(
      density_variation = 0,
      density_summary = "Unable to calculate coverage differences"
    )
  })

  # Compare code application patterns
  code_diff <- tryCatch({
    patterns <- lapply(coding_strategies, function(x) {
      if (!is.null(x$code_patterns) && length(x$code_patterns) > 0) {
        x$code_patterns
      } else {
        list()
      }
    })

    list(
      length_variation = "Analysis completed",
      pattern_summary = sprintf("Analyzed %d coding patterns", length(patterns))
    )
  }, error = function(e) {
    list(
      length_variation = "Unable to analyze patterns",
      pattern_summary = "Error in pattern analysis"
    )
  })

  return(list(
    coverage_differences = coverage_diff,
    code_differences = code_diff,
    combination_differences = "Comparison completed",
    sequence_differences = "Comparison completed"
  ))
}

#' Generate comparison visualizations
#'
#' @description
#' Creates a set of visualizations for comparing coding patterns between different
#' coders, including distribution comparisons, overlap patterns, and sequence patterns.
#'
#' @param comparison_results List containing results from generate_comparison_analysis:
#'   \itemize{
#'     \item coding_strategies: List of analyzed coding patterns
#'     \item comparison: List of comparative analyses
#'   }
#'
#' @return List containing plot objects:
#'   \itemize{
#'     \item distribution: Plot comparing code distribution patterns
#'     \item overlap: Plot showing code overlap patterns
#'     \item sequence: Plot displaying code sequence patterns
#'   }
#'
#' @importFrom graphics par barplot text title
#' @importFrom grDevices recordPlot
#' @keywords internal
generate_comparison_plots <- function(comparison_results) {
  if (is.null(comparison_results) || length(comparison_results$coding_strategies) < 2) {
    return(list(
      distribution = NULL,
      overlap = NULL,
      sequence = NULL
    ))
  }

  # Distribution comparison plot
  distribution_plot <- function() {
    # Save current par settings and restore on exit
    oldpar <- par(no.readonly = TRUE)
    on.exit(par(oldpar))

    # Calculate total height needed
    n_plots <- length(comparison_results$coding_strategies)

    # Set up the plotting area with adjusted margins
    par(mfrow = c(n_plots, 1),
        mar = c(3, 2, 2, 1),
        oma = c(2, 2, 1, 1))

    for (i in seq_along(comparison_results$coding_strategies)) {
      strategy <- comparison_results$coding_strategies[[i]]
      if (!is.null(strategy$coverage$distribution$frequencies)) {
        freqs <- strategy$coverage$distribution$frequencies
        bp <- barplot(freqs,
                      main = paste("Coder", i),
                      las = 2,
                      cex.names = 0.7,
                      cex.axis = 0.7,
                      col = "steelblue",
                      ylim = c(0, max(freqs) * 1.2))
        text(x = bp, y = freqs, labels = freqs, pos = 3, cex = 0.6)
      }
    }
    title(main = "Code Distribution Comparison",
          outer = TRUE,
          line = 0)
    recordPlot()
  }

  # Code overlap patterns plot
  overlap_plot <- function() {
    # Save current par settings and restore on exit
    oldpar <- par(no.readonly = TRUE)
    on.exit(par(oldpar))

    n_plots <- length(comparison_results$coding_strategies)

    par(mfrow = c(n_plots, 1),
        mar = c(3, 2, 2, 1),
        oma = c(2, 2, 1, 1))

    for (i in seq_along(comparison_results$coding_strategies)) {
      strategy <- comparison_results$coding_strategies[[i]]
      if (!is.null(strategy$co_occurrences$combinations$frequencies)) {
        freqs <- strategy$co_occurrences$combinations$frequencies
        if (length(freqs) > 0) {
          bp <- barplot(freqs,
                        main = paste("Coder", i),
                        las = 2,
                        cex.names = 0.7,
                        cex.axis = 0.7,
                        col = "lightgreen",
                        ylim = c(0, max(freqs) * 1.2))
          text(x = bp, y = freqs, labels = freqs, pos = 3, cex = 0.6)
        } else {
          plot.new()
          title(main = paste("Coder", i, "- No code co-occurrences"))
        }
      }
    }
    title(main = "Code Co-occurrence Comparison",
          outer = TRUE,
          line = 0)
    recordPlot()
  }

  # Sequence patterns plot
  sequence_plot <- function() {
    # Save current par settings and restore on exit
    oldpar <- par(no.readonly = TRUE)
    on.exit(par(oldpar))

    n_plots <- length(comparison_results$coding_strategies)

    par(mfrow = c(n_plots, 1),
        mar = c(3, 2, 2, 1),
        oma = c(2, 2, 1, 1))

    for (i in seq_along(comparison_results$coding_strategies)) {
      strategy <- comparison_results$coding_strategies[[i]]
      if (!is.null(strategy$sequences$transitions) &&
          length(strategy$sequences$transitions) > 0) {
        # Convert transitions to table
        trans_table <- table(sapply(strategy$sequences$transitions,
                                    function(x) paste(x["from"], "->", x["to"])))
        if (length(trans_table) > 0) {
          bp <- barplot(trans_table,
                        main = paste("Coder", i),
                        las = 2,
                        cex.names = 0.7,
                        cex.axis = 0.7,
                        col = "salmon",
                        ylim = c(0, max(trans_table) * 1.2))
          text(x = bp, y = trans_table, labels = trans_table, pos = 3, cex = 0.6)
        } else {
          plot.new()
          title(main = paste("Coder", i, "- No code sequences"))
        }
      }
    }
    title(main = "Code Sequence Comparison",
          outer = TRUE,
          line = 0)
    recordPlot()
  }

  # Generate and return all plots
  tryCatch({
    list(
      distribution = distribution_plot(),
      overlap = overlap_plot(),
      sequence = sequence_plot()
    )
  }, error = function(e) {
    warning("Error generating plots: ", e$message)
    list(
      distribution = NULL,
      overlap = NULL,
      sequence = NULL
    )
  })
}

#' Plot code distribution visualization
#'
#' @description
#' Creates a barplot visualization showing the distribution of codes in the annotations.
#' The plot includes rotated labels for better readability and handles empty or NULL
#' input data gracefully.
#'
#' @param distribution List containing code distribution information:
#'   \itemize{
#'     \item frequencies: Named numeric vector containing code frequencies
#'   }
#' @param main Character string specifying the plot title
#' @param ... Additional arguments passed to barplot()
#'
#' @return Invisible NULL, called for side effect of creating plot
#'
#' @importFrom graphics barplot text par
#' @importFrom grDevices recordPlot
#'
#' @keywords internal
plot_code_distribution <- function(distribution, main = "", ...) {
  if (is.null(distribution) || length(distribution$frequencies) == 0) {
    plot(0, 0, type = "n",
         main = main,
         xlab = "No distribution data available",
         ylab = "")
    return()
  }

  # Save current par settings and restore on exit
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar))

  # Create barplot with rotated labels
  bp <- barplot(distribution$frequencies,
                main = main,
                xlab = "",
                ylab = "Frequency",
                las = 2,
                cex.names = 0.8,
                ...)

  # Add labels below
  text(x = bp,
       y = par("usr")[3] - 0.1,
       labels = names(distribution$frequencies),
       xpd = TRUE,
       srt = 45,
       adj = 1,
       cex = 0.7)
}

#' Add code to theme in hierarchy
#'
#' @description
#' Adds a new code to a specific theme in the code hierarchy. The code can be added
#' to the root level or nested within existing themes.
#'
#' @param node Root node of the hierarchy tree
#' @param code_name Character string specifying the name of the code to add
#' @param theme_path Character vector specifying the path to the target theme
#' @param description Optional character string providing a description of the code
#'
#' @return Updated node with new code added
#'
#' @keywords internal
add_code_to_theme <- function(node, code_name, theme_path, description = "") {
  # If theme_path is empty, add to root
  if(length(theme_path) == 0) {
    new_code <- node$AddChild(code_name)
    new_code$description <- description
    new_code$type <- "code"
    new_code$created <- Sys.time()
    return(node)
  }

  # Navigate to the target theme
  current_node <- node
  for (theme in theme_path) {
    if (is.null(current_node$children[[theme]])) {
      stop(paste("Theme not found:", theme))
    }
    current_node <- current_node$children[[theme]]
  }

  # Add the code
  if (!is.null(current_node$children[[code_name]])) {
    stop("Code already exists in this theme")
  }

  new_code <- current_node$AddChild(code_name)
  new_code$description <- description
  new_code$type <- "code"
  new_code$created <- Sys.time()

  return(node)
}

#' Move item in code hierarchy
#'
#' @description
#' Moves a code or theme to a new location in the hierarchy while preserving its
#' properties and child nodes. Checks for circular references and maintains the
#' integrity of the hierarchy structure.
#'
#' @param node Root node of the hierarchy tree
#' @param item_path Character vector specifying the current path to the item
#' @param new_parent_path Character vector specifying the path to the new parent
#'
#' @return Updated node hierarchy with item moved to new location
#'
#' @keywords internal
move_item <- function(node, item_path, new_parent_path) {
  # Find the item to move
  item_node <- node$find(name = tail(item_path, 1),
                         filterFun = function(x) length(x$path) == length(item_path))

  if (is.null(item_node)) {
    stop("Item not found")
  }

  # Find the new parent
  new_parent <- node
  for (path_element in new_parent_path) {
    new_parent <- new_parent$children[[path_element]]
    if (is.null(new_parent)) {
      stop("New parent path not found")
    }
  }

  # Check for circular reference
  if (is_ancestor(item_node, new_parent)) {
    stop("Cannot move a node to its own descendant")
  }

  # Store item data
  item_data <- list(
    name = item_node$name,
    description = item_node$description,
    type = item_node$type,
    created = item_node$created,
    children = item_node$children
  )

  # Remove item from old location
  item_node$parent$RemoveChild(item_node$name)

  # Add item to new location
  new_item <- new_parent$AddChild(item_data$name)
  new_item$description <- item_data$description
  new_item$type <- item_data$type
  new_item$created <- item_data$created

  # Restore children if any
  if (length(item_data$children) > 0) {
    for (child in item_data$children) {
      restore_node(new_item, child)
    }
  }

  return(node)
}

#' Restore a node and its children in the hierarchy
#'
#' @description
#' Helper function to recursively restore a node and all its children
#' when moving items in the code hierarchy.
#'
#' @param parent Parent Node object where the node will be restored
#' @param node_data List containing node data to restore:
#'   \itemize{
#'     \item name: Character string of node name
#'     \item type: Character string specifying node type
#'     \item description: Character string of node description
#'     \item created: POSIXct creation timestamp
#'     \item children: List of child nodes
#'   }
#'
#' @return New Node object with restored data and children
#'
#' @importFrom data.tree Node
#' @keywords internal
restore_node <- function(parent, node_data) {
  new_node <- parent$AddChild(node_data$name)
  new_node$type <- node_data$type
  new_node$description <- node_data$description
  new_node$created <- node_data$created

  if (!is.null(node_data$children) && length(node_data$children) > 0) {
    for (child in node_data$children) {
      restore_node(new_node, child)
    }
  }

  return(new_node)
}

#' Check if one node is an ancestor of another
#'
#' @description
#' Helper function to check if a node is an ancestor of another node
#' in the code hierarchy, preventing circular references when moving items.
#'
#' @param potential_ancestor Node object to check as potential ancestor
#' @param node Node object to check ancestry against
#'
#' @return Logical indicating whether potential_ancestor is an ancestor of node
#' @keywords internal
is_ancestor <- function(potential_ancestor, node) {
  current <- node
  while (!is.null(current$parent)) {
    if (identical(current$parent, potential_ancestor)) {
      return(TRUE)
    }
    current <- current$parent
  }
  return(FALSE)
}

#' Export code hierarchy to JSON format
#'
#' @description
#' Converts the code hierarchy tree structure into a JSON string representation
#' that can be saved or transmitted while preserving all node properties and
#' relationships.
#'
#' @param node Root node of the hierarchy tree
#'
#' @return JSON string representation of the hierarchy
#'
#' @importFrom jsonlite toJSON
#'
#' @keywords internal
export_hierarchy <- function(node) {
  hierarchy_list <- as.list(node)
  toJSON(hierarchy_list, pretty = TRUE, auto_unbox = TRUE)
}

#' Import code hierarchy from JSON format
#'
#' @description
#' Reconstructs a code hierarchy tree structure from its JSON string representation,
#' restoring all node properties and relationships.
#'
#' @param json_string JSON string representation of the hierarchy
#'
#' @return Node object representing the reconstructed hierarchy
#'
#' @importFrom jsonlite fromJSON
#' @importFrom data.tree as.Node
#'
#' @keywords internal
import_hierarchy <- function(json_string) {
  hierarchy_list <- fromJSON(json_string)
  as.Node(hierarchy_list)
}

#' Generate visual representation of code hierarchy
#'
#' @description
#' Creates an HTML tree visualization of the code hierarchy with proper
#' indentation, icons, and interactive elements. Now also displays unassigned codes.
#'
#' @param node Root node of hierarchy tree with attributes
#' @param all_codes Character vector of all codes in the application
#'
#' @return Character string containing HTML markup for tree visualization
#' @keywords internal
visualize_hierarchy <- function(node, all_codes = NULL) {
  if (is.null(node)) return("Empty hierarchy")

  # Define icon HTML entities that work across browsers
  FOLDER_CLOSED_ICON <- "&#128193;" # Folder icon HTML entity
  FOLDER_OPEN_ICON <- "&#128194;"   # Open folder icon HTML entity
  FILE_ICON <- "&#128196;"          # File/document icon HTML entity
  CALENDAR_ICON <- "&#128197;"      # Calendar icon HTML entity

  print_tree <- function(node, indent = 0) {
    if (is.null(node)) return(character(0))

    # Get node type symbol using HTML entities
    symbol <- if (!is.null(node$type) && node$type == "theme")
      FOLDER_OPEN_ICON else FILE_ICON

    # Root node gets a special treatment
    if (node$name == "Root" && indent == 0) {
      symbol <- FOLDER_CLOSED_ICON
    }

    # Create the line for this node with proper data attributes and classes
    name_display <- if (!is.null(node$type)) {
      class_name <- if (node$type == "theme") "theme-item" else "code-item"
      sprintf('<span class="%s" data-name="%s" data-type="%s">%s</span>',
              class_name, node$name, node$type, node$name)
    } else {
      sprintf('<span>%s</span>', node$name)
    }

    # Add description preview if available
    description_preview <- if (!is.null(node$description) && node$description != "") {
      trimmed_desc <- substr(node$description, 1, 30)
      if (nchar(node$description) > 30) {
        trimmed_desc <- paste0(trimmed_desc, "...")
      }
      sprintf(' - <span class="description-preview">%s</span>', trimmed_desc)
    } else {
      ""
    }

    line <- paste0(
      paste(rep("  ", indent), collapse = ""),
      symbol, " ",
      name_display,
      description_preview
    )

    # Start with this node's line
    lines <- line

    # Add all children's lines, sorted alphabetically by type and name
    if (!is.null(node$children) && length(node$children) > 0) {
      # First sort themes, then codes - both alphabetically
      # First get theme children
      theme_children <- node$children[vapply(node$children, function(child) {
        !is.null(child$type) && child$type == "theme"
      }, logical(1))]

      # Sort theme children by name
      if (length(theme_children) > 0) {
        sorted_theme_names <- sort(names(theme_children))
        for (child_name in sorted_theme_names) {
          child_lines <- print_tree(node$children[[child_name]], indent + 1)
          lines <- c(lines, child_lines)
        }
      }

      # Then get code children
      code_children <- node$children[vapply(node$children, function(child) {
        !is.null(child$type) && child$type == "code"
      }, logical(1))]

      # Sort code children by name
      if (length(code_children) > 0) {
        sorted_code_names <- sort(names(code_children))
        for (child_name in sorted_code_names) {
          child_lines <- print_tree(node$children[[child_name]], indent + 1)
          lines <- c(lines, child_lines)
        }
      }
    }

    return(lines)
  }

  # Get all codes in the hierarchy
  codes_in_hierarchy <- character(0)
  collect_codes <- function(node) {
    if (is.null(node)) return()

    if (!is.null(node$type) && node$type == "code" && !is.null(node$name)) {
      codes_in_hierarchy <<- c(codes_in_hierarchy, node$name)
    }

    if (!is.null(node$children) && length(node$children) > 0) {
      children <- if (inherits(node$children, "Node")) {
        list(node$children)
      } else if (is.list(node$children)) {
        node$children
      } else {
        list()
      }

      for (child in children) {
        collect_codes(child)
      }
    }
  }

  collect_codes(node)

  # Start with the main hierarchy
  result <- print_tree(node)

  # Add unassigned codes section if we have all_codes information
  if (!is.null(all_codes) && length(all_codes) > 0) {
    unassigned_codes <- setdiff(all_codes, codes_in_hierarchy)

    if (length(unassigned_codes) > 0) {
      # Add a separator
      result <- c(result, "")

      # Add unassigned codes header using calendar icon HTML entity
      result <- c(result, paste0(CALENDAR_ICON, " <span class='theme-item' data-name='Unassigned Codes' data-type='theme'>Unassigned Codes</span> - Codes not in any theme"))

      # Add each unassigned code with file icon HTML entity
      sorted_unassigned <- sort(unassigned_codes)
      for (code in sorted_unassigned) {
        code_line <- paste0(
          "  ", FILE_ICON, " ",
          sprintf('<span class="code-item" data-name="%s" data-type="code">%s</span>',
                  code, code)
        )
        result <- c(result, code_line)
      }
    }
  }

  # Combine all lines and wrap in a div for proper styling
  paste(
    '<div class="hierarchy-container">',
    paste(result, collapse = "\n"),
    '</div>'
  )
}

#' Find node by name in a tree structure
#'
#' @description
#' Recursively searches through a tree structure to find a node with a specific name.
#' The search is performed depth-first and returns the first matching node found.
#'
#' @param node Node object representing the current position in the tree. Should have:
#'   \itemize{
#'     \item name: Character string identifier
#'     \item children: List of child nodes
#'   }
#' @param target_name Character string specifying the name to search for
#'
#' @return Node object if found, NULL otherwise
#'
#' @details
#' The function handles NULL inputs safely and performs a recursive depth-first
#' search through the tree structure. It checks node names and recursively
#' searches through child nodes.
#'
#' @keywords internal
find_node_by_name <- function(node, target_name) {
  if (is.null(node) || is.null(target_name)) return(NULL)

  if (!is.null(node$name) && node$name == target_name) {
    return(node)
  }

  if (!is.null(node$children)) {
    for (child in node$children) {
      result <- find_node_by_name(child, target_name)
      if (!is.null(result)) {
        return(result)
      }
    }
  }

  return(NULL)
}

#' Calculate hierarchy statistics
#'
#' @description
#' Calculates various statistics about the code hierarchy including the total number
#' of themes and codes, maximum depth, and distribution of codes across themes.
#' Now also accounts for codes not associated with any theme.
#'
#' @param node Root node of the hierarchy tree
#' @param all_codes Character vector of all codes in the application
#'
#' @return A list containing hierarchy statistics:
#'   \itemize{
#'     \item total_themes: Total number of themes in the hierarchy (excluding root)
#'     \item total_codes: Total number of codes in the hierarchy
#'     \item unassigned_codes: Number of codes not in the hierarchy
#'     \item max_depth: Maximum depth of the hierarchy tree
#'     \item codes_per_theme: List showing number of codes in each theme
#'     \item average_codes_per_theme: Average number of codes per theme
#'   }
#'
#' @keywords internal
calculate_hierarchy_stats <- function(node, all_codes = NULL) {
  if (is.null(node)) {
    return(list(
      total_themes = 0,
      total_codes = 0,
      unassigned_codes = length(all_codes),
      max_depth = 0,
      codes_per_theme = list(),
      average_codes_per_theme = 0
    ))
  }

  n_themes <- 0
  n_codes <- 0
  codes_in_hierarchy <- character(0)
  max_depth <- 0
  codes_per_theme <- list()

  traverse_node <- function(node, depth = 0) {
    if (is.null(node)) return()

    # Check if node type exists and is a character
    node_type <- if (!is.null(node$type) && is.character(node$type)) node$type else "unknown"

    # Only count themes that aren't the root node
    if (node_type == "theme" && !is.null(node$name) && node$name != "Root") {
      n_themes <<- n_themes + 1

      # Count codes that are direct children of this theme
      if (!is.null(node$children)) {
        # Safely get children as a list
        children <- if (inherits(node$children, "Node")) {
          list(node$children)
        } else if (is.list(node$children)) {
          node$children
        } else {
          list()
        }

        codes_in_theme <- vapply(children, function(x) {
          child_type <- if (!is.null(x$type) && is.character(x$type)) x$type else "unknown"
          child_type == "code"
        }, logical(1))

        theme_codes <- names(children)[codes_in_theme]
        codes_in_hierarchy <<- c(codes_in_hierarchy, theme_codes)

        count_codes_in_theme <- sum(codes_in_theme)
        if (count_codes_in_theme > 0) {
          codes_per_theme[[node$name]] <<- count_codes_in_theme
        }
      }
    } else if (node_type == "code") {
      n_codes <<- n_codes + 1
      if (!is.null(node$name)) {
        codes_in_hierarchy <<- c(codes_in_hierarchy, node$name)
      }
    }

    max_depth <<- max(max_depth, depth)

    # Safely traverse children
    if (!is.null(node$children)) {
      children <- if (inherits(node$children, "Node")) {
        list(node$children)
      } else if (is.list(node$children)) {
        node$children
      } else {
        list()
      }

      for (child in children) {
        traverse_node(child, depth + 1)
      }
    }
  }

  # Start traversal from root node
  traverse_node(node)

  # Check for unassigned codes
  unassigned_codes <- 0
  if (!is.null(all_codes)) {
    unassigned_codes <- length(setdiff(all_codes, codes_in_hierarchy))
  }

  # Calculate average codes per theme
  avg_codes <- if (n_themes > 0) n_codes / n_themes else 0

  # Return statistics
  list(
    total_themes = n_themes,
    total_codes = n_codes,
    unassigned_codes = unassigned_codes,
    max_depth = max_depth,
    codes_per_theme = codes_per_theme,
    average_codes_per_theme = avg_codes
  )
}

#' Find clusters of annotations in text
#'
#' @description
#' Identifies clusters of annotations that are close together in the text,
#' helping to identify dense coding regions.
#'
#' @param annotations Data frame containing sorted text annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @return List of annotation clusters, where each cluster contains annotations
#'         that are within a specified distance of each other
#'
#' @keywords internal
find_annotation_clusters <- function(annotations) {
  # Sort annotations by position
  sorted_anns <- annotations[order(annotations$start), ]

  # Find clusters where annotations are close together
  clusters <- list()
  current_cluster <- list()

  for (i in 1:(nrow(sorted_anns) - 1)) {
    if (nrow(sorted_anns) == 0) break

    current <- sorted_anns[i, ]
    next_ann <- sorted_anns[i + 1, ]

    # Add current annotation to cluster
    current_cluster <- append(current_cluster, list(current))

    # If gap to next annotation is large, start new cluster
    if ((next_ann$start - current$end) > 50) {  # Adjust threshold as needed
      if (length(current_cluster) > 0) {
        clusters <- append(clusters, list(current_cluster))
      }
      current_cluster <- list()
    }
  }

  # Add last cluster if exists
  if (length(current_cluster) > 0) {
    clusters <- append(clusters, list(current_cluster))
  }

  return(clusters)
}

#' Analyze coding density in text
#'
#' @description
#' Calculates metrics related to coding density in the text, including overall
#' density and identification of densely coded regions.
#'
#' @param annotations Data frame containing annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position
#'     \item end: numeric, ending position
#'     \item code: character, code identifier
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item overall_density: Numeric value representing proportion of text covered by codes
#'     \item dense_regions: List of vectors, each containing start and end positions
#'       of identified dense coding regions
#'   }
#'
#' @details
#' Density is calculated as the ratio of coded text to total text length.
#' Dense regions are identified where consecutive annotations are close together
#' (within 20 characters by default).
#'
#' @keywords internal
analyze_coding_density <- function(annotations) {
  if (nrow(annotations) == 0) return(list())

  # Calculate density metrics
  total_length <- max(annotations$end) - min(annotations$start)
  total_coded <- sum(annotations$end - annotations$start + 1)
  density <- total_coded / total_length

  # Identify dense regions
  dense_regions <- list()
  if (nrow(annotations) > 1) {
    for (i in 1:(nrow(annotations)-1)) {
      if ((annotations$start[i+1] - annotations$end[i]) < 20) {  # Adjust threshold
        dense_regions <- append(dense_regions,
                                list(c(annotations$start[i], annotations$end[i+1])))
      }
    }
  }

  return(list(
    overall_density = density,
    dense_regions = dense_regions
  ))
}

#' Analyze coding density across text
#'
#' @description
#' Analyzes the density of code applications across the text by calculating
#' overall density metrics and identifying regions of dense coding activity.
#'
#' @param annotations Data frame containing annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item overall_density: Numeric value representing the proportion of text covered by codes
#'     \item dense_regions: List of vector pairs indicating start and end positions of dense regions
#'   }
#'
#' @keywords internal
analyze_code_distribution <- function(annotations) {
  if (nrow(annotations) == 0) return(list())

  # Calculate distribution of codes across the text
  code_counts <- table(annotations$code)
  code_positions <- tapply(annotations$start, annotations$code,
                           function(x) list(positions = x))

  return(list(
    frequencies = code_counts,
    positions = code_positions
  ))
}

#' Analyze context around code applications
#'
#' @description
#' Examines the surrounding context where codes are applied by looking at
#' preceding and following annotations to understand code relationships.
#'
#' @param code_anns Data frame containing annotations for specific code:
#'   \itemize{
#'     \item start: numeric, starting position
#'     \item end: numeric, ending position
#'     \item code: character, code identifier
#'   }
#' @param all_anns Data frame containing all annotations in the text
#'
#' @return List of contexts for each code instance:
#'   \itemize{
#'     \item before: Preceding annotation if exists
#'     \item after: Following annotation if exists
#'   }
#'
#' @keywords internal
analyze_code_context <- function(code_anns, all_anns) {
  if (nrow(code_anns) == 0) return(list())

  contexts <- lapply(1:nrow(code_anns), function(i) {
    current <- code_anns[i, ]

    # Find preceding and following annotations
    preceding <- all_anns[all_anns$end < current$start, ]
    following <- all_anns[all_anns$start > current$end, ]

    # Get closest annotations
    before <- if (nrow(preceding) > 0) {
      preceding[which.max(preceding$end), ]
    } else NULL

    after <- if (nrow(following) > 0) {
      following[which.min(following$start), ]
    } else NULL

    list(
      before = before,
      after = after
    )
  })

  return(contexts)
}

#' Analyze memo usage patterns
#'
#' @description
#' Examines how memos are used with codes by analyzing memo frequency,
#' content, and patterns in memo application across code instances.
#'
#' @param code_anns Data frame containing code annotations with columns:
#'   \itemize{
#'     \item memo: character, memo text associated with annotation
#'     \item code: character, code identifier
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item memo_frequency: Proportion of annotations with memos
#'     \item has_memos: Logical vector indicating memo presence
#'   }
#'
#' @keywords internal
analyze_memo_patterns <- function(code_anns) {
  if (nrow(code_anns) == 0) return(list())

  # Extract and analyze memo patterns
  has_memo <- !is.na(code_anns$memo) & code_anns$memo != ""
  memo_count <- sum(has_memo)

  return(list(
    memo_frequency = memo_count / nrow(code_anns),
    has_memos = has_memo
  ))
}

#' Find overlapping code annotations
#'
#' @description
#' Identifies pairs of annotations that overlap in the text and returns their
#' intersection points and associated codes.
#'
#' @param annotations A data frame containing text annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position of annotation
#'     \item end: numeric, ending position of annotation
#'     \item code: character, code applied to the annotation
#'   }
#'
#' @return A list of overlapping code pairs, each containing:
#'   \itemize{
#'     \item code1: first code in the overlap
#'     \item code2: second code in the overlap
#'     \item overlap_start: starting position of overlap
#'     \item overlap_end: ending position of overlap
#'   }
#'
#' @keywords internal
find_overlapping_codes <- function(annotations) {
  if (!is.data.frame(annotations) || nrow(annotations) <= 1) {
    return(list())
  }

  # Ensure proper types and handle NAs
  annotations$start <- as.numeric(as.character(annotations$start))
  annotations$end <- as.numeric(as.character(annotations$end))
  annotations$code <- as.character(annotations$code)

  valid_rows <- !is.na(annotations$start) &
    !is.na(annotations$end) &
    !is.na(annotations$code) &
    annotations$start <= annotations$end

  annotations <- annotations[valid_rows, ]

  if (nrow(annotations) <= 1) {
    return(list())
  }

  overlaps <- list()
  for (i in 1:(nrow(annotations)-1)) {
    for (j in (i+1):nrow(annotations)) {
      # Check for overlap
      if (annotations$start[i] <= annotations$end[j] &&
          annotations$end[i] >= annotations$start[j]) {
        overlaps <- c(overlaps, list(list(
          code1 = annotations$code[i],
          code2 = annotations$code[j],
          overlap_start = max(annotations$start[i], annotations$start[j]),
          overlap_end = min(annotations$end[i], annotations$end[j])
        )))
      }
    }
  }

  return(overlaps)
}

#' Analyze combinations of code pairs
#'
#' @description
#' Analyzes the frequency of different code combinations by counting how often
#' different pairs of codes appear together in overlapping annotations.
#'
#' @param overlaps List of overlap information, where each element contains:
#'   \itemize{
#'     \item code1: character, identifier of first code
#'     \item code2: character, identifier of second code
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item frequencies: Table object containing counts of each code pair combination,
#'       where row names are formatted as "code1-code2" with codes sorted alphabetically
#'   }
#'
#' @details
#' The function processes overlapping code pairs and creates a frequency table of their
#' combinations. Code pairs are sorted alphabetically before counting to ensure consistent
#' ordering (e.g., "A-B" and "B-A" are counted as the same combination). Returns an empty
#' list if no overlaps are provided.
#'
#' @keywords internal
analyze_code_combinations <- function(overlaps) {
  if (length(overlaps) == 0) return(list())

  # Count frequency of code pairs
  combinations <- table(sapply(overlaps, function(x) {
    paste(sort(c(x$code1, x$code2)), collapse = "-")
  }))

  return(list(
    frequencies = combinations
  ))
}

#' Analyze characteristics of code overlaps
#'
#' @description
#' Analyzes the characteristics of overlapping code applications by calculating
#' various metrics about overlap patterns.
#'
#' @param overlaps List of overlap information, where each element contains:
#'   \itemize{
#'     \item overlap_start: numeric, starting position of overlap
#'     \item overlap_end: numeric, ending position of overlap
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item avg_length: Numeric value of average overlap length
#'     \item total_overlaps: Integer count of total overlapping instances
#'   }
#'
#' @details
#' Calculates metrics about code overlaps including the average length of
#' overlapping regions and the total number of overlaps. Returns empty list
#' for empty input.
#'
#' @keywords internal
analyze_overlap_characteristics <- function(overlaps) {
  if (length(overlaps) == 0) return(list())

  # Calculate overlap lengths
  lengths <- sapply(overlaps, function(x) {
    x$overlap_end - x$overlap_start + 1
  })

  return(list(
    avg_length = mean(lengths),
    total_overlaps = length(overlaps)
  ))
}

#' Find transitions between codes
#'
#' @description
#' Identifies and analyzes transitions between consecutive code applications
#' to understand coding sequence patterns.
#'
#' @param annotations Data frame of sorted annotations with columns:
#'   \itemize{
#'     \item start: numeric, starting position
#'     \item end: numeric, ending position
#'     \item code: character, code identifier
#'   }
#'
#' @return List of code transitions, each containing:
#'   \itemize{
#'     \item from: Source code
#'     \item to: Target code
#'   }
#'
#' @keywords internal
find_code_transitions <- function(annotations) {
  if (nrow(annotations) <= 1) return(list())

  # Find sequences of codes
  transitions <- list()
  for (i in 1:(nrow(annotations)-1)) {
    transitions <- append(transitions,
                          list(c(from = annotations$code[i],
                                 to = annotations$code[i+1])))
  }

  return(transitions)
}

#' Find repeated patterns in code sequences
#'
#' @description
#' Identifies repeating patterns of 2-3 codes in sequence to uncover recurring
#' coding structures.
#'
#' @param annotations Data frame of sorted annotations with columns:
#'   \itemize{
#'     \item code: character, code identifier
#'   }
#'
#' @return Named list of pattern frequencies where:
#'   \itemize{
#'     \item names: Code patterns (e.g. "code1-code2")
#'     \item values: Number of occurrences
#'   }
#'
#' @keywords internal
find_repeated_sequences <- function(annotations) {
  if (nrow(annotations) <= 1) return(list())

  # Look for repeated patterns in code sequences
  code_sequence <- annotations$code

  # Look for patterns of length 2-3
  patterns <- list()
  for (len in 2:min(3, length(code_sequence))) {
    for (i in 1:(length(code_sequence) - len + 1)) {
      pattern <- code_sequence[i:(i+len-1)]
      pattern_str <- paste(pattern, collapse = "-")
      patterns[[pattern_str]] <- sum(sapply(
        seq_along(code_sequence),
        function(j) {
          if (j + len - 1 > length(code_sequence)) return(FALSE)
          all(code_sequence[j:(j+len-1)] == pattern)
        }
      ))
    }
  }

  # Remove patterns that only occur once
  patterns <- patterns[patterns > 1]

  return(patterns)
}

#' Compare coverage patterns between coding strategies
#'
#' @description
#' Analyzes and compares the coverage patterns between different coding strategies,
#' including total codes used and unique code counts.
#'
#' @param coding_strategies List of coding strategies, where each strategy contains:
#'   \itemize{
#'     \item coverage: List containing distribution information
#'     \item frequencies: Table of code frequencies
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item total_codes_range: Numeric vector with min and max total codes
#'     \item unique_codes_range: Numeric vector with min and max unique codes
#'   }
#'
#' @keywords internal
compare_coverage <- function(coding_strategies) {
  tryCatch({
    coverage_stats <- lapply(coding_strategies, function(strategy) {
      freqs <- strategy$coverage$distribution$frequencies
      list(
        total_codes = sum(freqs),
        unique_codes = length(freqs),
        code_frequencies = freqs
      )
    })

    total_codes <- sapply(coverage_stats, `[[`, "total_codes")
    unique_codes <- sapply(coverage_stats, `[[`, "unique_codes")

    return(list(
      total_codes_range = if(length(total_codes) > 0) range(total_codes) else c(0, 0),
      unique_codes_range = if(length(unique_codes) > 0) range(unique_codes) else c(0, 0)
    ))
  }, error = function(e) {
    return(list(
      total_codes_range = c(0, 0),
      unique_codes_range = c(0, 0)
    ))
  })
}

#' Compare code application patterns between coders
#'
#' @description
#' Analyzes and compares how different coders apply codes by examining code segment
#' lengths and memo usage patterns across coding strategies.
#'
#' @param patterns_list List of coding patterns from different coders, where each
#'        pattern contains:
#'   \itemize{
#'     \item typical_length: numeric, average length of code segments
#'     \item memo_patterns: list containing memo usage statistics
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item length_variation: Character string describing variation in code segment lengths
#'     \item memo_usage_summary: Character string describing differences in memo usage
#'   }
#'
#' @keywords internal
compare_code_patterns <- function(patterns_list) {
  if (length(patterns_list) < 2) return("Need at least two sets for comparison")

  # Compare code application patterns
  lengths <- lapply(patterns_list, function(x) {
    sapply(x, function(p) p$typical_length)
  })

  # Compare memo usage
  memo_usage <- lapply(patterns_list, function(x) {
    sapply(x, function(p) p$memo_patterns$memo_frequency)
  })

  list(
    length_variation = "Variation in code segment lengths across coders",
    memo_usage_summary = "Differences in memo usage patterns"
  )
}

#' Compare code co-occurrence patterns between coders
#'
#' @description
#' Analyzes how different coders overlap in their code applications by comparing
#' the frequency and patterns of code co-occurrences.
#'
#' @param overlaps_list List of overlap patterns from different coders, where each
#'        entry contains:
#'   \itemize{
#'     \item combinations: List containing frequency table of code co-occurrences
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item overlap_variation: Numeric value indicating range of overlap counts
#'     \item summary: Character string describing variation in overlapping pairs
#'   }
#'
#' @keywords internal
compare_co_occurrences <- function(overlaps_list) {
  if (length(overlaps_list) < 2) return("Need at least two sets for comparison")

  # Compare overlap patterns
  overlap_counts <- sapply(overlaps_list, function(x) {
    length(x$combinations$frequencies)
  })

  list(
    overlap_variation = diff(range(overlap_counts)),
    summary = sprintf("Number of overlapping code pairs varies from %d to %d",
                      min(overlap_counts), max(overlap_counts))
  )
}

#' Compare code sequence patterns between coders
#'
#' @description
#' Analyzes how different coders sequence their codes by comparing the patterns
#' and frequency of code transitions.
#'
#' @param sequences_list List of sequence patterns from different coders, where each
#'        entry contains:
#'   \itemize{
#'     \item transitions: List of code transitions observed in the text
#'   }
#'
#' @return List containing:
#'   \itemize{
#'     \item sequence_variation: Numeric value indicating range of transition counts
#'     \item summary: Character string describing variation in code transitions
#'   }
#'
#' @keywords internal
compare_sequences <- function(sequences_list) {
  if (length(sequences_list) < 2) return("Need at least two sets for comparison")

  # Compare sequence patterns
  transition_counts <- sapply(sequences_list, function(x) {
    length(x$transitions)
  })

  list(
    sequence_variation = diff(range(transition_counts)),
    summary = sprintf("Number of code transitions varies from %d to %d",
                      min(transition_counts), max(transition_counts))
  )
}

#' Format coverage difference analysis results
#'
#' @description
#' Formats the results of coverage difference analysis into a human-readable
#' string, handling both character and list inputs appropriately.
#'
#' @param differences Either a character string containing direct analysis results
#'        or a list containing:
#'   \itemize{
#'     \item density_summary: Character string summarizing density differences
#'   }
#'
#' @return Character string containing formatted coverage analysis results
#'
#' @keywords internal
format_coverage_differences <- function(differences) {
  if (is.character(differences)) return(differences)
  if (is.list(differences) && !is.null(differences$density_summary)) {
    return(differences$density_summary)
  }
  return("Coverage analysis completed")
}

#' Format code difference analysis results
#'
#' @description
#' Formats the results of code difference analysis into a human-readable
#' string, handling both character and list inputs appropriately.
#'
#' @param differences Either a character string containing direct analysis results
#'        or a list containing:
#'   \itemize{
#'     \item pattern_summary: Character string summarizing code pattern differences
#'   }
#'
#' @return Character string containing formatted code analysis results
#'
#' @keywords internal
format_code_differences <- function(differences) {
  if (is.character(differences)) return(differences)
  if (is.list(differences) && !is.null(differences$pattern_summary)) {
    return(differences$pattern_summary)
  }
  return("Code pattern analysis completed")
}

#' Format overlap difference analysis results
#'
#' @description
#' Formats the results of overlap difference analysis into a human-readable
#' string, processing both character and complex input types.
#'
#' @param differences Either a character string containing direct analysis results
#'        or a complex analysis object
#'
#' @return Character string containing formatted overlap analysis results
#'
#' @keywords internal
format_overlap_differences <- function(differences) {
  if (is.character(differences)) return(differences)
  return("Overlap analysis completed")
}

#' Format sequence difference analysis results
#'
#' @description
#' Formats the results of sequence difference analysis into a human-readable
#' string, processing both character and complex input types.
#'
#' @param differences Either a character string containing direct analysis results
#'        or a complex analysis object
#'
#' @return Character string containing formatted sequence analysis results
#'
#' @keywords internal
format_sequence_differences <- function(differences) {
  if (is.character(differences)) return(differences)
  return("Sequence analysis completed")
}

#' Plot code overlap patterns
#'
#' @description
#' Creates a barplot visualization of code overlap patterns, showing the frequency
#' of different code co-occurrences with rotated labels for better readability.
#'
#' @param overlaps List containing overlap information:
#'   \itemize{
#'     \item combinations: List containing frequencies of code co-occurrences
#'   }
#' @param main Character string for plot title
#' @param ... Additional arguments passed to barplot()
#'
#' @return Invisible NULL, called for side effect of creating plot
#'
#' @importFrom graphics barplot text par
#'
#' @keywords internal
plot_overlap_patterns <- function(overlaps, main = "", ...) {
  if (is.null(overlaps) || length(overlaps$combinations$frequencies) == 0) {
    plot(0, 0, type = "n",
         main = main,
         xlab = "No overlaps available",
         ylab = "")
    return()
  }

  # Save current par settings and restore on exit
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar))

  # Create barplot with rotated labels
  bp <- barplot(overlaps$combinations$frequencies,
                main = main,
                xlab = "",
                ylab = "Overlap Count",
                las = 2,
                cex.names = 0.8,
                ...)

  # Add labels below
  text(x = bp,
       y = par("usr")[3] - 0.1,
       labels = names(overlaps$combinations$frequencies),
       xpd = TRUE,
       srt = 45,
       adj = 1,
       cex = 0.7)
}

#' Plot code sequence patterns
#'
#' @description
#' Creates a barplot visualization of code sequence patterns, showing the frequency
#' of different code transitions with rotated labels for better readability.
#'
#' @param sequences List containing sequence information:
#'   \itemize{
#'     \item transitions: List of code transitions
#'   }
#' @param main Character string for plot title
#' @param ... Additional arguments passed to barplot()
#'
#' @return Invisible NULL, called for side effect of creating plot
#'
#' @importFrom graphics barplot text par
#'
#' @keywords internal
plot_sequence_patterns <- function(sequences, main = "", ...) {
  if (is.null(sequences) || length(sequences$transitions) == 0) {
    plot(0, 0, type = "n",
         main = main,
         xlab = "No sequences available",
         ylab = "")
    return()
  }

  # Save current par settings and restore on exit
  oldpar <- par(no.readonly = TRUE)
  on.exit(par(oldpar))

  # Convert transitions to table
  trans_table <- table(sapply(sequences$transitions,
                              function(x) paste(x[1], "->", x[2])))

  # Create barplot with adjusted margins
  bp <- barplot(trans_table,
                main = main,
                xlab = "",
                ylab = "Frequency",
                las = 2,
                cex.names = 0.8,
                ...)

  # Add labels below
  text(x = bp,
       y = par("usr")[3] - 0.1,
       labels = names(trans_table),
       xpd = TRUE,
       srt = 45,
       adj = 1,
       cex = 0.7)
}

#' Handle directory creation confirmation
#'
#' @description
#' Creates the data directory after receiving user confirmation
#'
#' @param input Shiny input object
#' @param rv ReactiveValues object containing application state
#' @param session Shiny session object
#' @keywords internal
handle_dir_confirmation <- function(input, rv, session) {
  observeEvent(input$confirm_create_dir, {
    data_dir <- tools::R_user_dir("textAnnotatoR", "data")

    tryCatch({
      dir.create(data_dir, recursive = TRUE)
      rv$data_dir <- data_dir
      removeModal()
      showNotification("Directory created successfully", type = "message")
    }, error = function(e) {
      showNotification(
        sprintf("Failed to create directory: %s", e$message),
        type = "error"
      )
      rv$data_dir <- NULL
    })
  })
}

#' JavaScript code for handling text selection and UI interactions
#'
#' @description
#' This internal JavaScript code provides functionality for text selection,
#' popup menus, and interactive UI elements in the text annotation interface.
#' It manages mouse events for text selection, highlighting, and code application.
#'
#' @details
#' The JavaScript code implements the following functionality:
#' \itemize{
#'   \item Creation and management of popup menus for code operations
#'   \item Text selection handling with mouse events
#'   \item Highlighting of selected text
#'   \item Communication with Shiny server through custom message handlers
#'   \item Event handling for code replacement, renaming, and deletion
#' }
#'
#' @section Event Handlers:
#' \itemize{
#'   \item Text selection events (mousedown, mousemove, mouseup)
#'   \item Popup menu events for code operations
#'   \item Custom Shiny message handlers for selection state
#' }
#'
#' @note
#' This is an internal function used by the textAnnotatoR package and
#' should not be called directly by users.
#'
#' @keywords internal
#' @name addJS
#' @format A character string containing JavaScript code
NULL
addJS <- "
$(document).ready(function() {
  var popupMenu = $('<div id=\"popupMenu\" style=\"position: absolute; display: none; background-color: white; border: 1px solid black; padding: 5px;\"></div>');
  $('body').append(popupMenu);

  $(document).on('click', '.code-display', function(e) {
    var code = $(this).data('code');
    var start = $(this).data('start');
    var end = $(this).data('end');
    popupMenu.html(
      '<button id=\"removeAnnotation\" title=\"Remove this annotation only\">Remove Annotation</button><br>' +
      '<button id=\"replaceCode\" title=\"Replace code for this annotation\">Replace Code</button><br>' +
      '<hr>' +
      '<button id=\"renameCode\" title=\"Rename this code globally\">Rename Code</button><br>' +
      '<button id=\"deleteCode\" title=\"Delete this code from all annotations\">Delete Code</button>'
    );
    popupMenu.css({
      left: e.pageX + 'px',
      top: e.pageY + 'px'
    }).show();

    $('#removeAnnotation').click(function() {
      Shiny.setInputValue('remove_annotation', {code: code, start: start, end: end});
      popupMenu.hide();
    });

    $('#replaceCode').click(function() {
      Shiny.setInputValue('replace_code', {code: code, start: start, end: end});
      popupMenu.hide();
    });

    $('#renameCode').click(function() {
      Shiny.setInputValue('rename_code', {code: code, start: start, end: end});
      popupMenu.hide();
    });

    $('#deleteCode').click(function() {
      Shiny.setInputValue('delete_code', {code: code, start: start, end: end});
      popupMenu.hide();
    });
  });

  $(document).on('click', function(e) {
    if (!$(e.target).closest('#popupMenu, .code-display').length) {
      popupMenu.hide();
    }
  });
});
"

# Helper function to get all themed codes
get_all_themed_codes <- function(node) {
  codes <- character(0)

  traverse_node <- function(node) {
    if (is.null(node)) return()

    if (!is.null(node$type) && node$type == "code" && !is.null(node$name)) {
      codes <<- c(codes, node$name)
    }

    if (!is.null(node$children)) {
      children <- if (inherits(node$children, "Node")) {
        list(node$children)
      } else if (is.list(node$children)) {
        node$children
      } else {
        list()
      }

      for (child in children) {
        traverse_node(child)
      }
    }
  }

  traverse_node(node)
  return(unique(codes))
}

# UPDATE 6: Add text parsing functions for unit-based analysis

parse_text_into_units <- function(text, unit) {
  if (is.null(text) || nchar(text) == 0) {
    return(data.frame(unit_id = integer(0), start = integer(0),
                      end = integer(0), text = character(0),
                      stringsAsFactors = FALSE))
  }

  switch(unit,
         "sentence" = parse_sentences(text),
         "paragraph" = parse_paragraphs(text),
         "document" = parse_document(text)
  )
}

parse_sentences <- function(text) {
  # Split on sentence-ending punctuation followed by whitespace or end of string
  sentence_pattern <- "(?<=[.!?])\\s+(?=[A-Z])|(?<=[.!?])$"

  # Find sentence boundaries
  boundaries <- gregexpr(sentence_pattern, text, perl = TRUE)[[1]]

  if (length(boundaries) == 1 && boundaries[1] == -1) {
    # No sentence boundaries found, treat as single sentence
    return(data.frame(
      unit_id = 1,
      start = 1,
      end = nchar(text),
      text = text,
      stringsAsFactors = FALSE
    ))
  }

  # Calculate start and end positions
  starts <- c(1, boundaries + 1)
  ends <- c(boundaries, nchar(text))

  # Extract sentence texts
  sentences <- mapply(function(s, e) substr(text, s, e), starts, ends, USE.NAMES = FALSE)
  sentences <- trimws(sentences)

  # Remove empty sentences
  non_empty <- nchar(sentences) > 0

  data.frame(
    unit_id = seq_along(sentences)[non_empty],
    start = starts[non_empty],
    end = ends[non_empty],
    text = sentences[non_empty],
    stringsAsFactors = FALSE
  )
}

parse_paragraphs <- function(text) {
  # Split on double line breaks or single line breaks
  paragraphs <- strsplit(text, "\n\n|\n")[[1]]
  paragraphs <- trimws(paragraphs)

  # Remove empty paragraphs
  paragraphs <- paragraphs[nchar(paragraphs) > 0]

  if (length(paragraphs) == 0) {
    return(data.frame(unit_id = integer(0), start = integer(0),
                      end = integer(0), text = character(0),
                      stringsAsFactors = FALSE))
  }

  # Calculate positions in original text
  cumulative_lengths <- cumsum(nchar(paragraphs) + 1) # +1 for line breaks
  starts <- c(1, cumulative_lengths[-length(cumulative_lengths)] + 1)
  ends <- cumulative_lengths - 1
  ends[length(ends)] <- nchar(text) # Adjust last paragraph end

  data.frame(
    unit_id = seq_along(paragraphs),
    start = starts,
    end = ends,
    text = paragraphs,
    stringsAsFactors = FALSE
  )
}

parse_document <- function(text) {
  data.frame(
    unit_id = 1,
    start = 1,
    end = nchar(text),
    text = text,
    stringsAsFactors = FALSE
  )
}

assign_annotations_to_units <- function(annotations, text_units) {
  if (nrow(annotations) == 0 || nrow(text_units) == 0) {
    return(data.frame(unit_id = integer(0), code = character(0),
                      annotation_start = integer(0), annotation_end = integer(0),
                      stringsAsFactors = FALSE))
  }

  assignments <- list()

  for (i in seq_len(nrow(annotations))) {
    ann_start <- annotations$start[i]
    ann_end <- annotations$end[i]
    ann_code <- annotations$code[i]

    # Find units that overlap with this annotation
    overlapping_units <- which(
      text_units$start <= ann_end & text_units$end >= ann_start
    )

    if (length(overlapping_units) > 0) {
      for (unit_idx in overlapping_units) {
        assignments[[length(assignments) + 1]] <- data.frame(
          unit_id = text_units$unit_id[unit_idx],
          code = ann_code,
          annotation_start = ann_start,
          annotation_end = ann_end,
          stringsAsFactors = FALSE
        )
      }
    }
  }

  if (length(assignments) == 0) {
    return(data.frame(unit_id = integer(0), code = character(0),
                      annotation_start = integer(0), annotation_end = integer(0),
                      stringsAsFactors = FALSE))
  }

  do.call(rbind, assignments)
}

calculate_unit_cooccurrence_matrix <- function(codes, unit_assignments) {
  n_codes <- length(codes)
  co_matrix <- matrix(0, nrow = n_codes, ncol = n_codes,
                      dimnames = list(codes, codes))

  if (nrow(unit_assignments) == 0) {
    return(co_matrix)
  }

  # Get codes present in each unit
  units_with_codes <- split(unit_assignments$code, unit_assignments$unit_id)

  # Count co-occurrences
  for (unit_codes in units_with_codes) {
    unique_codes_in_unit <- unique(unit_codes)

    if (length(unique_codes_in_unit) > 1) {
      # Generate all pairs of codes in this unit
      code_pairs <- expand.grid(unique_codes_in_unit, unique_codes_in_unit,
                                stringsAsFactors = FALSE)

      for (j in seq_len(nrow(code_pairs))) {
        code1 <- code_pairs[j, 1]
        code2 <- code_pairs[j, 2]

        if (code1 != code2) {
          co_matrix[code1, code2] <- co_matrix[code1, code2] + 1
        }
      }
    }
  }

  co_matrix
}

calculate_jaccard_similarity_matrix <- function(codes, unit_assignments) {
  n_codes <- length(codes)
  jaccard_matrix <- matrix(0, nrow = n_codes, ncol = n_codes,
                           dimnames = list(codes, codes))

  if (nrow(unit_assignments) == 0) {
    return(jaccard_matrix)
  }

  # Create binary presence matrix: codes x units
  units_with_codes <- split(unit_assignments$code, unit_assignments$unit_id)
  all_unit_ids <- unique(unit_assignments$unit_id)

  code_presence <- matrix(0, nrow = n_codes, ncol = length(all_unit_ids),
                          dimnames = list(codes, as.character(all_unit_ids)))

  for (i in seq_along(all_unit_ids)) {
    unit_id <- all_unit_ids[i]
    codes_in_unit <- unique(units_with_codes[[as.character(unit_id)]])
    code_presence[codes_in_unit, i] <- 1
  }

  # Calculate Jaccard similarity for each pair
  for (i in 1:n_codes) {
    for (j in 1:n_codes) {
      if (i != j) {
        intersection <- sum(code_presence[i, ] & code_presence[j, ])
        union <- sum(code_presence[i, ] | code_presence[j, ])

        jaccard_matrix[i, j] <- if (union > 0) intersection / union else 0
      }
    }
  }

  jaccard_matrix
}

calculate_phi_coefficient_matrix <- function(codes, unit_assignments) {
  n_codes <- length(codes)
  phi_matrix <- matrix(0, nrow = n_codes, ncol = n_codes,
                       dimnames = list(codes, codes))

  if (nrow(unit_assignments) == 0) {
    return(phi_matrix)
  }

  # Create binary presence matrix: codes x units
  units_with_codes <- split(unit_assignments$code, unit_assignments$unit_id)
  all_unit_ids <- unique(unit_assignments$unit_id)
  n_units <- length(all_unit_ids)

  code_presence <- matrix(0, nrow = n_codes, ncol = n_units,
                          dimnames = list(codes, as.character(all_unit_ids)))

  for (i in seq_along(all_unit_ids)) {
    unit_id <- all_unit_ids[i]
    codes_in_unit <- unique(units_with_codes[[as.character(unit_id)]])
    code_presence[codes_in_unit, i] <- 1
  }

  # Calculate Phi coefficient for each pair
  for (i in 1:n_codes) {
    for (j in 1:n_codes) {
      if (i != j) {
        # 2x2 contingency table
        n11 <- sum(code_presence[i, ] & code_presence[j, ])  # Both present
        n10 <- sum(code_presence[i, ] & !code_presence[j, ]) # Only i present
        n01 <- sum(!code_presence[i, ] & code_presence[j, ]) # Only j present
        n00 <- sum(!code_presence[i, ] & !code_presence[j, ]) # Neither present

        # Phi coefficient calculation
        numerator <- (n11 * n00) - (n10 * n01)
        denominator <- sqrt((n11 + n10) * (n01 + n00) * (n11 + n01) * (n10 + n00))

        phi_matrix[i, j] <- if (denominator > 0) numerator / denominator else 0
      }
    }
  }

  phi_matrix
}

calculate_codes_per_unit_stats <- function(unit_assignments) {
  if (nrow(unit_assignments) == 0) {
    return(list(
      mean_codes_per_unit = 0,
      max_codes_per_unit = 0,
      units_with_multiple_codes = 0
    ))
  }

  codes_per_unit <- table(unit_assignments$unit_id)

  list(
    mean_codes_per_unit = mean(codes_per_unit),
    max_codes_per_unit = max(codes_per_unit),
    units_with_multiple_codes = sum(codes_per_unit > 1)
  )
}

#' Get next color from palette with cycling
#'
#' @description
#' Returns the next color from a predefined palette, cycling back to the beginning
#' when all colors have been used. This ensures consistent and readable colors.
#'
#' @param used_colors Character vector of already used colors
#' @return Character string containing a hex color code
#' @keywords internal
get_next_palette_color <- function(used_colors) {
  # Predefined palette of readable colors
  palette_colors <- c(
    "#FFE6CC", "#E6F3FF", "#E6FFE6", "#FFE6F3", "#F3E6FF",
    "#FFF2E6", "#E6F7FF", "#F0FFE6", "#FFE6F0", "#F7E6FF",
    "#FFEECC", "#CCE6FF", "#CCFFCC", "#FFCCEE", "#EECCFF"
  )

  # Find first unused color in palette
  for (color in palette_colors) {
    if (!color %in% used_colors) {
      return(color)
    }
  }

  # If all palette colors used, generate a random one
  return(generate_readable_color())
}

#' Generate readable highlight color
#'
#' @description
#' Generates a random color that provides good contrast with black text by ensuring
#' the color is bright enough. Uses HSL color space to maintain good saturation
#' while ensuring sufficient lightness.
#'
#' @return Character string containing a hex color code
#' @keywords internal
generate_readable_color <- function() {
  # Generate random hue (0-360)
  hue <- runif(1, 0, 360)

  # Set saturation between 0.3-0.6 for vibrant but not overwhelming colors
  saturation <- runif(1, 0.3, 0.6)

  # Set lightness between 0.8-0.95 to ensure readability with black text
  lightness <- runif(1, 0.8, 0.95)

  # Convert HSL to RGB
  hsl_to_rgb(hue, saturation, lightness)
}

#' Convert HSL to RGB color values
#'
#' @description
#' Converts HSL (Hue, Saturation, Lightness) color values to RGB values.
#' This allows for better control over color brightness and readability.
#'
#' @param h Numeric, hue value (0-360)
#' @param s Numeric, saturation value (0-1)
#' @param l Numeric, lightness value (0-1)
#'
#' @return Character string containing a hex color code
#' @keywords internal
hsl_to_rgb <- function(h, s, l) {
  h <- h / 360

  if (s == 0) {
    r <- g <- b <- l
  } else {
    hue_to_rgb <- function(p, q, t) {
      if (t < 0) t <- t + 1
      if (t > 1) t <- t - 1
      if (t < 1/6) return(p + (q - p) * 6 * t)
      if (t < 1/2) return(q)
      if (t < 2/3) return(p + (q - p) * (2/3 - t) * 6)
      return(p)
    }

    q <- if (l < 0.5) l * (1 + s) else l + s - l * s
    p <- 2 * l - q

    r <- hue_to_rgb(p, q, h + 1/3)
    g <- hue_to_rgb(p, q, h)
    b <- hue_to_rgb(p, q, h - 1/3)
  }

  sprintf("#%02X%02X%02X", round(r * 255), round(g * 255), round(b * 255))
}

#' Validate color readability
#'
#' @description
#' Checks if a given color provides sufficient contrast with black text.
#' Uses luminance calculation to determine readability.
#'
#' @param color Character string containing hex color code
#' @return Logical indicating whether color is readable
#' @keywords internal
is_color_readable <- function(color) {
  # Simple check for light colors (good contrast with black text)
  if (is.null(color) || !grepl("^#[0-9A-Fa-f]{6}$", color)) {
    return(FALSE)
  }

  # Extract RGB values
  r <- strtoi(substr(color, 2, 3), 16L) / 255
  g <- strtoi(substr(color, 4, 5), 16L) / 255
  b <- strtoi(substr(color, 6, 7), 16L) / 255

  # Calculate luminance
  luminance <- 0.299 * r + 0.587 * g + 0.114 * b

  # Return TRUE if light enough for good contrast with black text
  return(luminance > 0.7)
}

#' Update existing dark colors to readable ones
#'
#' @description
#' Checks all existing code colors and replaces any that are too dark
#' with more readable alternatives while trying to maintain color diversity.
#'
#' @param rv ReactiveValues object containing code colors
#' @return Updated ReactiveValues object
#' @keywords internal
update_dark_colors <- function(rv) {
  # Update any existing dark colors to more readable ones
  for (code in names(rv$code_colors)) {
    if (!is_color_readable(rv$code_colors[code])) {
      used_colors <- as.character(rv$code_colors[names(rv$code_colors) != code])
      rv$code_colors[code] <- get_next_palette_color(used_colors)
    }
  }
  return(rv)
}

# Position-based unit assignment (fallback when text is not available)
assign_annotations_to_position_units <- function(annotations, unit) {
  if (nrow(annotations) == 0) {
    return(data.frame(unit_id = integer(0), code = character(0),
                      annotation_start = integer(0), annotation_end = integer(0),
                      stringsAsFactors = FALSE))
  }

  switch(unit,
         "sentence" = {
           # Group annotations by proximity (within 100 characters = same "sentence")
           unit_assignments <- assign_by_proximity(annotations, max_distance = 100)
         },
         "paragraph" = {
           # Group annotations by proximity (within 500 characters = same "paragraph")
           unit_assignments <- assign_by_proximity(annotations, max_distance = 500)
         },
         "document" = {
           # All annotations in same document unit
           unit_assignments <- data.frame(
             unit_id = rep(1, nrow(annotations)),
             code = annotations$code,
             annotation_start = annotations$start,
             annotation_end = annotations$end,
             stringsAsFactors = FALSE
           )
         }
  )

  unit_assignments
}

assign_by_proximity <- function(annotations, max_distance) {
  # Sort annotations by start position
  sorted_annotations <- annotations[order(annotations$start), ]

  unit_id <- 1
  current_unit_end <- sorted_annotations$end[1]
  unit_assignments <- integer(nrow(sorted_annotations))
  unit_assignments[1] <- unit_id

  for (i in 2:nrow(sorted_annotations)) {
    if (sorted_annotations$start[i] - current_unit_end <= max_distance) {
      # Same unit
      unit_assignments[i] <- unit_id
      current_unit_end <- max(current_unit_end, sorted_annotations$end[i])
    } else {
      # New unit
      unit_id <- unit_id + 1
      unit_assignments[i] <- unit_id
      current_unit_end <- sorted_annotations$end[i]
    }
  }

  data.frame(
    unit_id = unit_assignments,
    code = sorted_annotations$code,
    annotation_start = sorted_annotations$start,
    annotation_end = sorted_annotations$end,
    stringsAsFactors = FALSE
  )
}

# Enhanced summary statistics calculation
calculate_cooccurrence_summary <- function(co_matrix, jaccard_matrix, phi_matrix, codes) {
  tryCatch({
    n_codes <- length(codes)

    if (n_codes == 0 || nrow(co_matrix) == 0) {
      return(list(
        total_codes = 0,
        max_co_occurrence = 0,
        max_jaccard = 0,
        mean_jaccard = 0,
        significant_pairs = 0
      ))
    }

    # Calculate maximum co-occurrence count
    max_cooccur <- if(length(co_matrix) > 0) max(co_matrix, na.rm = TRUE) else 0

    # Calculate Jaccard statistics (excluding diagonal)
    jaccard_upper <- jaccard_matrix[upper.tri(jaccard_matrix)]
    max_jaccard <- if(length(jaccard_upper) > 0) max(jaccard_upper, na.rm = TRUE) else 0
    mean_jaccard <- if(length(jaccard_upper) > 0) mean(jaccard_upper, na.rm = TRUE) else 0

    # Count significant Phi coefficient pairs (|phi| > 0.3)
    phi_upper <- phi_matrix[upper.tri(phi_matrix)]
    significant_pairs <- if(length(phi_upper) > 0) sum(abs(phi_upper) > 0.3, na.rm = TRUE) else 0

    return(list(
      total_codes = n_codes,
      max_co_occurrence = max_cooccur,
      max_jaccard = max_jaccard,
      mean_jaccard = mean_jaccard,
      significant_pairs = significant_pairs
    ))

  }, error = function(e) {
    # Return safe defaults if calculation fails
    return(list(
      total_codes = length(codes),
      max_co_occurrence = 0,
      max_jaccard = 0,
      mean_jaccard = 0,
      significant_pairs = 0
    ))
  })
}

#' Check if code exists in hierarchy
#'
#' @description
#' Helper function to check if a code already exists anywhere in the code hierarchy
#'
#' @param node Root node of the hierarchy tree
#' @param code_name Character string of the code name to search for
#'
#' @return Logical indicating whether the code exists in the hierarchy
#' @keywords internal
code_exists_in_hierarchy <- function(node, code_name) {
  if (is.null(node) || is.null(code_name)) return(FALSE)

  # Check if this node is the code we're looking for
  if (!is.null(node$name) && node$name == code_name &&
      !is.null(node$type) && node$type == "code") {
    return(TRUE)
  }

  # Recursively check children
  if (!is.null(node$children)) {
    for (child in node$children) {
      if (code_exists_in_hierarchy(child, code_name)) {
        return(TRUE)
      }
    }
  }

  return(FALSE)
}

#' Sync codes with hierarchy after loading project
#'
#' @description
#' Ensures that codes in rv$codes are properly represented in the hierarchy
#' after loading a project, adding any missing codes to the root level
#'
#' @param rv ReactiveValues object containing application state
#'
#' @return Updated ReactiveValues object
#' @keywords internal
sync_codes_with_hierarchy <- function(rv) {
  if (is.null(rv$codes) || length(rv$codes) == 0) {
    return(rv)
  }

  # Get all codes currently in the hierarchy
  codes_in_hierarchy <- get_all_themed_codes(rv$code_tree)

  # Find codes that exist in rv$codes but not in hierarchy
  missing_codes <- setdiff(rv$codes, codes_in_hierarchy)

  # Add missing codes to the root level of the hierarchy
  for (code in missing_codes) {
    new_code_node <- rv$code_tree$AddChild(code)
    new_code_node$type <- "code"
    new_code_node$description <- ""
    new_code_node$created <- Sys.time()
  }

  return(rv)
}

#' Remove annotation without deleting code
#'
#' @description
#' Removes a specific annotation instance while preserving the code
#' for use with other annotations.
#'
#' @param rv ReactiveValues object containing annotations
#' @param start Starting position of annotation to remove
#' @param end Ending position of annotation to remove
#' @param code Code of annotation to remove
#'
#' @return Updated rv object with annotation removed
#' @keywords internal
remove_annotation <- function(rv, start, end, code) {
  if (nrow(rv$annotations) > 0) {
    # Find the specific annotation to remove
    idx <- which(rv$annotations$start == start &
                   rv$annotations$end == end &
                   rv$annotations$code == code)

    if (length(idx) > 0) {
      # Remove only the annotation, keep the code
      rv$annotations <- rv$annotations[-idx[1], ]  # Remove only first match

      # Don't remove the code from rv$codes even if no more annotations use it
      # This preserves the code for potential future use

      return(rv)
    }
  }
  return(rv)
}
