% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get-tk_get_holiday_signature.R
\name{tk_get_holiday}
\alias{tk_get_holiday}
\alias{tk_get_holiday_signature}
\alias{tk_get_holidays_by_year}
\title{Get holiday features from a time-series index}
\usage{
tk_get_holiday_signature(
  idx,
  holiday_pattern = ".",
  locale_set = c("all", "none", "World", "US", "CA", "GB", "FR", "IT", "JP", "CH", "DE"),
  exchange_set = c("all", "none", "NYSE", "LONDON", "NERC", "TSX", "ZURICH")
)

tk_get_holidays_by_year(years = year(today()))
}
\arguments{
\item{idx}{A time-series index that is a vector of dates or datetimes.}

\item{holiday_pattern}{A regular expression pattern to search the "Holiday Set".}

\item{locale_set}{Return binary holidays based on locale.
One of: "all", "none", "World", "US", "CA", "GB", "FR", "IT", "JP", "CH", "DE".}

\item{exchange_set}{Return binary holidays based on Stock Exchange Calendars.
One of: "all", "none", "NYSE", "LONDON", "NERC", "TSX", "ZURICH".}

\item{years}{One or more years to collect holidays for.}
}
\value{
Returns a \code{tibble} object describing the timeseries holidays.
}
\description{
Get holiday features from a time-series index
}
\details{
Feature engineering holidays can help identify critical patterns for
machine learning algorithms. \code{tk_get_holiday_signature()} helps by providing
feature sets for 3 types of features:

\strong{1. Individual Holidays}

These are \strong{single holiday features} that can be filtered using a pattern.
This helps in identifying which holidays are important to a machine learning model.
This can be useful for example in \strong{e-commerce initiatives}
(e.g. sales during Christmas and Thanskgiving).

\strong{2. Locale-Based Summary Sets}

Locale-based holdiay sets are useful for \strong{e-commerce initiatives}
(e.g. sales during Christmas and Thanskgiving). Filter on a locale to
identify all holidays in that locale.

\strong{3. Stock Exchange Calendar Summary Sets}

Exchange-based holdiay sets are useful for identifying \strong{non-working days.}
Filter on an index to identify all holidays that are commonly non-working.
}
\examples{
library(dplyr)
library(stringr)

# Works with time-based tibbles
idx <- tk_make_timeseries("2017-01-01", "2017-12-31", by = "day")

# --- BASIC USAGE ----

tk_get_holiday_signature(idx)

# ---- FILTERING WITH PATTERNS & SETS ----

# List available holidays - see patterns
tk_get_holidays_by_year(2020) \%>\%
    filter(holiday_name \%>\% str_detect("US_"))

# Filter using holiday patterns
# - Get New Years, Christmas and Thanksgiving Features in US
tk_get_holiday_signature(
    idx,
    holiday_pattern = "(US_NewYears)|(US_Christmas)|(US_Thanks)",
    locale_set      = "none",
    exchange_set    = "none")

# ---- APPLYING FILTERS ----

# Filter with locale sets - Signals all holidays in a locale
tk_get_holiday_signature(
    idx,
    holiday_pattern = "$^", # Matches nothing on purpose
    locale_set      = "US",
    exchange_set    = "none")

# Filter with exchange sets - Signals Common Non-Business Days
tk_get_holiday_signature(
    idx,
    holiday_pattern = "$^", # Matches nothing on purpose
    locale_set      = "none",
    exchange_set    = "NYSE")

}
\seealso{
\itemize{
\item \code{\link[=tk_augment_holiday_signature]{tk_augment_holiday_signature()}} - A quick way to add holiday features to a data.frame
\item \code{\link[=step_holiday_signature]{step_holiday_signature()}} - Preprocessing and feature engineering steps for use with \code{recipes}
}
}
