% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel_regression_estimator.R
\name{adjusted_est}
\alias{adjusted_est}
\title{Compute the Kernel Regression Estimator.}
\usage{
adjusted_est(
  X,
  x,
  t,
  b,
  kernel_name = "gaussian",
  kernel_params = c(),
  pd = TRUE,
  type = "autocovariance",
  meanX = mean(X),
  custom_kernel = FALSE
)
}
\arguments{
\item{X}{A vector representing observed values of the time series.}

\item{x}{A vector of lags.}

\item{t}{The arguments at which the autocovariance function is calculated at.}

\item{b}{Bandwidth parameter, greater than 0.}

\item{kernel_name}{The name of the symmetric kernel (see \link{kernel_symm}) function to be used. Possible values are:
gaussian, wave, rational_quadratic, and bessel_j. Alternatively, a custom kernel function can be provided, see the examples.}

\item{kernel_params}{A vector of parameters of the kernel function. See \link{kernel_symm} for parameters.}

\item{pd}{Whether a positive-definite estimate should be used. Defaults to \code{TRUE}.}

\item{type}{Compute either the 'autocovariance' or 'autocorrelation'. Defaults to 'autocovariance'.}

\item{meanX}{The average value of \code{X}. Defaults to \code{mean(X)}.}

\item{custom_kernel}{If a custom kernel is to be used or not. Defaults to \code{FALSE}.}
}
\value{
A vector whose values are the kernel regression estimates.
}
\description{
This function computes the kernel regression estimator of the autocovariance function.
}
\details{
The kernel regression estimator of an autocovariance function is defined as
\deqn{
\hat{\rho}(t) = \left( \sum_{i=1}^{N} \sum_{j=1}^{N} \check{X}_{ij} K((t -  (t_{i} - t_{j})) / b) \right) \left( \sum_{i=1}^{N} \sum_{j=1}^{N} K((t -  (t_{i} - t_{j})) / b) \right)^{-1},
}
where \eqn{\check{X}_{ij} = (X(t_{i}) - \bar{X}) (X(t_{j}) - \bar{X}).}

If \code{pd} is \code{TRUE}, the estimator will be made positive-definite through the following procedure
\enumerate{
\item Take the discrete Fourier cosine transform,
\eqn{\widehat{\mathcal{F}}(\theta)}, of the estimated autocovariance function
\item Compute a modified spectrum \eqn{\widetilde{\mathcal{F}}(\theta) = \max(\widehat{\mathcal{F}}(\theta), 0)} for all sample frequencies.
\item Perform the Fourier inversion to obtain a new estimator.
}
}
\examples{
X <- c(1, 2, 3, 4)
adjusted_est(X, 1:4, 1:3, 0.1, "gaussian")
my_kernel <- function(x, theta, params) {
  stopifnot(theta > 0, length(x) >= 1)
  return(exp(-((abs(x) / theta)^params[1])) * (2 * theta  * gamma(1 + 1/params[1])))
}
adjusted_est(X, 1:4, 1:3, 0.1, my_kernel, c(0.25), custom_kernel = TRUE)
}
\references{
Hall, P. & Patil, P. (1994). Properties of nonparametric estimators of autocovariance for stationary random fields. Probability Theory and Related Fields 99(3), 399-424. https://doi.org/10.1007/bf01199899

Hall, P., Fisher, N. I., & Hoffmann, B. (1994). On the nonparametric estimation of covariance functions. The Annals of Statistics 22(4), 2115-2134. https://doi.org/10.1214/aos/1176325774
}
