% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods_IVDML.R
\name{robust_confint}
\alias{robust_confint}
\title{Compute Robust Confidence Interval for Treatment Effect in an IVDML Object}
\usage{
robust_confint(
  object,
  iv_method,
  level = 0.95,
  a = NULL,
  A = NULL,
  kernel_name = NULL,
  bandwidth = NULL,
  CI_range = NULL,
  agg_method = "DML_agg",
  gamma = 0.5
)
}
\arguments{
\item{object}{An object of class \code{IVDML}, produced by the \code{\link[=fit_IVDML]{fit_IVDML()}} function.}

\item{iv_method}{Character. The instrumental variable estimation method to use. Must be one of the methods specified in the fitted object.}

\item{level}{Numeric (default: 0.95). The confidence level for the confidence interval.}

\item{a}{Numeric (optional). A specific value of \code{A} at which to compute the confidence interval for the heterogeneous treatment effect. If \code{NULL}, the function returns the confidence interval for the homogeneous treatment effect.}

\item{A}{Numeric vector (optional). The variable with respect to which treatment effect heterogeneity is considered. If \code{NULL}, the function assumes the \code{A} used in model fitting.}

\item{kernel_name}{Character (optional). The name of the kernel function to use for smoothing (if a heterogeneous treatment effect is estimated). Must be one of \code{"boxcar"}, \code{"gaussian"}, \code{"epanechnikov"}, or \code{"tricube"}.}

\item{bandwidth}{Numeric (optional). The bandwidth for the kernel smoothing (if a heterogeneous treatment effect is estimated).}

\item{CI_range}{Numeric vector of length 2 (optional). The search range for the confidence interval. If \code{NULL}, the function sets \code{CI_range} to be four times as large as the standard confidence interval centered at the point estimate of the treatment effect.}

\item{agg_method}{Character (default: \code{"DML_agg"}). The aggregation method for computing the confidence interval. Options are:
\itemize{
\item \code{"DML_agg"}: Uses the Double Machine Learning (DML) aggregation approach.
\item \code{"MMB_agg"}: Uses the quantile-based aggregation method of Meinshausen, Meier, and Bühlmann (2009).
}}

\item{gamma}{Numeric (default: 0.5). Quantile level for the \code{"MMB_agg"} method. Ignored if \code{agg_method = "DML_agg"}.}
}
\value{
A list with the following elements:
\itemize{
\item \code{CI}: A named numeric vector with the lower and upper bounds of the confidence interval.
\item \code{level}: The confidence level used.
\item \code{message}: A message describing the nature of the confidence set (e.g., whether it spans the full range, is non-connected, or is empty due to optimization failure).
\item \code{heterogeneous_parameters}: A list of parameters (\code{a}, \code{kernel_name}, \code{bandwidth}) if a heterogeneous treatment effect is considered; otherwise, \code{NULL}.
}
}
\description{
This function computes a robust (with respect to weak IV) confidence interval/confidence set for the estimated treatment effect in a fitted \code{IVDML} object (output of \code{\link[=fit_IVDML]{fit_IVDML()}}). The confidence interval/confidence set is constructed by inverting the robust test from the \code{\link[=robust_p_value_aggregated]{robust_p_value_aggregated()}} function, which either uses the Double Machine Learning aggregation method (\code{"DML_agg"}) or the quantile-based method of Meinshausen, Meier, and Bühlmann (2009) (\code{"MMB_agg"}) to aggregate the p-values corresponding to the \code{S_split} cross-fitting sample splits (where \code{S_split} was an argument of the \code{\link[=fit_IVDML]{fit_IVDML()}} function).
}
\examples{
set.seed(1)
Z <- rnorm(100)
X <- Z + rnorm(100)
H <- rnorm(100)
D <- Z^2 + sin(X) + H + rnorm(100)
A <- X
Y <- tanh(A) * D + cos(X) - H + rnorm(100)
fit <- fit_IVDML(Y = Y, D = D, Z = Z, X = X, ml_method = "gam")
robust_confint(fit, iv_method = "mlIV", CI_range = c(-10, 10))
robust_confint(fit, iv_method = "mlIV", a = 0, A = A,
               kernel_name = "boxcar", bandwidth = 0.2, CI_range = c(-10, 10))

}
\references{
Meinshausen, N., Meier, L., & Bühlmann, P. (2009). \emph{P-values for high-dimensional regression}. Journal of the American Statistical Association, 104(488), 1671–1681.
}
