#' Reversal Association Pattern Analysis
#'
#' Performs reversal association pattern analysis for categorical data.
#' @param a numeric vector containing table entries
#' @param nr number of rows
#' @param nc number of columns
#' @param verbose logical; if TRUE, prints informational messages (default: TRUE)
#' @return List containing the analysis results
#' @examples
#' a = c(38,58,69,34,28,17,28,14,14)
#' nr = 3
#' nc = 3
#' fnRAP(a, nr, nc)
#' @importFrom stats chisq.test
#' @importFrom utils combn
#' @export
fnRAP <- function(a, nr, nc, verbose = TRUE) {
  
  tot_tab <- 0
  
  # Ensure matrix is large enough
  if ((nr == 2 && nc > 2) || (nr > 2 && nc == 2) || (nr > 2 && nc > 2)) {
    
    mat <- matrix(a, nr, nc, byrow = TRUE)
    ori_con <- round(chisq.test(mat)$p.value, 4)
    
    # Initialize the details data frame
    outd <- data.frame(
      "No. of rows" = nr,
      "No. of cols" = nc,
      "Selected rows" = toString(1:nr),
      "Selected cols" = toString(1:nc),
      "Pvalue" = ori_con,
      "Significant_5pct" = ori_con < 0.05,
      stringsAsFactors = FALSE
    )
    
    outlist <- list()
    
    for (i in 2:nr) {
      outlist[[i]] <- list()
      for (j in 2:nc) {
        outlist[[i]][[j]] <- list()
        rpermmat <- t(combn(1:nr, i))
        cpermmat <- t(combn(1:nc, j))
        rfact <- if (i == nr) 1 else factorial(nr)/(factorial(i) * factorial(nr-i))
        cfact <- if (j == nc) 1 else factorial(nc)/(factorial(j) * factorial(nc-j))
        
        if (i == nr && j == nc) {
          outlist[[i]][[j]][[1]] <- mat
          outlist[[i]][[j]][[2]] <- outd
        } else {
          for (ii in 1:rfact) {
            for (jj in 1:cfact) {
              temp_mat <- mat[rpermmat[ii, ], cpermmat[jj, ]]
              tempvar <- round(chisq.test(temp_mat)$p.value, 4)
              
              # Check if the submatrix produces a reversal
              if ((ori_con < 0.05 && tempvar > 0.05) || (ori_con > 0.05 && tempvar < 0.05)) {
                outlist[[i]][[j]][[(ii-1)*cfact + jj]] <- tempvar
                
                # Create a properly named data frame row
                new_row <- data.frame(
                  "No. of rows" = nrow(temp_mat),
                  "No. of cols" = ncol(temp_mat),
                  "Selected rows" = toString(rpermmat[ii, ]),
                  "Selected cols" = toString(cpermmat[jj, ]),
                  "Pvalue" = tempvar,
                  "Significant_5pct" = tempvar < 0.05,
                  stringsAsFactors = FALSE
                )
                
                outd <- rbind(outd, new_row)
                tot_tab <- tot_tab + 1
              }
            }
          }
        }
      }
    }
    
    if (verbose) message("Analysis complete. Returning results.")
    
    return(list(
      matrix = mat,
      details = outd,
      outlist = outlist,
      total_reversals = tot_tab
    ))
    
  } else {
    stop("Can only generate matrices of order greater than 2x2!")
  }
}