#' Find the average surface level
#'
#' \code{findAvgSurface} - This function looks at several images which
#' should have the same surface level, for example the root scans (topmost
#' depth window) of the same minirhizotron. Then, the best position of the
#' surface level is determined by finding the best across all images.
#'
#' @param img_paths (Optional, default = NULL) Character vector specifying all
#' of the individual image paths of interest. This is only used if
#' \code{imgs} is set to NULL.
#' @param imgs List of images (e.g., provided by the RootDetector). Each image
#' can be a PNG, i.e., an array with 3 dimensions (3 layers each containing a
#' 2-dim. numeric matrix with values between 0 and 1), or a 2-dim. matrix.
#' @param surface_col Color of the area to be split of the mask (default "red").
#' Can be of any of the three kinds of R color specifications, i.e., either a
#' color name (as listed by colors()), a hexadecimal string (see Details), or a
#' positive integer (indicates using palette()).
#' @param pos_highpoint_px Either one of "center" or "edge" (default),
#' indicating that scanning starts at the bottom or top facing side of the
#' minirhizotron, respectively, or it can also be an integer value (>=1 and
#' <=width of the \code{top_side} of the image) specifying where (left<->right)
#' in the top row of the image the highest point of the image lies, indicating
#' the column where the minimum points of the depth-level lines lie. \cr
#' "edge" is equivalent to using 1 or width of the \code{top_side} and
#' "center" to using half the width of the \code{top_side}.
#' @param radius_highpoint_px The radius specifying how large the are should be
#' to determine the split (default 10).
#' @param angle Numeric value >=0 and <=90 (default 45) specifying the
#' installation angle of the minirhizotron in degrees (angle between the ground
#' and the tube above the soil).
#' @param top_side One of "left","top" (default),"right","bottom". Indicates
#' where the upper side of the image is.
#' @param ppcm Numeric value specifying how many pixels there are per
#' centimeter (resolution of the image). Default NULL.
#' If \code{ppcm} is not NULL or NA, \code{ppi} is ignored.\cr
#' Examples: 300 ppi (printing) is 118 ppcm, 150 ppi is 59 ppcm,
#' 72 ppi (screens) is 28 ppcm.
#' @param ppi Numeric value specifying how many pixels there are per inch
#' (resolution of the image). Default NULL.
#' Leave/set \code{ppcm} to NULL or NA to use \code{ppi}.\cr
#' Examples: 300 ppi (printing) is 118 ppcm, 150 ppi is 59 ppcm,
#' 72 ppi (screens) is 28 ppcm.
#'
#' @return \code{findAvgSurface} A list with the two values
#' 'pos_surf_px' and 'depth_highpoint_cm', and a matrix
#' 'all_split_info' containing the information on all possible splits.
#' @export
#' @rdname findAvgSurface
#'
#' @examples
#' # Example of finding the best overlap of two sets of two matrices.
#' mat_R <- matrix(c(1,1,1,1,1,
#'                   0,1,1,1,1,
#'                   0,0,1,1,1), ncol = 5, nrow = 3, byrow = TRUE)
#' mat_G <- matrix(c(0,0,0,0,1,
#'                   0,1,0,0,1,
#'                   0,0,1,1,1), ncol = 5, nrow = 3, byrow = TRUE)
#' mat_G2 <- matrix(c(0,0,0,1,0,
#'                    0,0,0,0,0,
#'                    0,0,1,1,1), ncol = 5, nrow = 3, byrow = TRUE)
#' mat_B <- matrix(c(0,0,0,0,1,
#'                   0,0,0,0,1,
#'                   0,0,1,1,1), ncol = 5, nrow = 3, byrow = TRUE)
#' mat_B2 <- matrix(c(0,0,0,0,0,
#'                    0,1,0,0,1,
#'                    0,0,1,1,1), ncol = 5, nrow = 3, byrow = TRUE)
#' surface <- findAvgSurface(imgs = list(
#'                   array(c(mat_R,mat_G,mat_B), dim = c(dim(mat_R),3)),
#'                   array(c(mat_R,mat_G2,mat_B), dim = c(dim(mat_R),3)),
#'                   array(c(mat_R,mat_G,mat_B2), dim = c(dim(mat_R),3))),
#'                          radius_highpoint_px = 2, top_side = "top", ppcm = 1)
findAvgSurface <- function(img_paths = NULL, imgs = NULL,
                          surface_col = "red", pos_highpoint_px = "center",
                          radius_highpoint_px = 10,
                          angle = 45,
                          top_side = "left",
                          ppcm = NULL, ppi = NULL) {

  # Load the images. -----------------------------------------------------------
  # If only paths to images are provided, load the images.
  if(is.null(imgs)){
    imgs <- lapply(img_paths, function(path){png::readPNG(path)[,,1:3]})
  } else { # If image is already loaded.
    # If the image is only a matrix, give it three layers.
    if(sum(sapply(imgs,function(img){length(dim(img))})!=3)>0){
      stop("Arrays with 3 color channels needed.")
    }
  }


  # Check input. ---------------------------------------------------------------
  if(nrow(unique(t(sapply(imgs, dim))))>1){
    stop("The dimensions of the images must match.")
  }

  # Compute surface values for all images. -------------------------------------
  list_surfaces <- lapply(imgs, function(curr_img){
    findSurface(img_path = NULL, img = curr_img,
                surface_col = surface_col,
                pos_highpoint_px = pos_highpoint_px,
                radius_highpoint_px = radius_highpoint_px,
                angle = angle, top_side = top_side, ppcm = ppcm, ppi = ppi,
                return_all_results = TRUE)
  })
  all_split_info <- sapply(list_surfaces, function(sub_list){
    return(sub_list$split_info[,"dismatch_difference"])
  })
  all_split_info <- cbind(list_surfaces[[1]]$split_info[,"possible_splits"],
                          all_split_info)
  colnames(all_split_info) <- c("possible_splits",
                                paste("dismatch_difference image",
                                      1:length(imgs)))
  # Find the position of minimal difference over all images. -------------------
  pos_best <- which.min(rowSums(all_split_info[,-1, drop = FALSE]))
  # Find the best values over all sets. ----------------------------------------
  split_px <- all_split_info[pos_best,1]
  depth_px <- sin(angle/180 * pi)*(split_px-1)
  list_to_return <- list("pos_surf_px" = split_px,
                         "depth_highpoint_cm" = px2cm(depth_px, ppi = ppi,
                                                      ppcm = ppcm),
                         "all_split_info" = all_split_info)
  return(list_to_return)
}
