#' Error result handler
#'
#' Creates a structured error result when model fitting fails
#'
#' @param model Model type
#' @param message Error message
#' @return Structured error result
#' @keywords internal
error_result <- function(model, message, verbose = TRUE) {
  if (verbose) {
    warning("Error in model: ", model, "\n",
            "  -> The data may not be suitable. Details:\n",
            "  -> ", message, call. = FALSE)
  }

  # Return a proper structure even when there's an error
  list(
    posterior_means = list(
      alpha = NA,
      p1 = NA,
      p2 = NA,
      p3 = NA,
      theta1 = NA,
      theta2 = NA,
      theta3 = NA
    ),
    credible_intervals = list(
      alpha = c(NA, NA),
      p1 = c(NA, NA),
      p2 = c(NA, NA),
      p3 = c(NA, NA),
      theta1 = c(NA, NA),
      theta2 = c(NA, NA),
      theta3 = c(NA, NA)
    ),
    error = message
  )
}

safe_extract <- function(obj, path, default = NA) {
  tryCatch({
    if (is.null(obj)) return(default)

    # Handle nested access like posterior_means$alpha
    if (grepl("\\$", path)) {
      parts <- strsplit(path, "\\$")[[1]]
      current <- obj
      for (part in parts) {
        if (is.list(current) && part %in% names(current)) {
          current <- current[[part]]
        } else {
          return(default)
        }
      }
      return(current)
    } else {
      # Simple access
      if (is.list(obj) && path %in% names(obj)) {
        return(obj[[path]])
      } else {
        return(default)
      }
    }
  }, error = function(e) {
    return(default)
  })
}
