% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/asymptotic.r, R/simulated.r
\name{distribution}
\alias{distribution}
\alias{asymptotic}
\alias{simulated}
\title{Test statistic distribution under the null}
\usage{
asymptotic()

simulated(method = "approximate", nsims = 1000L, ncores = 1L, ...)
}
\arguments{
\item{method}{(Scalar string: \code{"approximate"})\cr
The method used to derive the distribution of the test statistic
under the null hypothesis. Must be one of \code{"approximate"} (default) or
\code{"exact"}. See 'Details' for additional information.}

\item{nsims}{(Scalar integer: \code{1000L}; \verb{[2, Inf)})\cr
The number of resamples for \code{method = "approximate"}. Not used for
\code{method = "exact"}, except for the case when the number of exact
resamples exceeds approximately \code{1e6} and then \code{method = "approximate"}
will be used as a fallback. In the \code{\link[=power]{power()}} context, \code{nsims}
defines the number of simulated datasets under the null hypothesis.
For this case you would typically set \code{nsims} as greater than or equal
to the number of simulated datasets in the design row of the power analysis.
See 'Details' for additional information.}

\item{ncores}{(Scalar integer: \code{1L}; \verb{[1, Inf)})\cr
The number of cores (number of worker processes) to use. Do not set
greater than the value returned by \code{\link[parallel:detectCores]{parallel::detectCores()}}.}

\item{...}{Optional arguments for internal use.}
}
\value{
list
}
\description{
Constructs a list which defines the test statistic reference distribution
under the null hypothesis.
}
\details{
The default asymptotic test is performed for \code{distribution = asymptotic()}.

When setting argument \code{distribution = simulated(method = "exact")}, the
exact randomization test is defined by:
\itemize{
\item Independent two-sample tests
\enumerate{
\item Calculate the observed test statistic.
\item Check if \code{length(combn(x=n1+n2, m=n1))<1e6}
\enumerate{
\item If \code{TRUE} continue with the exact randomization test.
\item If \code{FALSE} revert to the approximate randomization test.
}
\item For all \code{combn(x=n1+n2, m=n1)} permutations:
\enumerate{
\item Assign corresponding group labels.
\item Calculate the test statistic.
}
\item Calculate the exact randomization test p-value as the mean of the logical
vector \code{resampled_test_stats >= observed_test_stat}.
}
\item Dependent two-sample tests
\enumerate{
\item Calculate the observed test statistic.
\item Check if \code{npairs < 21} (maximum 2^20 resamples)
\enumerate{
\item If \code{TRUE} continue with the exact randomization test.
\item If \code{FALSE} revert to the approximate randomization test.
}
\item For all \code{2^npairs} permutations:
\enumerate{
\item Assign corresponding pair labels.
\item Calculate the test statistic.
}
\item Calculate the exact randomization test p-value as the mean of the logical
vector \code{resampled_test_stats >= observed_test_stat}.
}
}

For argument \code{distribution = simulated(method = "approximate")}, the
approximate randomization test is defined by:
\itemize{
\item Independent two-sample tests
\enumerate{
\item Calculate the observed test statistic.
\item For \code{nsims} iterations:
\enumerate{
\item Randomly assign group labels.
\item Calculate the test statistic.
}
\item Insert the observed test statistic to the vector of resampled test
statistics.
\item Calculate the approximate randomization test p-value as the mean of
the logical vector \code{resampled_test_stats >= observed_test_stat}.
}
\item Dependent two-sample tests
\enumerate{
\item Calculate the observed test statistic.
\item For \code{nsims} iterations:
\enumerate{
\item Randomly assign pair labels.
\item Calculate the test statistic.
}
\item Insert the observed test statistic to the vector of resampled test
statistics.
\item Calculate the approximate randomization test p-value as the mean of
the logical vector \code{resampled_test_stats >= observed_test_stat}.
}
}

In the power analysis setting, \code{\link[=power]{power()}}, we can simulate data for
groups 1 and 2 using their known distributions under the assumptions of the
null hypothesis. Unlike above where nonparametric randomization tests
are performed, in this setting approximate parametric tests are performed.

For example, \code{power(wald_test_nb(distribution = simulated()))} would result
in an approximate parametric Wald test defined by:
\enumerate{
\item For each relevant design row in \code{data}:
\enumerate{
\item For \code{simulated(nsims=integer())} iterations:
\enumerate{
\item Simulate new data for group 1 and group 2 under the null hypothesis.
\item Calculate the Wald test statistic, \eqn{\chi^2_{null}}.
}
\item Collect all \eqn{\chi^2_{null}} into a vector.
\item For each of the \code{sim_nb(nsims=integer())} simulated datasets:
\enumerate{
\item Calculate the Wald test statistic, \eqn{\chi^2_{obs}}.
\item Calculate the p-value based on the empirical null distribution of test statistics, \eqn{\chi^2_{null}}.
(the mean of the logical vector \code{null_test_stats >= observed_test_stat})
}
\item Collect all p-values into a vector.
\item Calculate power as \code{sum(p <= alpha) / nsims}.
}
\item Return all results from \code{power()}.
}

Randomization tests use the positive-biased p-value estimate in the style of
\insertCite{davison_1997;textual}{depower}
(see also \insertCite{phipson_2010;textual}{depower}):

\deqn{
\hat{p} = \frac{1 + \sum_{i=1}^B \mathbb{I} \{\chi^2_i \geq \chi^2_{obs}\}}{B + 1}.
}

The number of resamples defines the minimum observable p-value
(e.g. \code{nsims=1000L} results in min(p-value)=1/1001).
It's recommended to set \eqn{\text{nsims} \gg \frac{1}{\alpha}}.
}
\examples{
#----------------------------------------------------------------------------
# asymptotic() examples
#----------------------------------------------------------------------------
library(depower)

set.seed(1234)
data <- sim_nb(
  n1 = 60,
  n2 = 40,
  mean1 = 10,
  ratio = 1.5,
  dispersion1 = 2,
  dispersion2 = 8
)

data |>
  wald_test_nb(distribution = asymptotic())

#----------------------------------------------------------------------------
# simulated() examples
#----------------------------------------------------------------------------
data |>
  wald_test_nb(distribution = simulated(nsims = 200L))

}
\references{
\insertRef{davison_1997}{depower}

\insertRef{phipson_2010}{depower}
}
