#' A function that returns the coefficient from regressing an outcome
#' vector on a dose vector.
#'
#' @param z a vector of doses
#' @param r a vector of outcomes
#'
#' @return the OLS regression coefficient from regressing r on z.
#' @export
#'
#' @examples
#' # dose vector
#' dose <- c(0, 0.1, 0.4)
#' # outcome vector
#' outcome <- c(1, 1.1, 1.5)
#' beta <- extract_OLS(z = dose, r = outcome)
extract_OLS <- function(z, r) {
  # Run OLS
  model <- stats::lm(r ~ z)

  # Extract coefficient
  slope_coefficient <- stats::coef(model)["z"]

  return(as.numeric(slope_coefficient))
}

#' A function for variance estimation
#'
#' @param y a vector of length I containing the value of test statistics from
#' each matched set
#' @param W weight vector of length I.
#' @param H_Q hat matrix corresponding to a matrix Q with dimension I by L.
#'
#' @return a conservative estimate of the standard deviation of the test statistic.
#' @export
#'
#' @examples
#' test_stat <- c(1, 2, 1.5)
#' weight <- rep(1, 3)
#' Q <- matrix(1:9, nrow = 3, ncol = 2)
#' hat <- Q %*% solve(t(Q) %*% Q) %*% t(Q)
var_est <- function(y, W, H_Q) {
  # number of matched sets
  I <- length(y)

  # conservative variance estimator based on Q from Fogarty 2018
  var <- 1/I^2 * t(y) %*% W %*% (diag(1, I) - H_Q) %*% W %*% y

  return(sqrt(var))
}

#' A function that returns a permuted vector according to a matrix and permutation
#' vector.
#'
#' @param M a square matrix of dimension n
#' @param p a permuted version of the vector from 1:n
#'
#' @return a length n vector with ith entry corresponding to the i, p_i entry in M
#' @export
#'
#' @examples
#' mat <- matrix(1:9, nrow = 3, ncol = 3)
#' perm <- c(2, 1, 3)
#' permuted <- apply_permutation_to_matrix(M = mat, p = perm)
apply_permutation_to_matrix <- function(M, p) {
  n <- length(p)
  result_vector <- numeric(n)

  for (i in 1:n) {
    result_vector[i] <- M[i, p[i]]
  }

  return(result_vector)
}

#' Compute difference in average outcomes above and below a dose threshold.
#'
#' @param z a vector of doses
#' @param r a vector of outcomes
#' @param threshold a dose threshold
#'
#' @return the average of the outcomes with dose z above threshold c minus the
#' average of the outcomes with dose z below the threshold c.
#' @export
#'
#' @examples
#' # dose vector
#' dose <- c(0, 0.1, 0.4)
#' # outcome vector
#' outcome <- c(1, 1.1, 1.5)
#' theta <- extract_threshold_effect(z = dose, r = outcome, threshold = 0.3)
extract_threshold_effect <- function(z, r, threshold) {
  # threshold sate
  n <- length(z)
  m <- sum(z > threshold)
  above_threshold <- r[which(z > threshold)]
  below_threshold <- r[which(z <= threshold)]

  return(sum(above_threshold) / m - sum(below_threshold) / (n - m))
}

#' Function factory for extract_threshold_effect.
#'
#' @param threshold a dose threshold
#'
#' @return A function that corresponds to extract_threshold_effect with the given
#' threshold
#' @export
#'
#' @examples
#' threshold_function <- extract_threshold_effect_function(threshold = 0.3)
extract_threshold_effect_function <- function(threshold = 0) {
  func <- function(z, r) {
    return(extract_threshold_effect(z = z, r = r, threshold = threshold))
  }
  return(func)
}

#' Compute weighted sum of outcomes.
#'
#' @param z a vector of doses
#' @param r a vector of outcomes
#' @param s a set of weights, summing to 1
#'
#' @return the inner product of s and r
#' @export
#'
#' @examples
#' # dose vector
#' dose <- c(0, 0.1, 0.4)
#' # outcome vector
#' outcome <- c(1, 1.1, 1.5)
#' # weight vector
#' weight = c(0.3, 0.4, 0.3)
#' theta <- extract_stochastic_intervention(z = dose, r = outcome, s = weight)
extract_stochastic_intervention <- function(z, r, s) {
  # make weights sum to 1
  s <- s / sum(s)
  # threshold sate
  n <- length(z)

  return(sum(s * r))
}

#' Compute average of outcomes below dose threshold minus average of outcomes.
#'
#' @param z a vector of doses
#' @param r a vector of outcomes
#' @param threshold a dose threshold
#'
#' @return the average of the outcomes with dose z below threshold c minus the
#' average of the outcomes r.
#' @export
#'
#' @examples
#' # dose vector
#' dose <- c(0, 0.1, 0.4)
#' # outcome vector
#' outcome <- c(1, 1.1, 1.5)
#' theta <- extract_below_threshold_vs_baseline(z = dose, r = outcome, threshold = 0.3)
extract_below_threshold_vs_baseline <- function(z, r, threshold) {
  # threshold sate
  n <- length(z)
  m <- sum(z > threshold)
  below_threshold <- r[which(z <= threshold)]

  return(sum(below_threshold) / (n - m) - sum(rep(1 / n, n) * r))
}

#' Compute average of outcomes above dose threshold minus average of outcomes.
#'
#' @param z a vector of doses
#' @param r a vector of outcomes
#' @param threshold a dose threshold
#'
#' @return the average of the outcomes with dose z above threshold c minus the
#' average of the outcomes r.
#' @export
#'
#' @examples
#' # dose vector
#' dose <- c(0, 0.1, 0.4)
#' # outcome vector
#' outcome <- c(1, 1.1, 1.5)
#' theta <- extract_above_threshold_vs_baseline(z = dose, r = outcome, threshold = 0.3)
extract_above_threshold_vs_baseline <- function(z, r, threshold) {
  # threshold sate
  n <- length(z)
  m <- sum(z > threshold)
  above_threshold <- r[which(z > threshold)]

  return(sum(above_threshold) / m - sum(rep(1 / n, n) * r))
}

#' Compute smallest dose outcome minus average of other outcomes.
#'
#' @param z a vector of doses
#' @param r a vector of outcomes
#'
#' @return the outcome r corresponding to the smallest dose z minus the average
#' of the outcomes r.
#' @export
#'
#' @examples
#' # dose vector
#' dose <- c(0, 0.1, 0.4)
#' # outcome vector
#' outcome <- c(1, 1.1, 1.5)
#' theta <- extract_min_vs_baseline(z = dose, r = outcome)
extract_min_vs_baseline <- function(z, r) {
  n <- length(z)
  min_dose <- min(z)
  min_outcome <- r[which(z == min_dose)]

  return(min_outcome - sum(rep(1 / n, n) * r))
}

#' Compute largest dose outcome minus average of other outcomes.
#'
#' @param z a vector of doses
#' @param r a vector of outcomes
#'
#' @return the outcome r corresponding to the largest dose z minus the average
#' of the outcomes r.
#' @export
#'
#' @examples
#' # dose vector
#' dose <- c(0, 0.1, 0.4)
#' # outcome vector
#' outcome <- c(1, 1.1, 1.5)
#' theta <- extract_max_vs_baseline(z = dose, r = outcome)
extract_max_vs_baseline <- function(z, r) {
  n <- length(z)
  max_dose <- max(z)
  max_outcome <- r[which(z == max_dose)]

  return(max_outcome - sum(rep(1 / n, n) * r))
}


#' Statistic based on inner product between transformations of dose and outcome.
#'
#' @param z a vector of doses
#' @param r a vector of outcomes
#' @param q1 a function that transforms the doses z
#' @param q2 a function that transforms the outcomes r
#'
#' @return a vector with values corresponding to the inner product of transformed
#' by q1 permutations of z with transformed by q2 versions of r.
#'
#' @export
#'
#' @examples
#' # dose vector
#' dose <- c(0, 0.1, 0.4)
#' # outcome vector
#' outcome <- c(1, 1.1, 1.5)
#' # transforms
#' transform1 <- function(x) x
#' transform2 <- function (x) x
#' theta <- sharp_double_statistic(z = dose, r = outcome, q1 = transform1,
#' q2 = transform2)
sharp_double_statistic <- function(z, r, q1, q2) {
  # number of subjects.
  n <- length(z)
  indices <- order(z)

  z_sorted <- z[indices]

  # The matrix of permutations of the doses.
  permZ <- unique(gtools::permutations(n,n,z_sorted,set = FALSE))
  n_perms <- nrow(permZ)

  q_pi <- rep(NA, n_perms)
  # inner product for each permutation
  for (i in 1:n_perms) {
    q_pi[i] <- sum(vapply(permZ[i,], q1, FUN.VALUE = numeric(1)) * vapply(r, q2, FUN.VALUE = numeric(1)))
  }

  return(q_pi)
}



