% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/evaluate.R
\name{normtable_create}
\alias{normtable_create}
\title{Create a norm table based on a GAMLSS fitted model}
\usage{
normtable_create(
  model,
  data,
  age_name,
  score_name,
  datarel = NULL,
  normtype = "Z",
  min_age = NULL,
  max_age = NULL,
  min_score = NULL,
  max_score = NULL,
  step_size_score = 1,
  step_size_age = NULL,
  cont_cor = FALSE,
  ci_level = 0.95,
  trim = 3,
  excel = FALSE,
  excel_name = tempfile("norms", fileext = ".xlsx"),
  new_data = FALSE
)
}
\arguments{
\item{model}{a GAMLSS fitted model, for example the result of \code{\link[=fb_select]{fb_select()}}.}

\item{data}{data.frame. The sample on which the model has been fitted, or new data;
must contain the score variable (with name given in \code{score_name}) and age variable
(with name given in \code{age_name}).}

\item{age_name}{string. Name of the age variable.}

\item{score_name}{string. Name of the score variable.}

\item{datarel}{data.frame or numeric. If a data.frame, must contain columns \code{age} and \code{rel},
with estimated test reliability per age. If numeric, a constant reliability is assumed
for all ages (optional, only needed for confidence intervals).}

\item{normtype}{string. Norm score type: \code{"Z"} (N(0,1); default), \code{"T"} (N(50,10)), or \code{"IQ"} (N(100,15)).}

\item{min_age}{numeric. Lowest age value in the norm table; default is the first integer
below the minimum observed age.}

\item{max_age}{numeric. Highest age value in the norm table; default is the first integer
above the maximum observed age.}

\item{min_score}{numeric. Lowest score value in the norm table; default is the minimum
observed score.}

\item{max_score}{numeric. Highest score value in the norm table; default is the maximum
observed score.}

\item{step_size_score}{numeric. Increment of the scores in the norm table; default is 1.}

\item{step_size_age}{numeric. Increment of the ages in the norm table; defaults to
approximately 100 ages in total.}

\item{cont_cor}{logical. If \code{TRUE}, apply continuity correction for discrete test scores.
Default is \code{FALSE}.}

\item{ci_level}{numeric. Confidence interval level (if \code{datarel} is provided).
Default is 0.95.}

\item{trim}{numeric. Trim norm scores at ± \code{trim} standard deviations.
Default is 3.}

\item{excel}{logical. If \code{TRUE}, attempt to write results to an Excel file. Default is \code{FALSE}.}

\item{excel_name}{character. Path to the Excel file. Defaults to a temporary file.
Ignored if \code{excel = FALSE}.}

\item{new_data}{logical. If \code{FALSE} (default), create a full norm table and norm scores.
If \code{TRUE}, only return norm scores for the given data.}
}
\value{
A list of class \code{NormTable} containing:
\itemize{
\item \code{norm_sample}: Estimated norm scores (\code{normtype}) in the sample, trimmed at \code{trim}.
\item \code{norm_sample_lower}, \code{norm_sample_upper}: Lower and upper \code{ci_level} confidence bounds of \code{norm_sample}.
\item \code{norm_matrix}: Norm scores (\code{normtype}) by age (only if \code{new_data = FALSE}).
\item \code{norm_matrix_lower}, \code{norm_matrix_upper}: Lower and upper \code{ci_level} bounds of \code{norm_matrix}.
\item \code{znorm_sample}: Estimated Z scores in the sample.
\item \code{cdf_sample}: Estimated percentiles in the sample.
\item \code{cdf_matrix}: Percentile table by age (only if \code{new_data = FALSE}).
\item \code{data}, \code{age_name}, \code{score_name}: Copies of respective function arguments.
\item \code{pop_age}: Evaluated ages in the norm table (only if \code{new_data = FALSE}).
}
}
\description{
\code{normtable_create()} creates a norm table based on a fitted \code{GAMLSS} model.
}
\details{
If \code{excel = TRUE}, results are written to an Excel file via the \pkg{openxlsx2} package.
If the package is not installed, a message is printed and the function continues
without writing an Excel file. By default, the file is written to a temporary path
(see \code{\link[=tempfile]{tempfile()}}); if you want to keep the file permanently, provide your own file
name via the \code{excel_name} argument (e.g., \code{"norms.xlsx"}).
}
\examples{
\donttest{
# Load example data
invisible(data("ids_data"))

# Prepare data for modeling
mydata_BB_y14 <- shape_data(
  data = ids_data,
  age_name = "age",
  score_name = "y14",
  family = "BB"
)

# Fit model using BIC as selection criterion
mod_BB_y14 <- fb_select(
  data = mydata_BB_y14,
  age_name = "age",
  score_name = "shaped_score",
  family = "BB",
  selcrit = "BIC"
)

# Create norm table from fitted model
norm_mod_BB_y14 <- normtable_create(
  model = mod_BB_y14,
  data = mydata_BB_y14,
  age_name = "age",
  score_name = "shaped_score"
)

# Calculate norms for a new sample using reliability data
invisible(data("ids_rel_data"))
newdata <- ids_data[1:5, c("age", "y14")]

norm_mod_BB_newdata <- normtable_create(
  model = mod_BB_y14,
  data = newdata,
  age_name = "age",
  score_name = "y14",
  new_data = TRUE,
  datarel = ids_rel_data
)
}
}
\references{
\insertRef{timmerman2021tutorial}{normref}
}
\seealso{
\code{\link[=fb_select]{fb_select()}}, \code{\link[=plot_normtable]{plot_normtable()}}
}
